# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['annofabcli',
 'annofabcli.annotation',
 'annofabcli.annotation_specs',
 'annofabcli.common',
 'annofabcli.experimental',
 'annofabcli.filesystem',
 'annofabcli.input_data',
 'annofabcli.inspection_comment',
 'annofabcli.instruction',
 'annofabcli.job',
 'annofabcli.labor',
 'annofabcli.organization_member',
 'annofabcli.project',
 'annofabcli.project_member',
 'annofabcli.statistics',
 'annofabcli.supplementary',
 'annofabcli.task']

package_data = \
{'': ['*'], 'annofabcli': ['data/*']}

install_requires = \
['annofabapi>=0.38.3',
 'bokeh>=2.1.1,<3.0.0',
 'dictdiffer',
 'holoviews>=1.13.3,<2.0.0',
 'isodate',
 'jmespath',
 'more-itertools',
 'pandas>=1.1,<2.0',
 'pillow',
 'pyquery',
 'pyyaml',
 'requests']

entry_points = \
{'console_scripts': ['annofabcli = annofabcli.__main__:main']}

setup_kwargs = {
    'name': 'annofabcli',
    'version': '1.33.20',
    'description': 'Utility Command Line Interface for AnnoFab',
    'long_description': '[![Build Status](https://travis-ci.com/kurusugawa-computer/annofab-cli.svg?branch=master)](https://travis-ci.com/kurusugawa-computer/annofab-api-python-client)\n[![PyPI version](https://badge.fury.io/py/annofabcli.svg)](https://badge.fury.io/py/annofabcli)\n[![Python Versions](https://img.shields.io/pypi/pyversions/annofabcli.svg)](https://pypi.org/project/annofabcli/)\n\n\n\n# 概要\nAnnoFabのCLI(Command Line Interface)ツールです。\n「タスクの一括差し戻し」や、「プロジェクト間の差分表示」など、AnnoFabの画面で実施するには時間がかかる操作を、コマンドとして提供しています。\n\n# 注意\n* 作者または著作権者は、ソフトウェアに関してなんら責任を負いません。\n* 予告なく互換性のない変更がある可能性をご了承ください。\n* AnnoFabプロジェクトに大きな変更を及ぼすコマンドも存在します。間違えて実行してしまわないよう、注意してご利用ください。\n\n\n## 廃止予定\n* 2020-10-01以降：Pythonのサポートバージョンを3.6以上から、3.7以上に変更します。\n* 2020-09-01以降：`input_data list`コマンドの`--batch`オプションを削除します。入力データを１万件以上取得したい場合は、`project download input_data`コマンドを利用してください。\n\n# Requirements\n* Python 3.6+\n\n# Install\n\n```\n$ pip install annofabcli\n```\n\nhttps://pypi.org/project/annofabcli/\n\n\n## AnnoFabの認証情報の設定\nAnnoFabの認証情報を設定する方法は2つあります。\n* `.netrc`ファイル\n* 環境変数`ANNOFAB_USER_ID` , `ANNOFAB_PASSWORD`\n\n`.netrc`ファイルへの記載方法は、[annofab-api-python-client/README.md](https://github.com/kurusugawa-computer/annofab-api-python-client#netrc%E3%81%AB%E8%A8%98%E8%BC%89%E3%81%95%E3%82%8C%E3%81%9Fuser_id-password%E3%81%8B%E3%82%89%E7%94%9F%E6%88%90)を参照してください。\n\nAnnoFabの認証情報が設定されていない状態で`annofabcli`コマンドを実行すると、標準入力からAnnoFabの認証情報を入力できるようになります。\n\n```\n$ annofabcli project diff aaa bbb\nEnter AnnoFab User ID: XXXXXX\nEnter AnnoFab Password: \n```\n\nAnnoFabの認証情報の優先順位は以下の通りです。\n1. `.netrc`ファイル\n2. 環境変数\n\n## AnnoFab WebAPIのエンドポイントの設定（開発者用）\nAnnoFab WebAPIのエンドポイントを指定できます。デフォルトは https://annofab.com です。\n* コマンドライン引数`--endpoint_url`\n* 環境変数 `ANNOFAB_ENDPOINT_URL`\n\n設定したエンドポイントURLの優先順位は以下の通りです。\n1. コマンドライン引数\n2. 環境変数\n\n\n## Dockerを利用する場合\n\n```\n$ git clone https://github.com/kurusugawa-computer/annofab-cli.git\n$ cd annofab-cli\n$ chmod u+x docker-build.sh\n$ ./docker-build.sh\n\n$ docker run -it annofab-cli annofabcli --help\n\n# AnnoFabの認証情報を標準入力から指定する\n$ docker run -it annofab-cli annofabcli project diff prj1 prj2\nEnter AnnoFab User ID: XXXXXX\nEnter AnnoFab Password: \n\n# AnnoFabの認証情報を環境変数で指定する\n$ docker run -it -e ANNOFAB_USER_ID=XXXX -e ANNOFAB_PASSWORD=YYYYY annofab-cli annofabcli project diff prj1 prj2\n```\n\n# 機能一覧\n\n|コマンド| サブコマンド                  | 内容                                                                                                     |必要なロール|\n|----|-------------------------------|----------------------------------------------------------------------------------------------------------|------------|\n|annotation| change_attributes |アノテーションの属性を変更します。                              |オーナ|\n|annotation| delete | アノテーションを削除します。                              |オーナ|\n|annotation| dump | アノテーション情報をファイルに保存します。                             |-|\n|annotation| list_count | task_idまたはinput_data_idで集約したアノテーションの個数を出力します                              |-|\n|annotation| import | アノテーションをインポートします。                             |オーナ|\n|annotation| restore |\'annotation dump\'コマンドで保存したファイルから、アノテーション情報をリストアします。                            |オーナ|\n|annotation_specs| history | アノテーション仕様の履歴一覧を出力します。                              |-|\n|annotation_specs| list_label | アノテーション仕様のラベル情報を出力します。                              |-|\n|annotation_specs| list_label_color             | アノテーション仕様から、label_nameとRGBを対応付けたJSONを出力します。                                      |-|\n|filesystem| write_annotation_image        | アノテーションzip、またはそれを展開したディレクトリから、アノテーションの画像（Semantic Segmentation用）を生成します。 |-|\n|input_data|delete             | 入力データを削除します。                                                            |オーナ|\n|input_data|list             | 入力データ一覧を出力します。                                                            |-|\n|input_data| list_merged_task | タスク一覧と結合した入力データ一覧のCSVを出力します。                                                            |オーナ/アノテーションユーザ|\n|input_data|put             | 入力データを登録します。                                                            |オーナ|\n|inspection_comment| list | 検査コメントを出力します。                               |-|\n|inspection_comment| list_unprocessed | 未処置の検査コメントを出力します。                               |-|\n|instruction| copy             | 作業ガイドをコピーします。                                                         |チェッカー/オーナ|\n|instruction| upload             | HTMLファイルを作業ガイドとして登録します。                                                           |チェッカー/オーナ|\n|job|delete             | ジョブを削除します。                                                            |オーナ|\n|job|list             | ジョブ一覧を出力します。                                                            |-|\n|job|list_last             | 複数のプロジェクトに対して、最新のジョブを出力します。                                                            |-|\n|labor|list_worktime_by_user | ユーザごとに作業予定時間、作業実績時間を出力します。                                                          ||\n|organization_member|list             | 組織メンバ一覧を出力します。                                                            |-|\n|project| copy                 | プロジェクトをコピーします。                                                                           |オーナ and 組織管理者/組織オーナ|\n|project| diff                 | プロジェクト間の差分を表示します。                                                                           |チェッカー/オーナ|\n|project| download                 | タスクや検査コメント、アノテーションなどをダウンロードします。                                                                           |オーナ|\n|project| list                 | プロジェクト一覧を出力します。                                                                          |-|\n|project| update_annotation_zip                 | アノテーションzipを更新します。                                                                         |オーナ/アノテーションユーザ|\n|project_member| change                  | プロジェクトメンバを変更します。|オーナ|\n|project_member| copy                  | プロジェクトメンバをコピーします。|オーナ(コピー先プロジェクトに対して)|\n|project_member| delete                  | 複数のプロジェクトからユーザを削除します。                                                                 |オーナ|\n|project_member| invite                  | 複数のプロジェクトに、ユーザを招待します。                                                                 |オーナ|\n|project_member| list                  | プロジェクトメンバ一覧を出力します。                                                                |-|\n|project_member| put                  | CSVに記載されたユーザを、プロジェクトメンバとして登録します。|オーナ|\n|statistics| list_annotation_count             | 各ラベル、各属性値のアノテーション数を、タスクごと/入力データごとに出力します。                                                   |-|\n|statistics| list_by_date_user             | タスク数や作業時間などの情報を、日ごとユーザごとに出力します。                                                   |オーナ/アノテーションユーザ|\n|statistics| list_cumulative_labor_time             |       タスク進捗状況を出力します。                                                    |-|\n|statistics| list_task_progress             | タスクフェーズ別の累積作業時間を出力します。                                                            |-|\n|statistics|summarize_task_count|タスクのフェーズ、ステータス、ステップごとにタスク数を出力します。|オーナ/アノテーションユーザ|\n|statistics|summarize_task_count_by_task_id|task_idのプレフィックスごとに、タスク数を出力します。|オーナ/アノテーションユーザ|\n|statistics|summarize_task_count_by_user|ユーザごとに担当しているタスク数を出力します。|オーナ/アノテーションユーザ|\n|statistics| visualize             | 統計情報を可視化します。                                                            |オーナ/アノテーションユーザ|\n|supplementary| list             | 補助情報を出力します。                                                           |オーナ/アノテーションユーザ|\n|supplementary| put              | 補助情報を登録します。                                                           |オーナ|\n|task| cancel_acceptance             | 受け入れ完了タスクを、受け入れ取り消し状態にします。                                                         |オーナ|\n|task| change_operator             | タスクの担当者を変更します。                                                             |チェッカー/オーナ|\n|task| complete                | タスクを完了状態にして次のフェーズに進めます（教師付の提出、検査/受入の合格）。                                  |チェッカー/オーナ|\n|task| delete                | タスクを削除します。                                 |オーナ|\n|task|list             | タスク一覧を出力します。                                                            |-|\n|task|list_added_task_history             | タスク履歴情報を加えたタスク一覧を出力します。|オーナ/アノテーションユーザ|\n|task|list_task_history             | タスク履歴の一覧を出力します。|-|\n|task| put                | タスクを作成します。                                 |オーナ|\n|task| reject                  | タスクを強制的に差し戻します。                                                                 |オーナ|\n\n\n# Usage\n\n\n## すべてのコマンドで共通のオプション引数\n\n### `--disable_log`\nログを無効化にします。\n\n\n### `--endpoint_url\nAnnoFab WebAPIのエンドポイントURLを指定します。デフォルトは https://annofab.com です。\n\n\n### `-h` / `--help`\nコマンドのヘルプを出力します。\n\n```\n# annofabcli全体のヘルプ\n$ annofabcli -h\n\n# project diff コマンドのヘルプ\n$ annofabcli project diff -h\n```\n\n\n### `--logdir`\nログファイルを保存するディレクトリを指定します。指定しない場合、`.log`ディレクトリにログファイルを出力します。\n\n\n### `--logging_yaml`\n以下のような、ロギグングの設定ファイル(YAML)を指定します。指定した場合、`--logdir`オプションは無視されます。指定しない場合、デフォルトのロギング設定ファイルが読み込まれます。\n設定ファイルの書き方は https://docs.python.org/ja/3/howto/logging.html を参照してください。\n\n```yaml:logging-sample.yaml\n# WARNINGレベル以上のログをコンソールに出力する\n\nversion: 1\nhandlers:\n  consoleHandler:\n    class: logging.StreamHandler\nroot:\n  level: WARNING\n  handlers: [consoleHandler]\n\n# デフォルトのロガーを無効化しないようにする https://docs.djangoproject.com/ja/2.1/topics/logging/#configuring-logging\ndisable_existing_loggers: False\n```\n\n### `--yes`\n処理中に現れる問い合わせに対して、常に\'yes\'と回答します。\n\n\n## ほとんどのコマンドで共通のオプション引数\n\n\n### `--csv_format`\nCSVのフォーマットをJSON形式で指定します。`--format`が`csv`でないときは、このオプションは無視されます。\n先頭に`file://`を付けると、JSON形式のファイルを指定できます。\n指定した値は、[pandas.DataFrame.to_csv](https://pandas.pydata.org/pandas-docs/stable/reference/api/pandas.DataFrame.to_csv.html) の引数として渡されます。\nデフォルトはカンマ区切り、BOM付きUTF-8エンコーディングです。\n\n```\n--csv_format \'{"sep": "\\t"}\'\n```\n\n\n### `-f` / `--format`\nlist系のコマンドで、出力フォーマットを指定します。多くのコマンドでは、以下のフォーマットが指定できます。\n* `csv` : CSV(デフォルとはカンマ区切り)\n* `json` : インデントや空白がないJSON\n* `pretty_json` : インデントされたJSON\n\n\n\n\n\n### `-o` / `--output`\n出力先のファイルパスを指定します。指定しない場合は、標準出力に出力されます。\n主にlist系のコマンドで利用できます。\n\n\n### `-p` / `--project_id`\n対象のプロジェクトのproject_idを指定します。\n\n### `-q` / `--query`\n[JMESPath](http://jmespath.org/) を指定します。出力結果の抽出や、出力内容の変更に利用できます。\n\n\n### `-t` / `--task_id`\n対象のタスクのtask_idを指定します。`file://`を先頭に付けると、task_idの一覧が記載されたファイルを指定できます。\n\n* 相対パスで指定： `--task_id file://task.txt`\n* 絶対パスで指定： `--task_id file:///tmp/task.txt`\n\n\n## デフォルトのログ設定\n* ログは、標準エラー出力とログファイルに出力されます。\n* カレントディレクトリの`.log`ディレクトリに、`annofabcli.log`というログファイルが生成されます。\n* `annofabcli.log`ファイルは、1日ごとにログロテート（新しいログファイルが生成）されます\n\nデフォルトログは https://github.com/kurusugawa-computer/annofab-cli/blob/master/annofabcli/data/logging.yaml で定義されています。\n\n\n## よくある使い方\n\n### 受入完了状態のタスクを差し戻す\n"car"ラベルの"occluded"属性のアノテーションルールに間違いがあったため、以下の条件を満たすタスクを一括で差し戻します。\n* "car"ラベルの"occluded"チェックボックスがONのアノテーションが、タスクに1つ以上存在する。\n\n前提条件\n* プロジェクトのオーナが、annofabcliコマンドを実行する\n\n\n```\n# 受入完了のタスクのtask_id一覧を、acceptance_complete_task_id.txtに出力する。\n$ annofabcli task list --project_id prj1  --task_query \'{"status": "complete","phase":"acceptance"}\' \\\n --format task_id_list --output acceptance_complete_task_id.txt\n\n# 受入完了タスクの中で、 "car"ラベルの"occluded"チェックボックスがONのアノテーションの個数を出力する。\n$ annofabcli annotation list_count --project_id prj1 --task_id file://task.txt --output annotation_count.csv \\\n --annotation_query \'{"label_name_en": "car", "attributes":[{"additional_data_definition_name_en": "occluded", "flag": true}]}\'\n\n# annotation_count.csvを表計算ソフトで開き、アノテーションの個数が1個以上のタスクのtask_id一覧を、task_id.txtに保存する。\n\n# task_id.txtに記載されたタスクを差し戻す。検査コメントは「carラベルのoccluded属性を見直してください」。\n# 差し戻したタスクには、最後のannotation phaseを担当したユーザを割り当てる（画面と同じ動き）。\n$ annofabcli task reject --project_id prj1 --task_id file://tasks.txt --cancel_acceptance \\\n  --comment "carラベルのoccluded属性を見直してください"\n\n```\n\n### プロジェクトメンバをCSVで管理する\n\n```\n# prj1のプロジェクトメンバをCSVで出力する\n$ annofabcli project_member list --project_id prj1 --format csv --output members.csv \\\n --csv_format \'{"columns": ["user_id","member_role","sampling_inspection_rate","sampling_acceptance_rate"],"header":false}\' \n\n\n# members.csvの中身を確認\n$ head members.csv\nuser1,worker\nuser2,accepter\n...\n\n\n# members.csvに記載れたメンバを prj2に登録する\n$ annofabcli project_member put --project_id prj2 --csv members.csv\n\n```\n\n\n## コマンド一覧\n\n### annotation change_attributes\nアノテーションの属性を一括で変更します。ただし、作業中状態のタスクのアノテーションの属性は変更できません。間違えてアノテーション属性を変更したときに復元できるようにするため、`--backup`でバックアップ用のディレクトリを指定することを推奨します。\n\n```\n# task.txtに記載されたタスクのアノテーションの属性をを変更する\n# carラベルのoccludedチェックボックスがTRUEのアノテーションに対して、occludedチェックボックスをOFFにする\n$ annofabcli annotation change_attributes --project_id prj1 --task_id file://task.txt \\ \n--annotation_query \'{"label_name_en": "car", "attributes":[{"additional_data_definition_name_en": "occluded", "flag": true}]}\' \\\n--attributes \'[{"additional_data_definition_name_en": "occluded", "flag": false}]\' \\\n--backup backup_dir\n\n\n```\n\n\n\n\n### annotation delete\nタスク配下のアノテーションを削除します。ただし、作業中/完了状態のタスク、または「過去に割り当てられていて現在の担当者が自分自身でない」タスクのアノテーションは削除できません。\n\n間違えてアノテーションを削除してしまっときに復元できるようにするため、`--backup`でバックアップ用のディレクトリを指定することを推奨します。バックアップ用ディレクトリには、 [annotation dump](#annotation-dump) コマンドの出力結果と同等のアノテーション情報が保存されます。\nアノテーションの復元には [annotation restore](#annotation-restore) コマンドを使用してください。\n\n\n```\n# task.txtに記載されたタスクのアノテーションを削除します。削除する前のアノテーション情報は、`backup`ディレクトリに保存します。\n$ annofabcli annotation delete --project_id prj1 --task_id file://task.txt --backup backup\n\n# carラベルのoccludedチェックボックスがTRUEのアノテーションを削除します\n$ annofabcli annotation delete --project_id prj1 --task_id file://task.txt \\ \n--annotation_query \'{"label_name_en": "car", "attributes":[{"additional_data_definition_name_en": "occluded", "flag": true}]}\' \\\n--backup backup_dir\n```\n\n### annotation dump\n指定したタスク配下のアノテーション情報をディレクトリに保存します。アノテーションをバックアップしたいときなどに利用できます。\nアノテーションの復元には [annotation restore](#annotation-restore) コマンドを使用してください。\n\n\n```\n# task.txtに記載されたタスクのアノテーションを、`output`ディレクトリに保存します。\n$ annofabcli annotation dump --project_id prj1 --task_id file://task.txt --output backup-dir\n```\n\nバックアップディレクトリは、以下のディレクトリ構成です（Simpleアノテーション(v2)と同じディレクトリ構成）。\n`{input_data_id}.json`のフォーマットは、[getEditorAnnotation](https://annofab.com/docs/api/#operation/getEditorAnnotation) APIのレスポンスと同じです。\n\n```\nルートディレクトリ/\n├── {task_id}/\n│\xa0\xa0 ├── {input_data_id}.json\n│\xa0\xa0 ├── {input_data_id}/\n│\xa0\xa0        ├── {annotation_id}............ 塗りつぶしPNG画像\n```\n\n\n\n### annotation import\nアノテーションをプロジェクトにインポートします。\nアノテーションのフォーマットは、Simpleアノテーション(v2)と同じフォルダ構成のzipファイルまたはディレクトリです。\n\n```\nルートディレクトリ/\n├── {task_id}/\n│\xa0\xa0 ├── {input_data_id}.json\n│\xa0\xa0 ├── {input_data_id}/\n│\xa0\xa0        ├── {annotation_id}............ 塗りつぶしPNG画像\n```\n\nJSONフォーマットのサンプルをは以下の通りです。SimpleアノテーションのJSONフォーマットに対応しています。詳しくは https://annofab.com/docs/api/#section/Simple-Annotation-ZIP を参照してください。\n\n\n```json\n{\n    "details": [\n        {\n            "label": "car",\n            "data": {\n                "left_top": {\n                    "x": 878,\n                    "y": 566\n                },\n                "right_bottom": {\n                    "x": 1065,\n                    "y": 701\n                },\n                "_type": "BoundingBox"\n            },\n            "attributes": {}\n        },\n        {\n            "label": "road",\n            "data": {\n                "data_uri": "b803193f-827f-4755-8228-e2c67d0786d9",\n                "_type": "SegmentationV2"\n            },\n            "attributes": {}\n        },\n        {\n            "label": "weather",\n            "data": {\n                "_type": "Classification"\n            },\n            "attributes": {\n                "sunny": true\n            }\n        }\n    ]\n}\n```\n\n以下のように`annotation_id`が指定されている場合、`annotation_id`もインポートされます。\n\n\n```json\n{\n    "details": [\n        {\n            "label": "car",\n            "annotation_id": "12345678-abcd-1234-abcd-1234abcd5678",\n            "data": {\n                "left_top": {\n                    "x": 878,\n                    "y": 566\n                },\n                "right_bottom": {\n                    "x": 1065,\n                    "y": 701\n                },\n                "_type": "BoundingBox"\n            },\n            "attributes": {}\n        },\n\n```\n\nアノテーションをインポートするには、事前に入力データ、タスク、アノテーション仕様を作成する必要があります。\n\nタスクの状態が作業中/完了の場合はインポートしません。\n\n\n```\n# prj1にアノテーションをインポートします。すでにアノテーションが登録されてる場合はスキップします。\n$ annofabcli annotation import --project_id prj1 --annotation simple-annotation.zip \n\n# prj1にアノテーションをインポートします。すでに存在するアノテーションを上書きます。\n$ annofabcli annotation import --project_id prj1 --annotation simple-annotation.zip --overwrite\n```\n\n### annotation list_count\n`task_id`または`input_data_id`で集約したアノテーションの個数を、CSV形式で出力します。\nクエリのフォーマットは、[getAnnotationList API](https://annofab.com/docs/api/#operation/getAnnotationList)のクエリパラメータの`query`キー配下と同じです。\n`label_name_en`(label_idに対応), `additional_data_definition_name_en`(additional_data_definition_idに対応) キーも指定できます。\n\n\n```\n# car ラベルのアノテーション個数を出力する(task_idで集約)\n$ annofabcli annotation list_count --project_id prj1 \\\n --annotation_query \'{"label_name_en": "car"}\'\n\n# car ラベルのアノテーション個数を出力する(input_data_idで集約)\n$ annofabcli annotation list_count --project_id prj1 \\\n --annotation_query \'{"label_name_en": "car"}\' --gropu_by input_data_id\n\n# task.txtに記載されたtask_idの一覧から、car ラベルのアノテーション個数を出力する\n$ annofabcli annotation list_count --project_id prj1 \\\n --annotation_query \'{"label_name_en": "car"}\'  --task_id file://task.txt\n\n# carラベルの"occluded"チェックボックスがONのアノテーションの個数を出力する\n$ annofabcli annotation list_count --project_id prj1 \\\n --annotation_query \'{"label_name_en": "car", "attributes":[{"additional_data_definition_name_en": "occluded", "flag": true}]}\'\n\n# carラベルの"type"ラジオボタン/セレクトボックスが"bus"であるアノテーションの個数を出力する\n$ annofabcli annotation list_count --project_id prj1 \\\n --annotation_query \'{"label_name_en": "car", "attributes":[{"additional_data_definition_name_en": "occluded", "choice_name_en": "bus"}]}\'\n```\n\n#### task_idで集約したときの出力結果（CSV）\n\n| task_id    | annotation_count |\n|------------|------------------|\n| sample_030 | 1                |\n| sample_088 | 2                |\n\n\n#### input_data_idで集約したときの出力結果（CSV）\n\n| task_id    | input_data_id                        | annotation_count |\n|------------|--------------------------------------|------------------|\n| sample_030 | 5738d502-b0a0-4a82-9367-cceffd73cf57 | 1                |\n| sample_093 | dd82cf3a-a38c-4a04-91e7-a4f1ce9af585 | 2                |\n\n\n### annotation restore\n[annotation dump](#annotation-dump) コマンドの出力結果から、アノテーション情報をリストアします。ただし、作業中/完了状態のタスク、または「過去に割り当てられていて現在の担当者が自分自身でない」タスクはリストアできません。\n\n\n```\n# `backup`ディレクトリに保存したアノテーション情報を、`prj1`プロジェクトにリストアする。\n$annofabcli annotation restore --project_id prj1 --task_id file://task.txt --annotation backup-dir\n\n# `backup`ディレクトリに保存したアノテーション情報の内、`task.txt`に記載されたタスクのアノテーション情報を、`prj1`プロジェクトにリストアする。\n$ annofabcli annotation restore --project_id prj1 --task_id file://task.txt --annotation backup-dir\n```\n\n\n\n### annotation_specs history\nアノテーション仕様の履歴一覧を出力します。\n\n```\n# prj1のアノテーション仕様の履歴一覧を出力する\n$ annofabcli annotation_specs history --project_id prj1 \n```\n\n\n\n### annotation_specs list_label\nアノテーション仕様のラベル情報を出力します。\n\n```\n# prj1のアノテーション仕様のラベル情報を、人間が見やすい形式（`--format text`）で出力する\n$ annofabcli annotation_specs list_label --project_id prj1\n\n# prj1のアノテーション仕様のラベル情報を、インデントされたJSONで出力する。\n$ annofabcli annotation_specs list_label --project_id prj1 --format pretty_json\n\n# 最新より１つ前の履歴である、アノテーション仕様を出力する。\n$ annofabcli annotation_specs list_label --project_id prj1 --before 1\n\n# history_idがXXXのアノテーション仕様を出力する。（history_idは、`annofabcli annotation_specs history`コマンドで取得する）\n$ annofabcli annotation_specs list_label --project_id prj1 --history_id XXX\n\n\n```\n\n#### `--format text`の出力結果 \n`--format text`の出力結果は、以下のフォーマットで出力されます。\n\n```\nlabel_id    label_type    label_name_ja    label_name_en\n    attribute_id    attribute_type    attribute_name_ja    attribute_name_ja\n        choice_id    choice_name_ja    choice_name_en\n        ...\n    ...\n...\n```\n\nサンプルプロジェクトでコマンドを実行した結果は、以下の通りです。\n\n```\n15ba7932-24b9-4cf3-95bd-9bf6deede4fa\tbounding_box\tネコ\tCat\n\te6864d96-78fa-45f3-a786-6c8c900c92ae\tflag\t隠れ\toccluded\n\t51e8c91f-5de1-450b-a0f3-94fec582f5ce\tlink\t目のリンク\tlink-eye\n\taff2855e-2e3d-47a2-8c27-c7652e4dfb2f\tinteger\t体重\tweight\n\t7e6a577a-3410-4c8a-9624-2904bb2e6666\tcomment\t名前\tname\n\ta63a0513-a96e-4c7c-8754-88a24fef9ca9\ttext\t備考\tmemo\n\t649abf45-1ed7-459a-8282-a58228e9a302\ttracking\tobject id\tobject id\nc754f724-5f8c-48eb-81ec-ea77e55efee7\tpolyline\t足\tleg\nf50aa88d-36c7-43f5-8728-247a49b4f4d8\tpoint\t目\teye\n108ce1f7-217b-43e9-a407-8d0ac6aad87e\tsegmentation\t犬\tdog\n2ffb4c74-106b-44ac-81ce-3c3df77518e0\tsegmentation_v2\t人間\thuman\nded52dcb-bcd6-4e77-9626-61e546f635d0\tpolygon\t鳥\tbird\n5ac0d7d5-6738-4c4b-a69a-cd583ff458e1\tclassification\t気候\tclimatic\n\t896d7eeb-9c60-4fbf-b7c4-8f4209261049\tchoice\t天気\tweather\n\t\tc9615782-b872-4641-9be4-0fb4f905d966\t\t晴\tsunny\n\t\t553018a5-e594-4536-bc05-876fa6b48ed5\t\t雨\trainy\n\t60caffa5-6300-4819-9a99-c43ce49008c2\tselect\t気温\ttemparature\n\t\t89b3577d-a245-4b85-82ef-6569ecbf8ad7\t\t10\t10\n\t\tbdcd4d5b-cecc-4ec9-9038-d9284cd4f475\t\t20\t20\n\t\t9f3a0355-2cc8-412a-9129-3b62fa7b6ead\t\t30\t30\n\t\t2726336c-96d3-485b-9f96-7d4bcc97083b\t\t40\t40\n\n```\n\n\n### annotation_specs list_label_color\nアノテーション仕様から、label_name(english)とRGBを対応付けたJSONを出力します。\n\n```\n$ annofabcli annotation_specs list_label_color --project_id prj1 \n```\n\n以下のJSONのような出力結果になります。\n\n```json:出力結果\n{\n  "cat": [\n    255,\n    99,\n    71\n  ],\n  "dog": [\n    255,\n    0,\n    255\n  ],\n```\n\n\n\n\n### filesystem write_annotation_image\nアノテーションzip、またはそれを展開したディレクトリから、アノテーションの画像（Semantic Segmentation用）を生成します。\n以下のアノテーションが画像化対象です。\n* 矩形\n* ポリゴン\n* 塗りつぶし\n* 塗りつぶしv2\n\n\n```\n# アノテーションzipをダウンロードする。\n$ annofabcli project download simple_annotation --project_id prj1 --output annotation.zip\n\n\n# label_nameとRGBを対応付けたファイルを生成する\n$ annofabcli annotation_specs list_label_color --project_id prj1 --output label_color.json\n\n\n# annotation.zip から、アノテーション画像を生成する\n$ annofabcli filesystem write_annotation_image  --annotation annotation.zip \\\n --image_size 1280x720 \\\n --label_color file://label_color.json \\\n --output_dir /tmp/output\n\n\n# annotation.zip から、アノテーション画像を生成する。ただしタスクのステータスが"完了"で、task.txtに記載れたタスクのみ画像化する。\n$ annofabcli filesystem write_annotation_image  --annotation annotation.zip \\\n --image_size 1280x720 \\\n --label_color file://label_color.json \\\n --output_dir /tmp/output \\\n --task_status_complete \\\n --task_id file://task.txt\n```\n\n#### 出力結果（塗りつぶし画像）\n\n![filesystem write_annotation_iamgeの塗りつぶし画像](readme-img/write_annotation_image-output.png)\n\n\n### input_data delete\nタスクに使われていない入力データを削除します。\n\n```\n# 入力データ input1, input2 を削除する\n$ annofabcli input_data delete --project_id prj1 --input_data_id input1 input2\n\n# `input_data_id.txt` ファイルに記載されている入力データを削除する\n$ annofabcli input_data delete --project_id prj1 --input_data_id file://input_data_id.txt\n```\n\n\n### input_data list\n入力データ一覧を出力します。\n\n```\n# input_data_nameが"sample"の入力データ一覧を出力する\n$ annofabcli input_data list --project_id prj1 --input_data_query \'{"input_data_name": "sample"}\' \n\n# input_data_idが"id1", "id2"の入力データを取得する\n$ annofabcli input_data list --project_id prj1 --input_data_id id1 id2\n\n# 入力データの詳細情報（参照されているタスクのtask_id `parent_task_id_list`）も出力する\n$ annofabcli input_data list --project_id prj1 --input_data_query \'{"input_data_name": "sample"}\' --add_details\n\n```\n\n\n#### 出力結果（CSV）\n\n\n| etag                             | input_data_id                        | input_data_name                   | input_data_path                                                                                                                                                                 | original_input_data_path                                                                                                                                                        | original_resolution | project_id                           | resized_resolution | sign_required | task_id_list   | updated_datetime              | url                                                                                                                                                                        |\n|----------------------------------|--------------------------------------|-----------------------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|---------------------|--------------------------------------|--------------------|---------------|----------------|-------------------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------|\n| a43717273502b67a1989c9b25e252cde | 3c8d8f15-14f0-467a-a8fe-562cbbccf08a | val.zip/val/9a70bdec-1504e338.jpg | s3://example.com/example | s3://example.com/example |                     | 58a2a621-7d4b-41e7-927b-cdc570c1114a |                    | False         | [\'sample_247\'] | 2019-04-19T16:36:17.846+09:00 | https://annofab.com/projects/example/input_data/example |\n\n\n\n\n### input_data list_merged_task\nタスク一覧と結合した入力データ一覧のCSVを出力します。\n動画プロジェクトで、各タスクの動画時間を調べる際などに利用できます。\n\n```\n# prj1の入力データ全件ファイル、タスク全件ファイルをダウンロードして、マージしたCSVを出力する\n$ annofabcli input_data list_merged_task --project_id prj1 --output input_data.csv\n\n# prj1の入力データ全件ファイル、タスク全件ファイルの最新版をダウンロードして、マージしたCSVを出力する\n$ annofabcli input_data list_merged_task --project_id prj1 --output input_data.csv --latest\n\n# 入力データ全件ファイル、タスク全件ファイルのJSONから、マージしたCSVを出力する\n$ annofabcli input_data list_merged_task --input_data_json input_data.json --task_json task.json --output input_data.csv\n```\n\n\n### input_data put\nCSVに記載された入力データ情報やzipファイルを、入力データとして登録します。\n\n#### CSVに記載された入力データ情報を、入力データとして登録\n\n* ヘッダ行なし\n* カンマ区切り\n* 1列目: input_data_name. 必須\n* 2列目: input_data_path. 必須. 先頭が`file://`の場合、ローカルのファイルを入力データとしてアップロードします。\n* 3列目: input_data_id. 省略可能。省略した場合UUIDv4になる。\n* 4列目: sign_required. 省略可能. `true` or `false`\n\nCSVのサンプル（`input_data.csv`）です。\n\n```\ndata1,s3://example.com/data1,id1,\ndata2,s3://example.com/data2,id2,true\ndata3,s3://example.com/data3,id3,false\ndata4,https://example.com/data4,,\ndata5,file://sample.jpg,,\ndata6,file:///tmp/sample.jpg,,\n```\n\n\n```\n# input_data.csvに記載されている入力データを登録する。すでに入力データが存在する場合はスキップする。\n$ annofabcli input_data put --project_id prj1 --csv input_data.csv\n\n# input_data.csvに記載されている入力データを登録する。すでに入力データが存在する場合は上書きする。\n$ annofabcli input_data put --project_id prj1 --csv input_data.csv --overwrite\n\n# input_data.csvに記載されている入力データを、並列処理で登録する（`--yes`オプションが必要）。\n$ annofabcli input_data put --project_id prj1 --csv input_data.csv --parallelism 2 --yes\n```\n\n\n`input_data list`コマンドを使えば、プロジェクトに既に登録されている入力データからCSVを作成できます。\n\n```\n$ annofabcli input_data list --project_id prj1 --input_data_query \'{"input_data_name": "sample"}\'  \\\n --format csv --output input_data.csv \\\n --csv_format \'{"columns": ["input_data_name","input_data_path", "input_data_id", "sign_required"], "header":false}\' \n```\n\n\n#### zipファイルを入力データとして登録\n\n\n```\n# 画像や動画が格納されたinput_data.zipを、入力データとして登録する\n$ annofabcli input_data put --project_id prj1 --zip input_data.zip\n\n# zipファイルを入力データとして登録し、入力データの登録が完了するまで待つ。\n$ annofabcli input_data put --project_id prj1 --zip input_data.zip --wait\n\n# zipファイルを入力データとして登録する。そのときinput_data_nameを`foo.zip`にする。\n$ annofabcli input_data put --project_id prj1 --zip input_data.zip --input_data_name_for_zip foo.zip\n\n```\n\n\n\n\n### inspection_comment list\n検査コメント一覧を出力します。\n\n```\n# task1, task2の検査コメント一覧を、CSVで出力する\n$ annofabcli inspection_comment list --project_id prj1 --task_id task1 task2\n\n# タブ区切りの"out.tsv"を出力する\n$ annofabcli inspection_comment list --project_id prj1 --task_id task1 task2 \\\n --format csv --csv_format \'{"sep":"\\t"}\'  --output out.tsv\n\n# JSONで出力する\n$ annofabcli inspection_comment list --project_id prj1 --task_id file://task.txt --format json\n\n# 検査コメント情報が記載されたファイルを元にして、検査コメント一覧を出力します\n# 検査コメント情報が記載されたファイルは、`$ annofabcli project download inspection_comment`コマンドで取得できます。\n$ annofabcli inspection_comment list --project_id prj1 --inspection_comment_json inspection_comment.json\n\n# 返信コメントを除外した検査コメント一覧を出力します\n$ annofabcli inspection_comment list --project_id prj1 --task_id task1 task2 --exclude_reply\n\n# 返信コメントのみの一覧を出力します\n$ annofabcli inspection_comment list --project_id prj1 --task_id task1 task2 --only_reply\n\n\n```\n\n#### 出力結果（CSV）\n\n| project_id                           | task_id    | input_data_id                        | inspection_id                        | phase      | phase_stage | commenter_account_id                 | annotation_id                        | label_id                             | data                                  | parent_inspection_id | phrases | comment | status          | created_datetime              | updated_datetime              | commenter_user_id | commenter_username | phrase_names_en | phrase_names_ja | label_name_en | label_name_ja | input_data_index |\n|--------------------------------------|------------|--------------------------------------|--------------------------------------|------------|-------------|--------------------------------------|--------------------------------------|--------------------------------------|---------------------------------------|----------------------|---------|---------|-----------------|-------------------------------|-------------------------------|-------------------|--------------------|-----------------|-----------------|---------------|---------------|------------------|\n| 58a2a621-7d4b-41e7-927b-cdc570c1114a | sample_180 | bf6b4790-cdb8-4d4d-85bb-08550934ed61 | 5f096677-67e4-4e75-9eac-bbd8ac9694d9 | inspection | 1           | 12345678-abcd-1234-abcd-1234abcd5678 | 8aff181e-9df4-4c66-8fb2-10596c686d5c | 8aff181e-9df4-4c66-8fb2-10596c686d5c | {\'x\': 358, \'y\': 48, \'_type\': \'Point\'} |                      | []      | 枠がずれています     | error_corrected | 2019-07-26T17:41:16.626+09:00 | 2019-08-01T10:57:45.639+09:00 | user_id   | username          | []              | []              | car           | car           | 0                |\n\n\n### inspection_comment list_unprocessed\n未処置の検査コメント一覧を出力します。\n\n```\n# 未処置の検査コメント一覧を出力する\n$ annofabcli inspection_comment list_unprocessed --project_id prj1 --task_id file://task.txt\n\n# 未処置で、user1が"hoge"とコメントした検査コメント一覧を出力する\n$ annofabcli inspection_comment list_unprocessed  --project_id prj1 --task_id file://task.txt \\\n --inspection_comment "hoge" --commenter_user_id user1 --format pretty_json --output inspection.json\n\n# 検査コメント情報が記載されたファイルを元にして、検査コメント一覧を追加します\n$ annofabcli inspection_comment list_unprocessed --project_id prj1 --inspection_comment_json inspection_comment.json\n```\n\n\n### instruction copy\n作業ガイドを別のプロジェクトにコピーします。\n\n\n```\n# prj1の作業ガイドをprj2にコピーする\n$ annofabcli instruction copy prj1 prj2\n```\n\n\n\n### instruction upload\nHTMLファイルを作業ガイドとして登録します。\nimg要素のsrc属性がローカルの画像を参照している場合（http, https, dataスキーマが付与されていない）、画像もアップロードします。\n\n`instruction.html`の中身。\n\n```html\n<html>\n<head></head>\n<body>\n作業ガイドのサンプル\n<img src="lena.png">\n</body>\n</html>\n```\n\n```\n$ annofabcli instruction upload --project_id prj1 --html instruction.html\n```\n\n\n#### Confluenceのページを作業ガイド用にHTMLとして保存する場合\n1. Confluenceのエクスポート機能で、ページをエクスポートする。\n    * HTMLファイルと添付画像が含まれたzipファイルをダウンロードする。\n2. エクスポートしたHTMLのスタイルを、style属性に反映させる。AnnoFabの作業ガイドには、スタイルシートを登録できないため。\n    1. エクスポートしたファイルをChromeで開く\n    2. Chrome開発ツールのConfoleタブで以下のJavaScriptを実行して、全要素のborder, color, backgroundスタイルを、style属性に反映させる\n\n        ```js\n        elms = document.querySelectorAll("body *");\n        for (let e of elms) {\n            s = window.getComputedStyle(e);\n            e.style.background = s.background;\n            e.style.color = s.color;\n            e.style.border = s.border;\n        }\n        ```\n    3. Chrome開発ツールのElementタブで、html要素をコピー(Copy outerHTML)して、HTMLファイルを上書きする\n\n\n### job delete\nジョブを削除します。\n削除対象のjob_idは`annofabcli job list`コマンドで確認できます。\n\n```\n# アノテーション更新のジョブを削除します。\n$ annofabcli job delete --project_id prj1 --job_type gen-annotation --job_id 12345678-abcd-1234-abcd-1234abcd5678\n```\n\n\n### job list\nジョブ一覧を出力します。\n\n```\n# アノテーション更新のジョブ一覧を取得します（最新のジョブ1個のみ）\n$ annofabcli job list --project_id prj1 --job_type gen-annotation\n\n# タスク作成のジョブ一覧を取得します（最大200個）\n$ annofabcli job list --project_id prj1 --job_type gen-tasks --job_query \'{"limit": 200}\'\n\n```\n\n\n\n### job list_last\n複数のプロジェクトに対して、最新のジョブを出力します。\n\n```\n# prj1, prj2に対して、「アノテーション更新」のジョブを出力します。\n$ annofabcli job list_last --project_id prj1 prj2 --job_type gen-annotation\n\n# 組織 org1配下のプロジェクト（進行中で、自分自身が所属している）に対して、「タスク全件ファイル更新」のジョブを出力します。\n$ annofabcli job list_last --organization org1 --job_type gen-tasks-list\n\n# アノテーションの最終更新日時を、タスクの最終更新日時と比較して出力します。\n$ annofabcli job list_last --project_id prj1 --job_type gen-annotation --add_details \\\n --csv_format \'{"columns": ["project_id","project_title","job_status","updated_datetime", "task_last_updated_datetime"]}\' \n\n```\n\n\n### labor list_worktime_by_user\n\nユーザごとに作業予定時間、作業実績時間を出力します。\n\n```\n# 組織org1, org2に対して、user1, user2の作業時間を集計します。\n$ annofabcli labor list_worktime_by_user --organization org1 org2 --user_id user1 user2 \\\n --start_date 2019-10-01 --end_date 2019-10-31 --output_dir /tmp/output\n\n# プロジェクトprj1, prj2に対して作業時間を集計します。集計対象のユーザはプロジェクトに所属するメンバです。\n$ annofabcli labor list_worktime_by_user --project_id prj1 prj2 --user_id user1 user2 \\\n --start_date 2019-10-01 --end_date 2019-10-31 --output_dir /tmp/output\n\n\n# user.txtに記載されているユーザの予定稼働時間も一緒に出力します。\n$ annofabcli labor list_worktime_by_user --project_id prj1 prj2 --user_id file://user.txt \\\n --start_month 2019-10 --end_month 2019-11 --add_availability --output_dir /tmp/output\n\n```\n\n\n### organization_member list\n組織メンバ一覧を出力します。\n\n```\n# 組織org1の組織メンバ一覧を出力します。\n$ annofabcli organization_member list --organization org1\n\n```\n\n\n\n### project cooy\nプロジェクトをコピーして（アノテーション仕様やメンバーを引き継いで）、新しいプロジェクトを作成します。\n\n\n\n```\n# prj1 プロジェクトをコピーして、"prj2-title"というプロジェクトを作成する\n$ annofabcli project copy --project_id prj1 --dest_title "prj2-title"\n\n\n# prj1 プロジェクトをコピーして、"prj2"というプロジェクトIDのプロジェクトを作成する。\n# コピーが完了するまで待つ(処理を継続する)\n$ annofabcli project copy --project_id prj1 --dest_title "prj2-title" --dest_project_id prj2 \\\n --wait\n\n\n# prj1 プロジェクトの入力データと、タスクをコピーして、"prj2-title"というプロジェクトを作成する\n$ annofabcli project copy --project_id prj1 --dest_title "prj2-title" --copy_inputs --copy_tasks\n\n\n```\n\n\n\n\n### project diff\nプロジェクト間の差分を、以下の項目について表示します。差分がない場合、標準出力は空になります。\n* アノテーション仕様のラベル情報\n* 定型指摘\n* プロジェクトメンバ\n* プロジェクトの設定\n\n\n```\n# すべての差分\n$ annofabcli project diff  prj1 prj2\n\n# アノテーション仕様のラベル情報の差分\n$ annofabcli project diff prj1 prj2 --target annotation_labels\n\n# 定型指摘の差分\n$ annofabcli project diff prj1 prj2 --target inspection_phrases\n\n# プロジェクトメンバの差分\n$ annofabcli project diff  prj1 prj2 --target members\n\n# プロジェクト設定の差分\n$ annofabcli project diff  prj1 prj2 --target settings\n\n```\n\n\n\nプロジェクト間の差分は、以下のように出力されます。\n`dict`型の差分は、[dictdiffer](https://dictdiffer.readthedocs.io/en/latest/)のフォーマットで出力します。\n\n```\n=== prj1_title1(prj1) と prj1_title2(prj2) の差分を表示\n=== プロジェクトメンバの差分 ===\nプロジェクトメンバは同一\n=== プロジェクト設定の差分 ===\nプロジェクト設定は同一\n=== 定型指摘の差分 ===\n定型指摘は同一\n=== アノテーションラベル情報の差分 ===\nラベル名(en): car は差分あり\n[(\'change\', \'color.red\', (4, 0)),\n (\'change\', \'color.green\', (251, 255)),\n (\'change\', \'color.blue\', (171, 204))]\nラベル名(en): bike は同一\n```\n\n\n\n### project download\nプロジェクトに対して、タスクや検査コメント、アノテーションなどをダウンロードします。\nダウンロード対象は以下の通りです。\n* すべてのタスクが記載されたJSON\n* すべての入力データが記載されたJSON\n* すべての検査コメントが記載されたJSON\n* すべてのタスク履歴イベントが記載されたJSON\n* すべてのタスク履歴イベントが記載されたJSON（非推奨）\n* Simpleアノテーションzip\n* Fullアノテーションzip（非推奨）\n\n\n```\n# タスクの全一覧が記載されたJSONファイルをダウンロードする\n$ annofabcli project download task --project_id prj1 --output task.json\n\n# 入力データのすべてが記載されたJSONファイルをダウンロードする\n$ annofabcli project download input_data --project_id prj1 --output input_data.json\n\n# 検査コメントの全一覧が記載されたJSONファイルをダウンロードする\n$ annofabcli project  download inspection_comment --project_id prj1 --output inspection_comment.json\n\n# タスク履歴イベントの全一覧が記載されたJSONファイルをダウンロードする\n$ annofabcli project download task_history --project_id prj1 --output task_history.json\n\n# Simpleアノテーションのzipファイルをダウンロードする\n$ annofabcli project download simple_annotation --project_id prj1 --output simple_annotation.zip\n\n# 最新のSimpleアノテーションのzipファイルをダウンロードする\n$ annofabcli project download simple_annotation --project_id prj1 --output simple_annotation.zip --latest\n\n# 最新のタスク全一覧が記載されたJSONファイルをダウンロードする\n$ annofabcli project download task --project_id prj1 --output task.json --latest\n\n# アノテーションの最新化を最大60分(60秒間隔で最大60回アクセス)待つ\n$ annofabcli project download simple_annotation --project_id prj1  58a2a621-7d4b-41e7-927b-cdc570c1114a --output simple_annotation.zip --latest \\\n --wait_options \'{"interval":60, "max_tries":60}\' \n```\n\n\n### project list\nプロジェクト一覧を出力します。\n\n```\n# org1配下のプロジェクトで、user1が所属する進行中のプロジェクト一覧を出力する。\n$ annofabcli project list --organization org1 --project_query \'{"status": "active", "user_id": "user1}\'\n\n# prj1, prj2のプロジェクト一覧を出力する\n$ annofabcli project list --project_id prj1 prj2\n```\n\n\n### project update_annotation_zip\nアノテーションzipを更新します。\n\n```\n# prj1, prj2のアノテーションzipを更新します。更新する必要がなければ更新しません。\n$ annofabcli project update_anotation_zip --project_id prj1 prj2\n\n# prj1, prj2のアノテーションzipを更新します。更新する必要がなくても常に更新します。\n$ annofabcli project update_anotation_zip --project_id prj1 prj2 --force\n\n# prj1, prj2のアノテーションzipを更新して、すべてのプロジェクトの更新が完了するまで待ちます\n$ annofabcli project update_anotation_zip --project_id prj1 prj2 --wait\n\n# アノテーションzipの更新が完了するまで待つ処理を並列で実行します。\n$ annofabcli project update_anotation_zip --project_id prj1 prj2 --wait --parallelism 4\n```\n\n\n\n### project_member change\n複数のプロジェクトメンバに対して、メンバ情報を変更します。ただし、自分自身は変更できません。\n\n```\n# user1, user2のロールを"worker"（アノテータ）に変更する\n$ annofabcli project_member change --project_id prj1 --user_id user1 user2 --role worker\n\n# `user_id.txt`に記載されたuser_idに対して、抜取検査率、抜取受入率を指定する\n$ annofabcli project_member change --project_id prj1 --user_id file://user_id.txt \\\n --member_info \'{"sampling_inspection_rate": 10, "sampling_acceptance_rate": 20}\'\n\n# すべてのユーザに対して、抜取検査率を未設定にする\n$ annofabcli project_member change --project_id prj1 --all_user \\\n --member_info \'{"sampling_inspection_rate": null}\'\n\n```\n\n\n\n### project_member copy\nプロジェクトメンバを別のプロジェクトにコピーします。\n\n```\n# prj1のメンバをprj2にコピーする。\n$ annofabcli project_member copy prj1 prj2\n\n# prj1のメンバをprj2にコピーする。prj2にしか存在しないメンバは削除される。\n$ annofabcli project_member copy prj1 prj2 --delete_dest\n```\n\n\n\n### project_member delete\n複数のプロジェクトからユーザを削除します。\n\n```\n# ORG組織配下のすべてのプロジェクトから、user1, user2を削除する\n$ annofabcli project_member delete --user_id user1 user2  --organization ORG\n\n# prj1, prj2のプロジェクトからuser1をaccepterロールで招待する\n$ annofabcli project_member invite --user_id user1  --project_id prj1 prj2\n```\n\n\n\n### project_member invite\n複数のプロジェクトに、ユーザを招待します。\n\n```\n# ORG組織配下のすべてのプロジェクトに、user1, user2をownerロールで招待する\n$ annofabcli project_member invite --user_id user1 user2 --role owner --organization ORG\n\n# prj1, prj2のプロジェクトに、user1をaccepterロールで招待する\n$ annofabcli project_member invite --user_id user1 --role accepter --project_id prj1 prj2\n```\n\n\n### project_member list\nプロジェクトメンバ一覧を出力します。\n\n```\n# ORG組織配下のすべてのプロジェクトのプロジェクトメンバ一覧を出力する\n$ annofabcli project_member list --organization ORG\n\n# prj1, prj2のプロジェクトのプロジェクトメンバ一覧を出力する\n$ annofabcli project_member list --project_id prj1 prj2\n```\n\n#### 出力結果（CSV）\n\n| project_id                           | account_id                           | user_id         | username  | member_status | member_role | updated_datetime              | created_datetime              | sampling_inspection_rate | sampling_acceptance_rate | project_title                |\n|--------------------------------------|--------------------------------------|-----------------|-----------|---------------|-------------|-------------------------------|-------------------------------|--------------------------|--------------------------|------------------------------|\n| 12345678-abcd-1234-abcd-1234abcd5678 | 12345678-abcd-1234-abcd-1234abcd5678 | user_id | username | active        | owner       | 2019-09-10T14:51:00.908+09:00 | 2019-04-19T16:29:41.069+09:00 |                          |                          | sample_project |\n\n\n### project_member put\nCSVに記載されたユーザを、プロジェクトメンバとして登録します。\n\nmembers.csvの中身は以下の通りです。\n\n* ヘッダ行なし\n* カンマ区切り\n* 1列目: user_id. 必須\n* 2列目: member_role. 必須.  `owner`, `worker`, `accepter`, `training_data_user` のいずれか。\n* 3列目: sampling_inspection_rate. 省略可能。\n* 4列目: sampling_acceptance_rate. 省略可能。\n\n\n```\nuser1,worker\nuser2,accepter,80,40\n```\n\n\n```\n# CSVに記載れたユーザを、prj1プロジェクトのメンバとして登録します。\n$ annofabcli project_member put --project_id prj1 --csv members.csv\n\n# CSVに記載れたユーザを、prj1プロジェクトのメンバとして登録します。csvに記載されていないユーザは削除します。\n$ annofabcli project_member put --project_id prj1 --csv members.csv --delete\n```\n\n### statistics list_annotation_count\n各ラベル、各属性値のアノテーション数を、タスクごと/入力データごとに出力します。\n`--annotation`にはAnnoFabからダウンロードしたSimpleアノテーションzipのパスを渡します。指定しない場合はAnnoFabからダウンロードします。\n出力結果には以下のファイルが含まれています。\n* `labels_count.csv`：各ラベルのアノテーション数\n* `attirbutes_count.csv`：各属性値のアノテーション数（ただし属性の種類がチェックボックス、ラジオボタン、セレクトボックスの属性のみが対象）\n\n\n```\n# タスクごとにアノテーション数を、output ディレクトリに出力\n$ annofabcli statistics list_annotation_count --project_id prj1 --output_dir output --annotation annotataion.zip\n\n# 入力データごとにアノテーション数を、output ディレクトリに出力。アノテーション情報はAnnoFabからダウンロードする\n$ annofabcli statistics list_annotation_count --project_id prj1 --output_dir output --group_by input_data_id\n\n```\n\n\n### statistics list_by_date_user\n\nタスク数や作業時間などの情報を、日ごとユーザごとに出力します。\n\n```\n$ annofabcli statistics list_by_date_user --project_id prj1 --output data.csv\n\n```\n\n\n\n### statistics list_cumulative_labor_time\nタスクフェーズ別の累積作業時間をCSV形式で出力します。\n\n```\n$ annofabcli statistics list_cumulative_labor_time --project_id prj1 --output stat.csv\n```\n\n### statistics list_task_progress\nタスク進捗状況をCSV形式で出力します。\n\n```\n$ annofabcli statistics list_task_progress --project_id prj1 --output stat.csv\n```\n\n### statistics summarize_task_count\nタスクのフェーズ、ステータス、ステップごとにタスク数を、CSV形式で出力します。\n「1回目の教師付」と「2回目の教師付」を区別して集計されます。\n\n\n```\n# prj1のタスク数を出力します。ダウンロードしたタスク全件ファイルを元にして出力します（AM02:00頃更新）。\n$ annofabcli statistics summarize_task_count --project_id prj1 --output task-count.csv\n\n# `annofabcli project download task`でダウンロードした`task.json`を元にして、タスク数を出力します。\n$ annofabcli statistics summarize_task_count --project_id prj1 --task_json task.json --output task-count.csv\n\n```\n\n以下のようなCSVが出力されます。\n\n```csv\nstep,phase,phase_stage,simple_status,task_count\n1,annotation,1,not_started,3761\n1,annotation,1,working_break_hold,30\n1,acceptance,1,not_started,1861\n1,acceptance,1,working_break_hold,20\n2,annotation,1,not_started,225\n2,annotation,1,working_break_hold,3\n2,acceptance,1,not_started,187\n5,acceptance,1,not_started,1\n,acceptance,1,complete,3000\n```\n\n\n* step：何回目のフェーズか\n* simple_status：タスクステータスを簡略化したもの\n    * not_started：未着手\n    * working_break_hold：作業中か休憩中か保留中\n    * complete：完了\n\n「一度も作業されていない教師付未着手」のタスク数は、先頭行（step=1, phase=annotation, simple_status=not_started）のtask_countから分かります。\n\n\n### statistics summarize_task_count_by_task_id\ntask_idのプレフィックスごとに、タスク数をCSV形式で出力します。\ntask_idは`{prefix}_{連番}`のようなフォーマットを想定しています。\n\n\n```\n# prj1のタスク数を出力します。ダウンロードしたタスク全件ファイルを元にして出力します（AM02:00頃更新）。\n$ annofabcli statistics summarize_task_count_by_task_id --project_id prj1 --output task-count.csv\n\n# `annofabcli project download task`でダウンロードした`task.json`を元にして、タスク数を出力します。\n$ annofabcli statistics summarize_task_count_by_task_id --project_id prj1 --task_json task.json --output task-count.csv\n\n```\n\n以下のようなCSVが出力されます。\n\n```csv\ntask_id_prefix,complete,on_hold,annotation_not_started,inspection_not_started,acceptance_not_started,other,sum\n20200401,10,0,0,0,0,0,10\n20200501,10,1,4,0,1,4,20\n```\n\n各列\n* annotation_not_started: 教師付フェーズが一度も作業されていないタスク数\n* inspection_not_started: 検査フェーズが一度も作業されていないタスク数\n* acceptance_not_started: 受入フェーズが一度も作業されていないタスク数\n* other: 休憩中、作業中、\n* simple_status：タスクステータスを簡略化したもの\n    * not_started：未着手\n    * working_break_hold：作業中か休憩中か保留中\n    * complete：完了\n\n「一度も作業されていない教師付未着手」のタスク数は、先頭行（step=1, phase=annotation, simple_status=not_started）のtask_countから分かります。\n\n\n### statistics summarize_task_count_by_user\nユーザごとに担当しているタスク数をCSV形式で出力します。\n\n\n```\n# prj1のタスク数を出力します。ダウンロードしたタスク全件ファイルを元にして出力します（AM02:00頃更新）。\n$ annofabcli statistics summarize_task_count_by_user --project_id prj1 --output task-count.csv\n\n# `annofabcli project download task`でダウンロードした`task.json`を元にして、タスク数を出力します。\n$ annofabcli statistics summarize_task_count_by_task_id --project_id prj1 --task_json task.json --output task-count.csv\n\n```\n\n以下のようなCSVが出力されます。\n\n```csv\ntask_id_prefix,complete,on_hold,annotation_not_started,inspection_not_started,acceptance_not_started,other,sum\n20200401,10,0,0,0,0,0,10\n20200501,10,1,4,0,1,4,20\n```\n\n各列\n* annotation_not_started: 教師付フェーズが一度も作業されていないタスク数\n* inspection_not_started: 検査フェーズが一度も作業されていないタスク数\n* acceptance_not_started: 受入フェーズが一度も作業されていないタスク数\n* other: 休憩中、作業中、\n* simple_status：タスクステータスを簡略化したもの\n    * not_started：未着手\n    * working_break_hold：作業中か休憩中か保留中\n    * complete：完了\n\n「一度も作業されていない教師付未着手」のタスク数は、先頭行（step=1, phase=annotation, simple_status=not_started）のtask_countから分かります。\n\n\n### statistics visualize\n統計情報を可視化します。\n\n```\n# prj1の統計情報を可視化したファイルを、/tmp/outputに出力する\n$ annofabcli statistics visualize --project_id prj1 --output_dir /tmp/output\n\n# statusがcompleteのタスクを統計情報を可視化したファイルを、/tmp/outputに出力する\n$ annofabcli statistics visualize --project_id prj1 --output_dir /tmp/output \\\n  --task_query \'{"status": "complete"}\' \n\n# アノテーションzipを更新してから、アノテーションzipをダウンロードします。\n$ annofabcli statistics visualize --project_id prj1 --output_dir /tmp/output --update_annotation\n\n\n# WebAPIを実行せずに、作業ディレクトリ（デフォルトは`$XDG_CACHE_HOME/annofabcli`）内のファイルを参照して、統計情報を可視化する。\n$ annofabcli statistics visualize --project_id prj1 --not_update\n\n```\n\n### supplementary list\n補助情報一覧を出力します。\n\n```\n# input_data_idが"id1", "id2"に紐づく補助情報一覧を出力します。\n$ annofabcli supplementary list --project_id prj1 --input_data_id id1 id2\n```\n\n### supplementary put\nCSVに記載された補助情報を登録します。\n\nsupplementary_data_id（省略時は supplementary_data_number）が一致する補助情報が既に存在する場合は、スキップまたは上書きします。\n\n* ヘッダ行なし\n* カンマ区切り\n* 1列目: input_data_id. 必須\n* 2列目: supplementary_data_number. 必須\n* 3列目: supplementary_data_name. 必須\n* 4列目: supplementary_data_path. 必須. 先頭が`file://`の場合、ローカルのファイルを入力データとしてアップロードします。\n* 5列目: supplementary_data_id. 省略可能。省略した場合UUIDv4になる。\n* 6列目: supplementary_data_type. 省略可能. `image` or `text`\n\nCSVのサンプル（`supplementary_data.csv`）です。\n\n```\ninput1,1,data1-1,s3://example.com/data1,id1,\ninput1,2,data1-2,s3://example.com/data2,id2,image\ninput1,3,data1-3,s3://example.com/data3,id3,text\ninput2,1,data2-1,https://example.com/data4,,\ninput2,2,data2-2,file://sample.jpg,,\ninput2,3,data2-3,file:///tmp/sample.jpg,,\n```\n\n\n```\n# supplementary_data.csvに記載されている補助情報を登録する。すでに補助情報が存在する場合はスキップする。\n$ annofabcli supplementary put --project_id prj1 --csv supplementary_data.csv\n\n# supplementary_data.csvに記載されている補助情報を登録する。すでに補助情報が存在する場合は上書きする。\n$ annofabcli supplementary put --project_id prj1 --csv supplementary_data.csv --overwrite\n\n# supplementary_data.csvに記載されている補助情報を、並列処理で登録する（`--yes`オプションが必要）。\n$ annofabcli supplementary put --project_id prj1 --csv supplementary_data.csv --parallelism 2 --yes\n```\n\n\n`supplementary list`コマンドを使えば、プロジェクトに既に登録されている補助情報からCSVを作成できます。\n\n```\n$ annofabcli supplementary list --project_id prj1 --input_data_id id1 id2 \\\n --format csv --output supplementary_data.csv \\\n --csv_format \'{"columns": ["input_data_id", "supplementary_data_number", "supplementary_data_name", "supplementary_data_path", "supplementary_data_id", "supplementary_data_type"], "header":false}\' \n```\n\n\n\n### task cancel_acceptance\n受け入れ完了タスクに対して、受け入れ取り消しにします。\nアノテーションルールを途中で変更したときなどに、利用します。\n\n\n```\n# prj1プロジェクトのタスクを、受け入れ取り消しにする。最後に受入を担当したユーザに割り当てる\n$ annofabcli task cancel_acceptance --project_id prj1 --task_id file://task.txt\n\n# prj1プロジェクトのタスクを、受け入れ取り消しにする。ユーザuser1に割り当てる。\n$ annofabcli task cancel_acceptance --project_id prj1 --task_id file://task.txt --assigned_acceptor_user_id user1\n\n# prj1プロジェクトのタスクを、受け入れ取り消しにする。担当者は未割り当て\n$ annofabcli task cancel_acceptance --project_id prj1 --task_id file://task.txt --not_assign\n\n\n```\n\n\n### task change_operator\nタスクの担当者を変更します。\n\n\n```\n# 指定されたタスクの担当者を \'user1\' に変更する。\n$ annofabcli task change_operator --project_id prj1 --task_id file://task.txt --user_id uer1\n\n# 指定されたタスクの担当者を未割り当てに変更する。\n$ annofabcli task change_operator --project_id prj1 --task_id file://task.txt --not_assign\n\n# 教師付フェーズが未着手のタスクのみ、担当者を変更する\n$ annofabcli task change_operator --project_id prj1 --task_id file://task.txt --not_assign --task_query \'{"status:"not_started", "phase"}\'\n\n```\n\n\n### task complete\nタスクを完了状態にして次のフェーズに進めます。（教師付の提出、検査/受入の合格） 。\n教師付フェーズを完了にする場合は、未回答の検査コメントに対して返信することができます（未回答の検査コメントに対して返信しないと、タスクを提出できないため）。\n検査/受入フェーズを完了する場合は、未処置の検査コメントを対応完了/対応不要状態に変更できます（未処置の検査コメントが残っている状態では、タスクを合格にできないため）。\n\n\n```\n# 教師付フェーズのタスクを提出して、次のフェーズに進めます。未回答の検査コメントがある場合は、スキップします。\n$ annofabcli task complete --project_id prj1 --task_id file://task.txt --phase annotation\n\n# 教師付フェーズのタスクを提出して、次のフェーズに進めます。未回答の検査コメントには「対応しました」と返信します。\n$ annofabcli task complete --project_id prj1 --task_id file://task.txt --phase annotation --reply_comment "対応しました"\n\n# 検査フェーズでステージ2のタスクを合格にして、次のフェーズに進めます。未処置の検査コメントがある場合は次に進めます\n$ annofabcli task complete --project_id prj1 --task_id file://task.txt --phase inspection --phase_stage 2\n\n# 受入フェーズのタスクを合格にして、次のフェーズに進めます。未処置の検査コメントは「対応不要」状態にします。\n$ annofabcli  task complete --project_id prj1 --task_id file://task.txt --phase acceptance \\\n --inspection_status no_correction_required \n```\n\n\n\n### task delete\nタスクを削除します。ただし、作業中/完了状態のタスクは削除できません。デフォルトでは、アノテーションが付与されているタスクは削除できません。\nアノテーションの削除には [annotation delete](#annotation-delete) コマンドを利用できます。\n\n```\n# task_id.txtに記載されたtask_idのタスクを削除します。\n$ annofabcli task delete --project_id prj1 --task_id file://task_id.txt\n\n# アノテーションが付与されているかどうかにかかわらず、タスクを削除します。\n$ annofabcli task delete --project_id prj1 --task_id file://task_id.txt --force\n\n```\n\n\n### task list\nタスク一覧を出力します。\n\n```\n# 受入フェーズで、"usr1"が担当しているタスクの一覧を出力する\n$ annofabcli task list --project_id prj1 --task_query \'{"user_id": "usr1","phase":"acceptance"}\' \n\n# task_id"id1", "id2"のタスクを取得する\n$ annofabcli task list --project_id prj1 --task_id id1 id2\n\n# 休憩中で、過去の担当者が"usr1"であるタスクの一覧を出力する。task.jsonファイルにJSON形式で出力する。\n$ annofabcli task list --project_id prj1 \\\n --task_query \'{"previous_user_id": "usr1","status":"break"}\' --format json --out task.json\n\n# 差し戻されたタスクのtask_idを出力する\n$ annofabcli task list --project_id prj1 --task_query \'{"rejected_only": true}\' --format task_id_list \n\n# タスク情報が記載されたファイルを元にして、タスク一覧を出力します\n# タスク情報が記載されたファイルは、`$ annofabcli project download task`コマンドで取得できます。\n$ annofabcli task list --project_id prj1 --task_json task.json\n\n\n```\n\n#### 出力結果\n\n| project_id                           | task_id                                | phase      | phase_stage | status      | input_data_id_list                       | account_id                           | histories_by_phase                                                                                                                                       | work_time_span | number_of_rejections | started_datetime              | updated_datetime              | sampling | user_id         | username  | worktime_hour       | number_of_rejections_by_inspection | number_of_rejections_by_acceptance |\n|--------------------------------------|----------------------------------------|------------|-------------|-------------|------------------------------------------|--------------------------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------|----------------|----------------------|-------------------------------|-------------------------------|----------|-----------------|-----------|---------------------|------------------------------------|------------------------------------|\n| 12345678-abcd-1234-abcd-1234abcd5678 | 12345678-abcd-1234-abcd-1234abcd5678   | annotation | 1           | break       | [\'12345678-abcd-1234-abcd-1234abcd5678\'] | 12345678-abcd-1234-abcd-1234abcd5678 | [{\'account_id\': \'12345678-abcd-1234-abcd-1234abcd5678\', \'phase\': \'annotation\', \'phase_stage\': 1, \'user_id\': \'user_id1\', \'username\': \'username1\'}] | 539662         | 0                    | 2019-05-08T13:53:21.338+09:00 | 2019-05-08T14:15:07.318+09:00 |          | user_id1 | user_name2 | 0.14990611111111113 | 0                                  | 0                                  |\n\n\n\n\n\n### task list_added_task_history\nタスク履歴情報（フェーズごとの作業時間、担当者、開始日時）を加えたタスク一覧をCSV形式で出力します。\n最初に教師付を開始した日時や担当者を調べるときなどに利用できます。\n\n\n```\n# prj1のタスク全件ファイル、タスク履歴全件ファイルをダウンロードして、タスク一覧のCSVを出力する\n$ annofabcli task list_added_task_history --project_id prj1 --output task.csv\n\n# prj1のタスク全件ファイルの最新版をダウンロードして、タスク一覧のCSVを出力する。タスク履歴全件ファイルはWebAPIの都合上最新化できません。\n$ annofabcli task list_added_task_history --project_id prj1 --output task.csv --latest\n\n# タスク全件ファイルJSON、タスク履歴全件ファイルJSONを参照して、タスク一覧のCSVを出力する\n$ annofabcli task list_added_task_history --project_id prj1 --output task.csv --task_json task.json --task_history_json task_history.json\n```\n\n### task list_task_history\nタスク履歴の一覧を出力します。\n\n\n```\n# prj1の全タスクのタスク履歴の一覧を、CSV形式で出力する\n$ annofabcli task list_task_history --project_id prj1 --output task_history.csv\n\n```\n\n\n### task put\nタスクを作成します。\n\n### CSVファイルに記載された情報を元にタスクを作成する場合\nCSVに記載された情報を元に、タスクを登録します。\nCSVのフォーマットは以下の通りです。\n\n```\n1列目: task_id\n2列目: 空欄（どんな値でもよい）\n3列目: input_data_id\n```\n\nCSVのサンプル（`task.csv`）です。\n\n```\ntask_1,,ca0cb2f9-fec5-49b4-98df-dc34490f9785\ntask_1,,5ac1987e-ca7c-42a0-9c19-b5b23a41836b\ntask_2,,4f2ae4d0-7a38-4f9a-be6f-170ba76aba73\ntask_2,,45ac5852-f20c-4938-9ee9-cc0274401df7\n```\n\n```\n# prj1に、タスク登録処理を投入する。\n$ annofabcli task put --project_id prj1 --csv task.csv\n\n# prj1に、タスク登録処理を投入する。タスク登録が完了するまで待つ.\n$ annofabcli task put --project_id prj1 --csv task.csv --wait\n```\n\n#### 1つのタスクに割り当てる入力データの個数を指定してタスクを作成する場合\n1つのタスクに割り当てる入力データの個数などの情報を、`--by_count`引数に指定します。\nフォーマットは [initiateTasksGeneration](https://annofab.com/docs/api/#operation/initiateTasksGeneration) APIのリクエストボディ `task_generate_rule` を参照してください。\n\n```\n# 「タスクIDのプレフィックスを"sample"、1タスクの入力データ数を10個」でタスクを作成する。\n$ annofabcli task  put --project_id prj1 --by_count \'{"task_id_prefix":"sample","input_data_count":10}\' \n\n# 「タスクIDのプレフィックスを"sample"、1タスクの入力データ数を10個、入力データ名を降順、\n# 既にタスクに使われている入力データも利用」でタスクを作成し、タスク作成が完了するまで待つ\n$ annofabcli task  put --project_id prj1 --by_count \'{"task_id_prefix":"sample","input_data_count":10, \\\n "input_data_order":"random", "allow_duplicate_input_data":true}\' --wait\n```\n\n\n### task reject\nタスクを強制的に差し戻します。差し戻す際に検査コメントを付与することもできます。検査コメントは、画像プロジェクトならばタスク内の先頭の画像の左上(`x=0,y=0`)に、動画プロジェクトなら動画の先頭（`start=0, end=100`)に付与します。\nこの差戻しは差戻しとして扱われず、抜取検査・抜取受入のスキップ判定に影響を及ぼしません。\n\nタスクの状態・フェーズを無視して、フェーズを教師付け(annotation)に、状態を未作業(not started)に変更します。\nタスクの担当者としては、直前の教師付け(annotation)フェーズの担当者を割り当てます。\nこの差戻しは差戻しとして扱われず、抜取検査・抜取受入のスキップ判定に影響を及ぼしません。\n\n```\n# tasks.txtに記載れたタスクを強制的に差し戻す\n# 最後のannotation phaseを担当したユーザを割り当てます（画面と同じ動き）。検査コメントは付与しません。\n$ annofabcli task reject --project_id prj1 --task_id file://tasks.txt \n\n# 受入完了の場合は、受入を取り消してから差し戻します\n$ annofabcli task reject --project_id prj1 --task_id file://tasks.txt --cancel_acceptance\n\n# 「hoge」という検査コメントを付与して、タスクを差し戻します。その際、担当者は割り当てません\n$ annofabcli task reject --project_id prj1 --task_id file://tasks.txt \\\n --comment "hoge" --not_assign\n\n# 差し戻したタスクに、ユーザuser1を割り当てる\n$ annofabcli task reject --project_id prj1 --task_id file://tasks.txt \\\n --comment "hoge" --assigned_annotator_user_id user1\n\n\n```\n\n\n\n',
    'author': 'yuji38kwmt',
    'author_email': 'yuji38kwmt@gmail.com',
    'maintainer': 'yuji38kwmt',
    'maintainer_email': 'yuji38kwmt@gmail.com',
    'url': 'https://github.com/kurusugawa-computer/annofab-cli',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6.1,<4.0.0',
}


setup(**setup_kwargs)
