"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TableSortStyle = exports.TableDistStyle = exports.Table = exports.TableAction = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const database_query_1 = require("./private/database-query");
const handler_name_1 = require("./private/database-query-provider/handler-name");
const util_1 = require("./private/database-query-provider/util");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
/**
 * An action that a Redshift user can be granted privilege to perform on a table.
 */
var TableAction;
(function (TableAction) {
    /**
     * Grants privilege to select data from a table or view using a SELECT statement.
     */
    TableAction[TableAction["SELECT"] = 0] = "SELECT";
    /**
     * Grants privilege to load data into a table using an INSERT statement or a COPY statement.
     */
    TableAction[TableAction["INSERT"] = 1] = "INSERT";
    /**
     * Grants privilege to update a table column using an UPDATE statement.
     */
    TableAction[TableAction["UPDATE"] = 2] = "UPDATE";
    /**
     * Grants privilege to delete a data row from a table.
     */
    TableAction[TableAction["DELETE"] = 3] = "DELETE";
    /**
     * Grants privilege to drop a table.
     */
    TableAction[TableAction["DROP"] = 4] = "DROP";
    /**
     * Grants privilege to create a foreign key constraint.
     *
     * You need to grant this privilege on both the referenced table and the referencing table; otherwise, the user can't create the constraint.
     */
    TableAction[TableAction["REFERENCES"] = 5] = "REFERENCES";
    /**
     * Grants all available privileges at once to the specified user or user group.
     */
    TableAction[TableAction["ALL"] = 6] = "ALL";
})(TableAction = exports.TableAction || (exports.TableAction = {}));
class TableBase extends core_1.Construct {
    grant(user, ...actions) {
        user.addTablePrivileges(this, ...actions);
    }
}
/**
 * A table in a Redshift cluster.
 */
class Table extends TableBase {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        jsiiDeprecationWarnings._aws_cdk_aws_redshift_TableProps(props);
        this.validateDistKeyColumns(props.tableColumns);
        if (props.distStyle) {
            this.validateDistStyle(props.distStyle, props.tableColumns);
        }
        if (props.sortStyle) {
            this.validateSortStyle(props.sortStyle, props.tableColumns);
        }
        this.tableColumns = props.tableColumns;
        this.cluster = props.cluster;
        this.databaseName = props.databaseName;
        this.resource = new database_query_1.DatabaseQuery(this, 'Resource', {
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            ...props,
            handler: handler_name_1.HandlerName.Table,
            properties: {
                tableName: {
                    prefix: (_b = props.tableName) !== null && _b !== void 0 ? _b : cdk.Names.uniqueId(this),
                    generateSuffix: !props.tableName ? 'true' : 'false',
                },
                tableColumns: this.tableColumns,
                distStyle: props.distStyle,
                sortStyle: (_c = props.sortStyle) !== null && _c !== void 0 ? _c : this.getDefaultSortStyle(props.tableColumns),
            },
        });
        this.tableName = this.resource.ref;
    }
    /**
     * Specify a Redshift table using a table name and schema that already exists.
     */
    static fromTableAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings._aws_cdk_aws_redshift_TableAttributes(attrs);
        return new class extends TableBase {
            constructor() {
                super(...arguments);
                this.tableName = attrs.tableName;
                this.tableColumns = attrs.tableColumns;
                this.cluster = attrs.cluster;
                this.databaseName = attrs.databaseName;
            }
        }(scope, id);
    }
    /**
     * Apply the given removal policy to this resource
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be destroyed (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * This resource is retained by default.
     */
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    validateDistKeyColumns(columns) {
        try {
            util_1.getDistKeyColumn(columns);
        }
        catch (err) {
            throw new Error('Only one column can be configured as distKey.');
        }
    }
    validateDistStyle(distStyle, columns) {
        const distKeyColumn = util_1.getDistKeyColumn(columns);
        if (distKeyColumn && distStyle !== TableDistStyle.KEY) {
            throw new Error(`Only 'TableDistStyle.KEY' can be configured when distKey is also configured. Found ${distStyle}`);
        }
        if (!distKeyColumn && distStyle === TableDistStyle.KEY) {
            throw new Error('distStyle of "TableDistStyle.KEY" can only be configured when distKey is also configured.');
        }
    }
    validateSortStyle(sortStyle, columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        if (sortKeyColumns.length === 0 && sortStyle !== TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${sortStyle}' can only be configured when sortKey is also configured.`);
        }
        if (sortKeyColumns.length > 0 && sortStyle === TableSortStyle.AUTO) {
            throw new Error(`sortStyle of '${TableSortStyle.AUTO}' cannot be configured when sortKey is also configured.`);
        }
    }
    getDefaultSortStyle(columns) {
        const sortKeyColumns = util_1.getSortKeyColumns(columns);
        return (sortKeyColumns.length === 0) ? TableSortStyle.AUTO : TableSortStyle.COMPOUND;
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-redshift.Table", version: "1.151.0" };
/**
 * The data distribution style of a table.
 */
var TableDistStyle;
(function (TableDistStyle) {
    /**
     *  Amazon Redshift assigns an optimal distribution style based on the table data
     */
    TableDistStyle["AUTO"] = "AUTO";
    /**
     * The data in the table is spread evenly across the nodes in a cluster in a round-robin distribution.
     */
    TableDistStyle["EVEN"] = "EVEN";
    /**
     * The data is distributed by the values in the DISTKEY column.
     */
    TableDistStyle["KEY"] = "KEY";
    /**
     * A copy of the entire table is distributed to every node.
     */
    TableDistStyle["ALL"] = "ALL";
})(TableDistStyle = exports.TableDistStyle || (exports.TableDistStyle = {}));
/**
 * The sort style of a table.
 */
var TableSortStyle;
(function (TableSortStyle) {
    /**
     * Amazon Redshift assigns an optimal sort key based on the table data.
     */
    TableSortStyle["AUTO"] = "AUTO";
    /**
     * Specifies that the data is sorted using a compound key made up of all of the listed columns,
     * in the order they are listed.
     */
    TableSortStyle["COMPOUND"] = "COMPOUND";
    /**
     * Specifies that the data is sorted using an interleaved sort key.
     */
    TableSortStyle["INTERLEAVED"] = "INTERLEAVED";
})(TableSortStyle = exports.TableSortStyle || (exports.TableSortStyle = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFibGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0YWJsZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxxQ0FBcUM7QUFJckMsNkRBQXlEO0FBQ3pELGlGQUE2RTtBQUM3RSxpRUFBNkY7QUFJN0YsaUdBQWlHO0FBQ2pHLDhEQUE4RDtBQUM5RCx3Q0FBMkQ7QUFFM0Q7O0dBRUc7QUFDSCxJQUFZLFdBcUNYO0FBckNELFdBQVksV0FBVztJQUNyQjs7T0FFRztJQUNILGlEQUFNLENBQUE7SUFFTjs7T0FFRztJQUNILGlEQUFNLENBQUE7SUFFTjs7T0FFRztJQUNILGlEQUFNLENBQUE7SUFFTjs7T0FFRztJQUNILGlEQUFNLENBQUE7SUFFTjs7T0FFRztJQUNILDZDQUFJLENBQUE7SUFFSjs7OztPQUlHO0lBQ0gseURBQVUsQ0FBQTtJQUVWOztPQUVHO0lBQ0gsMkNBQUcsQ0FBQTtBQUNMLENBQUMsRUFyQ1csV0FBVyxHQUFYLG1CQUFXLEtBQVgsbUJBQVcsUUFxQ3RCO0FBNEhELE1BQWUsU0FBVSxTQUFRLGdCQUFhO0lBSzVDLEtBQUssQ0FBQyxJQUFXLEVBQUUsR0FBRyxPQUFzQjtRQUMxQyxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxFQUFFLEdBQUcsT0FBTyxDQUFDLENBQUM7S0FDM0M7Q0FDRjtBQUVEOztHQUVHO0FBQ0gsTUFBYSxLQUFNLFNBQVEsU0FBUztJQW9CbEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFpQjs7UUFDekQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQzs7UUFFakIsSUFBSSxDQUFDLHNCQUFzQixDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUNoRCxJQUFJLEtBQUssQ0FBQyxTQUFTLEVBQUU7WUFDbkIsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDO1NBQzdEO1FBQ0QsSUFBSSxLQUFLLENBQUMsU0FBUyxFQUFFO1lBQ25CLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQztTQUM3RDtRQUVELElBQUksQ0FBQyxZQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztRQUN2QyxJQUFJLENBQUMsT0FBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7UUFDN0IsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO1FBRXZDLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSw4QkFBYSxDQUFvQixJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3JFLGFBQWEsRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLE1BQU07WUFDdkMsR0FBRyxLQUFLO1lBQ1IsT0FBTyxFQUFFLDBCQUFXLENBQUMsS0FBSztZQUMxQixVQUFVLEVBQUU7Z0JBQ1YsU0FBUyxFQUFFO29CQUNULE1BQU0sUUFBRSxLQUFLLENBQUMsU0FBUyxtQ0FBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUM7b0JBQ25ELGNBQWMsRUFBRSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsT0FBTztpQkFDcEQ7Z0JBQ0QsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO2dCQUMvQixTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7Z0JBQzFCLFNBQVMsUUFBRSxLQUFLLENBQUMsU0FBUyxtQ0FBSSxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQzthQUMzRTtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxTQUFTLEdBQUcsSUFBSSxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUM7S0FDcEM7SUFsREQ7O09BRUc7SUFDSCxNQUFNLENBQUMsbUJBQW1CLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBc0I7O1FBQzdFLE9BQU8sSUFBSSxLQUFNLFNBQVEsU0FBUztZQUF2Qjs7Z0JBQ0EsY0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7Z0JBQzVCLGlCQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztnQkFDbEMsWUFBTyxHQUFHLEtBQUssQ0FBQyxPQUFPLENBQUM7Z0JBQ3hCLGlCQUFZLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztZQUM3QyxDQUFDO1NBQUEsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7S0FDZDtJQTBDRDs7Ozs7Ozs7Ozs7O09BWUc7SUFDSSxrQkFBa0IsQ0FBQyxNQUF5QjtRQUNqRCxJQUFJLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sQ0FBQyxDQUFDO0tBQzFDO0lBRU8sc0JBQXNCLENBQUMsT0FBaUI7UUFDOUMsSUFBSTtZQUNGLHVCQUFnQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1NBQzNCO1FBQUMsT0FBTyxHQUFHLEVBQUU7WUFDWixNQUFNLElBQUksS0FBSyxDQUFDLCtDQUErQyxDQUFDLENBQUM7U0FDbEU7S0FDRjtJQUVPLGlCQUFpQixDQUFDLFNBQXlCLEVBQUUsT0FBaUI7UUFDcEUsTUFBTSxhQUFhLEdBQUcsdUJBQWdCLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDaEQsSUFBSSxhQUFhLElBQUksU0FBUyxLQUFLLGNBQWMsQ0FBQyxHQUFHLEVBQUU7WUFDckQsTUFBTSxJQUFJLEtBQUssQ0FBQyxzRkFBc0YsU0FBUyxFQUFFLENBQUMsQ0FBQztTQUNwSDtRQUNELElBQUksQ0FBQyxhQUFhLElBQUksU0FBUyxLQUFLLGNBQWMsQ0FBQyxHQUFHLEVBQUU7WUFDdEQsTUFBTSxJQUFJLEtBQUssQ0FBQywyRkFBMkYsQ0FBQyxDQUFDO1NBQzlHO0tBQ0Y7SUFFTyxpQkFBaUIsQ0FBQyxTQUF5QixFQUFFLE9BQWlCO1FBQ3BFLE1BQU0sY0FBYyxHQUFHLHdCQUFpQixDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQ2xELElBQUksY0FBYyxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksU0FBUyxLQUFLLGNBQWMsQ0FBQyxJQUFJLEVBQUU7WUFDcEUsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQkFBaUIsU0FBUywyREFBMkQsQ0FBQyxDQUFDO1NBQ3hHO1FBQ0QsSUFBSSxjQUFjLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxTQUFTLEtBQUssY0FBYyxDQUFDLElBQUksRUFBRTtZQUNsRSxNQUFNLElBQUksS0FBSyxDQUFDLGlCQUFpQixjQUFjLENBQUMsSUFBSSx5REFBeUQsQ0FBQyxDQUFDO1NBQ2hIO0tBQ0Y7SUFFTyxtQkFBbUIsQ0FBQyxPQUFpQjtRQUMzQyxNQUFNLGNBQWMsR0FBRyx3QkFBaUIsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNsRCxPQUFPLENBQUMsY0FBYyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLFFBQVEsQ0FBQztLQUN0Rjs7QUFyR0gsc0JBc0dDOzs7QUFFRDs7R0FFRztBQUNILElBQVksY0FvQlg7QUFwQkQsV0FBWSxjQUFjO0lBQ3hCOztPQUVHO0lBQ0gsK0JBQWEsQ0FBQTtJQUViOztPQUVHO0lBQ0gsK0JBQWEsQ0FBQTtJQUViOztPQUVHO0lBQ0gsNkJBQVcsQ0FBQTtJQUVYOztPQUVHO0lBQ0gsNkJBQVcsQ0FBQTtBQUNiLENBQUMsRUFwQlcsY0FBYyxHQUFkLHNCQUFjLEtBQWQsc0JBQWMsUUFvQnpCO0FBRUQ7O0dBRUc7QUFDSCxJQUFZLGNBZ0JYO0FBaEJELFdBQVksY0FBYztJQUN4Qjs7T0FFRztJQUNILCtCQUFhLENBQUE7SUFFYjs7O09BR0c7SUFDSCx1Q0FBcUIsQ0FBQTtJQUVyQjs7T0FFRztJQUNILDZDQUEyQixDQUFBO0FBQzdCLENBQUMsRUFoQlcsY0FBYyxHQUFkLHNCQUFjLEtBQWQsc0JBQWMsUUFnQnpCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBJQ2x1c3RlciB9IGZyb20gJy4vY2x1c3Rlcic7XG5pbXBvcnQgeyBEYXRhYmFzZU9wdGlvbnMgfSBmcm9tICcuL2RhdGFiYXNlLW9wdGlvbnMnO1xuaW1wb3J0IHsgRGF0YWJhc2VRdWVyeSB9IGZyb20gJy4vcHJpdmF0ZS9kYXRhYmFzZS1xdWVyeSc7XG5pbXBvcnQgeyBIYW5kbGVyTmFtZSB9IGZyb20gJy4vcHJpdmF0ZS9kYXRhYmFzZS1xdWVyeS1wcm92aWRlci9oYW5kbGVyLW5hbWUnO1xuaW1wb3J0IHsgZ2V0RGlzdEtleUNvbHVtbiwgZ2V0U29ydEtleUNvbHVtbnMgfSBmcm9tICcuL3ByaXZhdGUvZGF0YWJhc2UtcXVlcnktcHJvdmlkZXIvdXRpbCc7XG5pbXBvcnQgeyBUYWJsZUhhbmRsZXJQcm9wcyB9IGZyb20gJy4vcHJpdmF0ZS9oYW5kbGVyLXByb3BzJztcbmltcG9ydCB7IElVc2VyIH0gZnJvbSAnLi91c2VyJztcblxuLy8ga2VlcCB0aGlzIGltcG9ydCBzZXBhcmF0ZSBmcm9tIG90aGVyIGltcG9ydHMgdG8gcmVkdWNlIGNoYW5jZSBmb3IgbWVyZ2UgY29uZmxpY3RzIHdpdGggdjItbWFpblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWR1cGxpY2F0ZS1pbXBvcnRzLCBpbXBvcnQvb3JkZXJcbmltcG9ydCB7IENvbnN0cnVjdCBhcyBDb3JlQ29uc3RydWN0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbi8qKlxuICogQW4gYWN0aW9uIHRoYXQgYSBSZWRzaGlmdCB1c2VyIGNhbiBiZSBncmFudGVkIHByaXZpbGVnZSB0byBwZXJmb3JtIG9uIGEgdGFibGUuXG4gKi9cbmV4cG9ydCBlbnVtIFRhYmxlQWN0aW9uIHtcbiAgLyoqXG4gICAqIEdyYW50cyBwcml2aWxlZ2UgdG8gc2VsZWN0IGRhdGEgZnJvbSBhIHRhYmxlIG9yIHZpZXcgdXNpbmcgYSBTRUxFQ1Qgc3RhdGVtZW50LlxuICAgKi9cbiAgU0VMRUNULFxuXG4gIC8qKlxuICAgKiBHcmFudHMgcHJpdmlsZWdlIHRvIGxvYWQgZGF0YSBpbnRvIGEgdGFibGUgdXNpbmcgYW4gSU5TRVJUIHN0YXRlbWVudCBvciBhIENPUFkgc3RhdGVtZW50LlxuICAgKi9cbiAgSU5TRVJULFxuXG4gIC8qKlxuICAgKiBHcmFudHMgcHJpdmlsZWdlIHRvIHVwZGF0ZSBhIHRhYmxlIGNvbHVtbiB1c2luZyBhbiBVUERBVEUgc3RhdGVtZW50LlxuICAgKi9cbiAgVVBEQVRFLFxuXG4gIC8qKlxuICAgKiBHcmFudHMgcHJpdmlsZWdlIHRvIGRlbGV0ZSBhIGRhdGEgcm93IGZyb20gYSB0YWJsZS5cbiAgICovXG4gIERFTEVURSxcblxuICAvKipcbiAgICogR3JhbnRzIHByaXZpbGVnZSB0byBkcm9wIGEgdGFibGUuXG4gICAqL1xuICBEUk9QLFxuXG4gIC8qKlxuICAgKiBHcmFudHMgcHJpdmlsZWdlIHRvIGNyZWF0ZSBhIGZvcmVpZ24ga2V5IGNvbnN0cmFpbnQuXG4gICAqXG4gICAqIFlvdSBuZWVkIHRvIGdyYW50IHRoaXMgcHJpdmlsZWdlIG9uIGJvdGggdGhlIHJlZmVyZW5jZWQgdGFibGUgYW5kIHRoZSByZWZlcmVuY2luZyB0YWJsZTsgb3RoZXJ3aXNlLCB0aGUgdXNlciBjYW4ndCBjcmVhdGUgdGhlIGNvbnN0cmFpbnQuXG4gICAqL1xuICBSRUZFUkVOQ0VTLFxuXG4gIC8qKlxuICAgKiBHcmFudHMgYWxsIGF2YWlsYWJsZSBwcml2aWxlZ2VzIGF0IG9uY2UgdG8gdGhlIHNwZWNpZmllZCB1c2VyIG9yIHVzZXIgZ3JvdXAuXG4gICAqL1xuICBBTExcbn1cblxuLyoqXG4gKiBBIGNvbHVtbiBpbiBhIFJlZHNoaWZ0IHRhYmxlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvbHVtbiB7XG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgY29sdW1uLlxuICAgKi9cbiAgcmVhZG9ubHkgbmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgZGF0YSB0eXBlIG9mIHRoZSBjb2x1bW4uXG4gICAqL1xuICByZWFkb25seSBkYXRhVHlwZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBCb29sZWFuIHZhbHVlIHRoYXQgaW5kaWNhdGVzIHdoZXRoZXIgdGhlIGNvbHVtbiBpcyB0byBiZSBjb25maWd1cmVkIGFzIERJU1RLRVkuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gY29sdW1uIGlzIG5vdCBESVNUS0VZXG4gICAqL1xuICByZWFkb25seSBkaXN0S2V5PzogYm9vbGVhbjtcblxuICAvKipcbiAgICogQm9vbGVhbiB2YWx1ZSB0aGF0IGluZGljYXRlcyB3aGV0aGVyIHRoZSBjb2x1bW4gaXMgdG8gYmUgY29uZmlndXJlZCBhcyBTT1JUS0VZLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGNvbHVtbiBpcyBub3QgYSBTT1JUS0VZXG4gICAqL1xuICByZWFkb25seSBzb3J0S2V5PzogYm9vbGVhbjtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBjb25maWd1cmluZyBhIFJlZHNoaWZ0IHRhYmxlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRhYmxlUHJvcHMgZXh0ZW5kcyBEYXRhYmFzZU9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIHRhYmxlLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGEgbmFtZSBpcyBnZW5lcmF0ZWRcbiAgICovXG4gIHJlYWRvbmx5IHRhYmxlTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGNvbHVtbnMgb2YgdGhlIHRhYmxlLlxuICAgKi9cbiAgcmVhZG9ubHkgdGFibGVDb2x1bW5zOiBDb2x1bW5bXTtcblxuICAvKipcbiAgICogVGhlIGRpc3RyaWJ1dGlvbiBzdHlsZSBvZiB0aGUgdGFibGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IFRhYmxlRGlzdFN0eWxlLkFVVE9cbiAgICovXG4gIHJlYWRvbmx5IGRpc3RTdHlsZT86IFRhYmxlRGlzdFN0eWxlO1xuXG4gIC8qKlxuICAgKiBUaGUgc29ydCBzdHlsZSBvZiB0aGUgdGFibGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IFRhYmxlU29ydFN0eWxlLkFVVE8gaWYgbm8gc29ydCBrZXkgaXMgc3BlY2lmaWVkLCBUYWJsZVNvcnRTdHlsZS5DT01QT1VORCBpZiBhIHNvcnQga2V5IGlzIHNwZWNpZmllZFxuICAgKi9cbiAgcmVhZG9ubHkgc29ydFN0eWxlPzogVGFibGVTb3J0U3R5bGU7XG5cbiAgLyoqXG4gICAqIFRoZSBwb2xpY3kgdG8gYXBwbHkgd2hlbiB0aGlzIHJlc291cmNlIGlzIHJlbW92ZWQgZnJvbSB0aGUgYXBwbGljYXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IGNkay5SZW1vdmFsUG9saWN5LlJldGFpblxuICAgKi9cbiAgcmVhZG9ubHkgcmVtb3ZhbFBvbGljeT86IGNkay5SZW1vdmFsUG9saWN5O1xufVxuXG4vKipcbiAqIFJlcHJlc2VudHMgYSB0YWJsZSBpbiBhIFJlZHNoaWZ0IGRhdGFiYXNlLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElUYWJsZSBleHRlbmRzIGNkay5JQ29uc3RydWN0IHtcbiAgLyoqXG4gICAqIE5hbWUgb2YgdGhlIHRhYmxlLlxuICAgKi9cbiAgcmVhZG9ubHkgdGFibGVOYW1lOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBjb2x1bW5zIG9mIHRoZSB0YWJsZS5cbiAgICovXG4gIHJlYWRvbmx5IHRhYmxlQ29sdW1uczogQ29sdW1uW107XG5cbiAgLyoqXG4gICAqIFRoZSBjbHVzdGVyIHdoZXJlIHRoZSB0YWJsZSBpcyBsb2NhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgY2x1c3RlcjogSUNsdXN0ZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBkYXRhYmFzZSB3aGVyZSB0aGUgdGFibGUgaXMgbG9jYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGRhdGFiYXNlTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBHcmFudCBhIHVzZXIgcHJpdmlsZWdlIHRvIGFjY2VzcyB0aGlzIHRhYmxlLlxuICAgKi9cbiAgZ3JhbnQodXNlcjogSVVzZXIsIC4uLmFjdGlvbnM6IFRhYmxlQWN0aW9uW10pOiB2b2lkO1xufVxuXG4vKipcbiAqIEEgZnVsbCBzcGVjaWZpY2F0aW9uIG9mIGEgUmVkc2hpZnQgdGFibGUgdGhhdCBjYW4gYmUgdXNlZCB0byBpbXBvcnQgaXQgZmx1ZW50bHkgaW50byB0aGUgQ0RLIGFwcGxpY2F0aW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRhYmxlQXR0cmlidXRlcyB7XG4gIC8qKlxuICAgKiBOYW1lIG9mIHRoZSB0YWJsZS5cbiAgICovXG4gIHJlYWRvbmx5IHRhYmxlTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgY29sdW1ucyBvZiB0aGUgdGFibGUuXG4gICAqL1xuICByZWFkb25seSB0YWJsZUNvbHVtbnM6IENvbHVtbltdO1xuXG4gIC8qKlxuICAgKiBUaGUgY2x1c3RlciB3aGVyZSB0aGUgdGFibGUgaXMgbG9jYXRlZC5cbiAgICovXG4gIHJlYWRvbmx5IGNsdXN0ZXI6IElDbHVzdGVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgZGF0YWJhc2Ugd2hlcmUgdGhlIHRhYmxlIGlzIGxvY2F0ZWQuXG4gICAqL1xuICByZWFkb25seSBkYXRhYmFzZU5hbWU6IHN0cmluZztcbn1cblxuYWJzdHJhY3QgY2xhc3MgVGFibGVCYXNlIGV4dGVuZHMgQ29yZUNvbnN0cnVjdCBpbXBsZW1lbnRzIElUYWJsZSB7XG4gIGFic3RyYWN0IHJlYWRvbmx5IHRhYmxlTmFtZTogc3RyaW5nO1xuICBhYnN0cmFjdCByZWFkb25seSB0YWJsZUNvbHVtbnM6IENvbHVtbltdO1xuICBhYnN0cmFjdCByZWFkb25seSBjbHVzdGVyOiBJQ2x1c3RlcjtcbiAgYWJzdHJhY3QgcmVhZG9ubHkgZGF0YWJhc2VOYW1lOiBzdHJpbmc7XG4gIGdyYW50KHVzZXI6IElVc2VyLCAuLi5hY3Rpb25zOiBUYWJsZUFjdGlvbltdKSB7XG4gICAgdXNlci5hZGRUYWJsZVByaXZpbGVnZXModGhpcywgLi4uYWN0aW9ucyk7XG4gIH1cbn1cblxuLyoqXG4gKiBBIHRhYmxlIGluIGEgUmVkc2hpZnQgY2x1c3Rlci5cbiAqL1xuZXhwb3J0IGNsYXNzIFRhYmxlIGV4dGVuZHMgVGFibGVCYXNlIHtcbiAgLyoqXG4gICAqIFNwZWNpZnkgYSBSZWRzaGlmdCB0YWJsZSB1c2luZyBhIHRhYmxlIG5hbWUgYW5kIHNjaGVtYSB0aGF0IGFscmVhZHkgZXhpc3RzLlxuICAgKi9cbiAgc3RhdGljIGZyb21UYWJsZUF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IFRhYmxlQXR0cmlidXRlcyk6IElUYWJsZSB7XG4gICAgcmV0dXJuIG5ldyBjbGFzcyBleHRlbmRzIFRhYmxlQmFzZSB7XG4gICAgICByZWFkb25seSB0YWJsZU5hbWUgPSBhdHRycy50YWJsZU5hbWU7XG4gICAgICByZWFkb25seSB0YWJsZUNvbHVtbnMgPSBhdHRycy50YWJsZUNvbHVtbnM7XG4gICAgICByZWFkb25seSBjbHVzdGVyID0gYXR0cnMuY2x1c3RlcjtcbiAgICAgIHJlYWRvbmx5IGRhdGFiYXNlTmFtZSA9IGF0dHJzLmRhdGFiYXNlTmFtZTtcbiAgICB9KHNjb3BlLCBpZCk7XG4gIH1cblxuICByZWFkb25seSB0YWJsZU5hbWU6IHN0cmluZztcbiAgcmVhZG9ubHkgdGFibGVDb2x1bW5zOiBDb2x1bW5bXTtcbiAgcmVhZG9ubHkgY2x1c3RlcjogSUNsdXN0ZXI7XG4gIHJlYWRvbmx5IGRhdGFiYXNlTmFtZTogc3RyaW5nO1xuXG4gIHByaXZhdGUgcmVzb3VyY2U6IERhdGFiYXNlUXVlcnk8VGFibGVIYW5kbGVyUHJvcHM+O1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBUYWJsZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMudmFsaWRhdGVEaXN0S2V5Q29sdW1ucyhwcm9wcy50YWJsZUNvbHVtbnMpO1xuICAgIGlmIChwcm9wcy5kaXN0U3R5bGUpIHtcbiAgICAgIHRoaXMudmFsaWRhdGVEaXN0U3R5bGUocHJvcHMuZGlzdFN0eWxlLCBwcm9wcy50YWJsZUNvbHVtbnMpO1xuICAgIH1cbiAgICBpZiAocHJvcHMuc29ydFN0eWxlKSB7XG4gICAgICB0aGlzLnZhbGlkYXRlU29ydFN0eWxlKHByb3BzLnNvcnRTdHlsZSwgcHJvcHMudGFibGVDb2x1bW5zKTtcbiAgICB9XG5cbiAgICB0aGlzLnRhYmxlQ29sdW1ucyA9IHByb3BzLnRhYmxlQ29sdW1ucztcbiAgICB0aGlzLmNsdXN0ZXIgPSBwcm9wcy5jbHVzdGVyO1xuICAgIHRoaXMuZGF0YWJhc2VOYW1lID0gcHJvcHMuZGF0YWJhc2VOYW1lO1xuXG4gICAgdGhpcy5yZXNvdXJjZSA9IG5ldyBEYXRhYmFzZVF1ZXJ5PFRhYmxlSGFuZGxlclByb3BzPih0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICByZW1vdmFsUG9saWN5OiBjZGsuUmVtb3ZhbFBvbGljeS5SRVRBSU4sXG4gICAgICAuLi5wcm9wcyxcbiAgICAgIGhhbmRsZXI6IEhhbmRsZXJOYW1lLlRhYmxlLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICB0YWJsZU5hbWU6IHtcbiAgICAgICAgICBwcmVmaXg6IHByb3BzLnRhYmxlTmFtZSA/PyBjZGsuTmFtZXMudW5pcXVlSWQodGhpcyksXG4gICAgICAgICAgZ2VuZXJhdGVTdWZmaXg6ICFwcm9wcy50YWJsZU5hbWUgPyAndHJ1ZScgOiAnZmFsc2UnLFxuICAgICAgICB9LFxuICAgICAgICB0YWJsZUNvbHVtbnM6IHRoaXMudGFibGVDb2x1bW5zLFxuICAgICAgICBkaXN0U3R5bGU6IHByb3BzLmRpc3RTdHlsZSxcbiAgICAgICAgc29ydFN0eWxlOiBwcm9wcy5zb3J0U3R5bGUgPz8gdGhpcy5nZXREZWZhdWx0U29ydFN0eWxlKHByb3BzLnRhYmxlQ29sdW1ucyksXG4gICAgICB9LFxuICAgIH0pO1xuXG4gICAgdGhpcy50YWJsZU5hbWUgPSB0aGlzLnJlc291cmNlLnJlZjtcbiAgfVxuXG4gIC8qKlxuICAgKiBBcHBseSB0aGUgZ2l2ZW4gcmVtb3ZhbCBwb2xpY3kgdG8gdGhpcyByZXNvdXJjZVxuICAgKlxuICAgKiBUaGUgUmVtb3ZhbCBQb2xpY3kgY29udHJvbHMgd2hhdCBoYXBwZW5zIHRvIHRoaXMgcmVzb3VyY2Ugd2hlbiBpdCBzdG9wc1xuICAgKiBiZWluZyBtYW5hZ2VkIGJ5IENsb3VkRm9ybWF0aW9uLCBlaXRoZXIgYmVjYXVzZSB5b3UndmUgcmVtb3ZlZCBpdCBmcm9tIHRoZVxuICAgKiBDREsgYXBwbGljYXRpb24gb3IgYmVjYXVzZSB5b3UndmUgbWFkZSBhIGNoYW5nZSB0aGF0IHJlcXVpcmVzIHRoZSByZXNvdXJjZVxuICAgKiB0byBiZSByZXBsYWNlZC5cbiAgICpcbiAgICogVGhlIHJlc291cmNlIGNhbiBiZSBkZXN0cm95ZWQgKGBSZW1vdmFsUG9saWN5LkRFU1RST1lgKSwgb3IgbGVmdCBpbiB5b3VyIEFXU1xuICAgKiBhY2NvdW50IGZvciBkYXRhIHJlY292ZXJ5IGFuZCBjbGVhbnVwIGxhdGVyIChgUmVtb3ZhbFBvbGljeS5SRVRBSU5gKS5cbiAgICpcbiAgICogVGhpcyByZXNvdXJjZSBpcyByZXRhaW5lZCBieSBkZWZhdWx0LlxuICAgKi9cbiAgcHVibGljIGFwcGx5UmVtb3ZhbFBvbGljeShwb2xpY3k6IGNkay5SZW1vdmFsUG9saWN5KTogdm9pZCB7XG4gICAgdGhpcy5yZXNvdXJjZS5hcHBseVJlbW92YWxQb2xpY3kocG9saWN5KTtcbiAgfVxuXG4gIHByaXZhdGUgdmFsaWRhdGVEaXN0S2V5Q29sdW1ucyhjb2x1bW5zOiBDb2x1bW5bXSk6IHZvaWQge1xuICAgIHRyeSB7XG4gICAgICBnZXREaXN0S2V5Q29sdW1uKGNvbHVtbnMpO1xuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdPbmx5IG9uZSBjb2x1bW4gY2FuIGJlIGNvbmZpZ3VyZWQgYXMgZGlzdEtleS4nKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIHZhbGlkYXRlRGlzdFN0eWxlKGRpc3RTdHlsZTogVGFibGVEaXN0U3R5bGUsIGNvbHVtbnM6IENvbHVtbltdKTogdm9pZCB7XG4gICAgY29uc3QgZGlzdEtleUNvbHVtbiA9IGdldERpc3RLZXlDb2x1bW4oY29sdW1ucyk7XG4gICAgaWYgKGRpc3RLZXlDb2x1bW4gJiYgZGlzdFN0eWxlICE9PSBUYWJsZURpc3RTdHlsZS5LRVkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgT25seSAnVGFibGVEaXN0U3R5bGUuS0VZJyBjYW4gYmUgY29uZmlndXJlZCB3aGVuIGRpc3RLZXkgaXMgYWxzbyBjb25maWd1cmVkLiBGb3VuZCAke2Rpc3RTdHlsZX1gKTtcbiAgICB9XG4gICAgaWYgKCFkaXN0S2V5Q29sdW1uICYmIGRpc3RTdHlsZSA9PT0gVGFibGVEaXN0U3R5bGUuS0VZKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ2Rpc3RTdHlsZSBvZiBcIlRhYmxlRGlzdFN0eWxlLktFWVwiIGNhbiBvbmx5IGJlIGNvbmZpZ3VyZWQgd2hlbiBkaXN0S2V5IGlzIGFsc28gY29uZmlndXJlZC4nKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIHZhbGlkYXRlU29ydFN0eWxlKHNvcnRTdHlsZTogVGFibGVTb3J0U3R5bGUsIGNvbHVtbnM6IENvbHVtbltdKTogdm9pZCB7XG4gICAgY29uc3Qgc29ydEtleUNvbHVtbnMgPSBnZXRTb3J0S2V5Q29sdW1ucyhjb2x1bW5zKTtcbiAgICBpZiAoc29ydEtleUNvbHVtbnMubGVuZ3RoID09PSAwICYmIHNvcnRTdHlsZSAhPT0gVGFibGVTb3J0U3R5bGUuQVVUTykge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBzb3J0U3R5bGUgb2YgJyR7c29ydFN0eWxlfScgY2FuIG9ubHkgYmUgY29uZmlndXJlZCB3aGVuIHNvcnRLZXkgaXMgYWxzbyBjb25maWd1cmVkLmApO1xuICAgIH1cbiAgICBpZiAoc29ydEtleUNvbHVtbnMubGVuZ3RoID4gMCAmJiBzb3J0U3R5bGUgPT09IFRhYmxlU29ydFN0eWxlLkFVVE8pIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgc29ydFN0eWxlIG9mICcke1RhYmxlU29ydFN0eWxlLkFVVE99JyBjYW5ub3QgYmUgY29uZmlndXJlZCB3aGVuIHNvcnRLZXkgaXMgYWxzbyBjb25maWd1cmVkLmApO1xuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgZ2V0RGVmYXVsdFNvcnRTdHlsZShjb2x1bW5zOiBDb2x1bW5bXSk6IFRhYmxlU29ydFN0eWxlIHtcbiAgICBjb25zdCBzb3J0S2V5Q29sdW1ucyA9IGdldFNvcnRLZXlDb2x1bW5zKGNvbHVtbnMpO1xuICAgIHJldHVybiAoc29ydEtleUNvbHVtbnMubGVuZ3RoID09PSAwKSA/IFRhYmxlU29ydFN0eWxlLkFVVE8gOiBUYWJsZVNvcnRTdHlsZS5DT01QT1VORDtcbiAgfVxufVxuXG4vKipcbiAqIFRoZSBkYXRhIGRpc3RyaWJ1dGlvbiBzdHlsZSBvZiBhIHRhYmxlLlxuICovXG5leHBvcnQgZW51bSBUYWJsZURpc3RTdHlsZSB7XG4gIC8qKlxuICAgKiAgQW1hem9uIFJlZHNoaWZ0IGFzc2lnbnMgYW4gb3B0aW1hbCBkaXN0cmlidXRpb24gc3R5bGUgYmFzZWQgb24gdGhlIHRhYmxlIGRhdGFcbiAgICovXG4gIEFVVE8gPSAnQVVUTycsXG5cbiAgLyoqXG4gICAqIFRoZSBkYXRhIGluIHRoZSB0YWJsZSBpcyBzcHJlYWQgZXZlbmx5IGFjcm9zcyB0aGUgbm9kZXMgaW4gYSBjbHVzdGVyIGluIGEgcm91bmQtcm9iaW4gZGlzdHJpYnV0aW9uLlxuICAgKi9cbiAgRVZFTiA9ICdFVkVOJyxcblxuICAvKipcbiAgICogVGhlIGRhdGEgaXMgZGlzdHJpYnV0ZWQgYnkgdGhlIHZhbHVlcyBpbiB0aGUgRElTVEtFWSBjb2x1bW4uXG4gICAqL1xuICBLRVkgPSAnS0VZJyxcblxuICAvKipcbiAgICogQSBjb3B5IG9mIHRoZSBlbnRpcmUgdGFibGUgaXMgZGlzdHJpYnV0ZWQgdG8gZXZlcnkgbm9kZS5cbiAgICovXG4gIEFMTCA9ICdBTEwnLFxufVxuXG4vKipcbiAqIFRoZSBzb3J0IHN0eWxlIG9mIGEgdGFibGUuXG4gKi9cbmV4cG9ydCBlbnVtIFRhYmxlU29ydFN0eWxlIHtcbiAgLyoqXG4gICAqIEFtYXpvbiBSZWRzaGlmdCBhc3NpZ25zIGFuIG9wdGltYWwgc29ydCBrZXkgYmFzZWQgb24gdGhlIHRhYmxlIGRhdGEuXG4gICAqL1xuICBBVVRPID0gJ0FVVE8nLFxuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhhdCB0aGUgZGF0YSBpcyBzb3J0ZWQgdXNpbmcgYSBjb21wb3VuZCBrZXkgbWFkZSB1cCBvZiBhbGwgb2YgdGhlIGxpc3RlZCBjb2x1bW5zLFxuICAgKiBpbiB0aGUgb3JkZXIgdGhleSBhcmUgbGlzdGVkLlxuICAgKi9cbiAgQ09NUE9VTkQgPSAnQ09NUE9VTkQnLFxuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhhdCB0aGUgZGF0YSBpcyBzb3J0ZWQgdXNpbmcgYW4gaW50ZXJsZWF2ZWQgc29ydCBrZXkuXG4gICAqL1xuICBJTlRFUkxFQVZFRCA9ICdJTlRFUkxFQVZFRCcsXG59XG4iXX0=