"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
/**
 * Use a Lambda function as a subscription target
 */
class LambdaSubscription {
    constructor(fn, props = {}) {
        this.fn = fn;
        this.props = props;
    }
    /**
     * Returns a configuration for a Lambda function to subscribe to an SNS topic
     */
    bind(topic) {
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        if (!core_1.Construct.isConstruct(this.fn)) {
            throw new Error('The supplied lambda Function object must be an instance of Construct');
        }
        this.fn.addPermission(`AllowInvoke:${topic.node.uniqueId}`, {
            sourceArn: topic.topicArn,
            principal: new iam.ServicePrincipal('sns.amazonaws.com'),
        });
        return {
            subscriberScope: this.fn,
            subscriberId: topic.node.id,
            endpoint: this.fn.functionArn,
            protocol: sns.SubscriptionProtocol.LAMBDA,
            filterPolicy: this.props.filterPolicy,
            region: this.regionFromArn(topic),
        };
    }
    regionFromArn(topic) {
        // no need to specify `region` for topics defined within the same stack.
        if (topic instanceof sns.Topic) {
            return undefined;
        }
        return core_1.Stack.of(topic).parseArn(topic.topicArn).region;
    }
}
exports.LambdaSubscription = LambdaSubscription;
//# sourceMappingURL=data:application/json;base64,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