"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomLambdaDeploymentConfig = exports.CustomLambdaDeploymentConfigType = void 0;
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
const utils_1 = require("../utils");
/**
 * Lambda Deployment config type.
 */
var CustomLambdaDeploymentConfigType;
(function (CustomLambdaDeploymentConfigType) {
    CustomLambdaDeploymentConfigType["CANARY"] = "Canary";
    CustomLambdaDeploymentConfigType["LINEAR"] = "Linear";
})(CustomLambdaDeploymentConfigType = exports.CustomLambdaDeploymentConfigType || (exports.CustomLambdaDeploymentConfigType = {}));
/**
 * A custom Deployment Configuration for a Lambda Deployment Group.
 *
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class CustomLambdaDeploymentConfig extends core_1.Resource {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.validateParameters(props);
        // In this section we make the argument for the AWS API call
        const deploymentType = 'TimeBased' + props.type.toString();
        const intervalMinutes = props.interval.toMinutes().toString();
        const percentage = props.percentage.toString();
        let routingConfig; // The argument to the AWS API call
        if (props.type == CustomLambdaDeploymentConfigType.CANARY) {
            routingConfig = {
                type: deploymentType,
                timeBasedCanary: {
                    canaryInterval: intervalMinutes,
                    canaryPercentage: percentage,
                },
            };
        }
        else if (props.type == CustomLambdaDeploymentConfigType.LINEAR) {
            routingConfig = {
                type: deploymentType,
                timeBasedLinear: {
                    linearInterval: intervalMinutes,
                    linearPercentage: percentage,
                },
            };
        }
        // Generates the name of the deployment config. It's also what you'll see in the AWS console
        // The name of the config is <construct unique id>.Lambda<deployment type><percentage>Percent<interval>Minutes
        // Unless the user provides an explicit name
        this.deploymentConfigName = props.deploymentConfigName !== undefined
            ? props.deploymentConfigName
            : `${this.node.uniqueId}.Lambda${props.type}${props.percentage}Percent${props.type === CustomLambdaDeploymentConfigType.LINEAR
                ? 'Every'
                : ''}${props.interval.toMinutes()}Minutes`;
        this.deploymentConfigArn = utils_1.arnForDeploymentConfig(this.deploymentConfigName);
        // The AWS Custom Resource that calls CodeDeploy to create and delete the resource
        new custom_resources_1.AwsCustomResource(this, 'DeploymentConfig', {
            onCreate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // This `resourceName` is the initial physical ID of the config
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onUpdate: {
                service: 'CodeDeploy',
                action: 'createDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                    computePlatform: 'Lambda',
                    trafficRoutingConfig: routingConfig,
                },
                // If `resourceName` is different from the last stack update (or creation),
                // the old config gets deleted and the new one is created
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(this.deploymentConfigName),
            },
            onDelete: {
                service: 'CodeDeploy',
                action: 'deleteDeploymentConfig',
                parameters: {
                    deploymentConfigName: this.deploymentConfigName,
                },
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
    }
    // Validate the inputs. The percentage/interval limits come from CodeDeploy
    validateParameters(props) {
        if (!(1 <= props.percentage && props.percentage <= 99)) {
            throw new Error(`Invalid deployment config percentage "${props.percentage.toString()}". \
        Step percentage must be an integer between 1 and 99.`);
        }
        if (props.interval.toMinutes() > 2880) {
            throw new Error(`Invalid deployment config interval "${props.interval.toString()}". \
        Traffic shifting intervals must be positive integers up to 2880 (2 days).`);
        }
    }
}
exports.CustomLambdaDeploymentConfig = CustomLambdaDeploymentConfig;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3VzdG9tLWRlcGxveW1lbnQtY29uZmlnLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiY3VzdG9tLWRlcGxveW1lbnQtY29uZmlnLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUFtRDtBQUNuRCxnRUFBMkc7QUFFM0csb0NBQWtEOzs7O0FBTWxELElBQVksZ0NBVVg7QUFWRCxXQUFZLGdDQUFnQztJQUkxQyxxREFBaUIsQ0FBQTtJQUtqQixxREFBaUIsQ0FBQTtBQUNuQixDQUFDLEVBVlcsZ0NBQWdDLEdBQWhDLHdDQUFnQyxLQUFoQyx3Q0FBZ0MsUUFVM0M7Ozs7OztBQXNDRCxNQUFhLDRCQUE2QixTQUFRLGVBQVE7Ozs7SUFjeEQsWUFBbUIsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBd0M7UUFDdkYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFFL0IsNERBQTREO1FBQzVELE1BQU0sY0FBYyxHQUFHLFdBQVcsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQzNELE1BQU0sZUFBZSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsU0FBUyxFQUFFLENBQUMsUUFBUSxFQUFFLENBQUM7UUFDOUQsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQyxRQUFRLEVBQUUsQ0FBQztRQUMvQyxJQUFJLGFBQWEsQ0FBQyxDQUFDLG1DQUFtQztRQUN0RCxJQUFJLEtBQUssQ0FBQyxJQUFJLElBQUksZ0NBQWdDLENBQUMsTUFBTSxFQUFFO1lBQ3pELGFBQWEsR0FBRztnQkFDZCxJQUFJLEVBQUUsY0FBYztnQkFDcEIsZUFBZSxFQUFFO29CQUNmLGNBQWMsRUFBRSxlQUFlO29CQUMvQixnQkFBZ0IsRUFBRSxVQUFVO2lCQUM3QjthQUNGLENBQUM7U0FDSDthQUFNLElBQUksS0FBSyxDQUFDLElBQUksSUFBSSxnQ0FBZ0MsQ0FBQyxNQUFNLEVBQUU7WUFDaEUsYUFBYSxHQUFHO2dCQUNkLElBQUksRUFBRSxjQUFjO2dCQUNwQixlQUFlLEVBQUU7b0JBQ2YsY0FBYyxFQUFFLGVBQWU7b0JBQy9CLGdCQUFnQixFQUFFLFVBQVU7aUJBQzdCO2FBQ0YsQ0FBQztTQUNIO1FBRUQsNEZBQTRGO1FBQzVGLDhHQUE4RztRQUM5Ryw0Q0FBNEM7UUFDNUMsSUFBSSxDQUFDLG9CQUFvQixHQUFHLEtBQUssQ0FBQyxvQkFBb0IsS0FBSyxTQUFTO1lBQ2xFLENBQUMsQ0FBQyxLQUFLLENBQUMsb0JBQW9CO1lBQzVCLENBQUMsQ0FBQyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxVQUFVLEtBQUssQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLFVBQVUsVUFBVSxLQUFLLENBQUMsSUFBSSxLQUFLLGdDQUFnQyxDQUFDLE1BQU07Z0JBQzVILENBQUMsQ0FBQyxPQUFPO2dCQUNULENBQUMsQ0FBQyxFQUFFLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDO1FBQy9DLElBQUksQ0FBQyxtQkFBbUIsR0FBRyw4QkFBc0IsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsQ0FBQztRQUU3RSxrRkFBa0Y7UUFDbEYsSUFBSSxvQ0FBaUIsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDOUMsUUFBUSxFQUFFO2dCQUNSLE9BQU8sRUFBRSxZQUFZO2dCQUNyQixNQUFNLEVBQUUsd0JBQXdCO2dCQUNoQyxVQUFVLEVBQUU7b0JBQ1Ysb0JBQW9CLEVBQUUsSUFBSSxDQUFDLG9CQUFvQjtvQkFDL0MsZUFBZSxFQUFFLFFBQVE7b0JBQ3pCLG9CQUFvQixFQUFFLGFBQWE7aUJBQ3BDO2dCQUNELCtEQUErRDtnQkFDL0Qsa0JBQWtCLEVBQUUscUNBQWtCLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQzthQUNyRTtZQUNELFFBQVEsRUFBRTtnQkFDUixPQUFPLEVBQUUsWUFBWTtnQkFDckIsTUFBTSxFQUFFLHdCQUF3QjtnQkFDaEMsVUFBVSxFQUFFO29CQUNWLG9CQUFvQixFQUFFLElBQUksQ0FBQyxvQkFBb0I7b0JBQy9DLGVBQWUsRUFBRSxRQUFRO29CQUN6QixvQkFBb0IsRUFBRSxhQUFhO2lCQUNwQztnQkFDRCwyRUFBMkU7Z0JBQzNFLHlEQUF5RDtnQkFDekQsa0JBQWtCLEVBQUUscUNBQWtCLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxvQkFBb0IsQ0FBQzthQUNyRTtZQUNELFFBQVEsRUFBRTtnQkFDUixPQUFPLEVBQUUsWUFBWTtnQkFDckIsTUFBTSxFQUFFLHdCQUF3QjtnQkFDaEMsVUFBVSxFQUFFO29CQUNWLG9CQUFvQixFQUFFLElBQUksQ0FBQyxvQkFBb0I7aUJBQ2hEO2FBQ0Y7WUFDRCxNQUFNLEVBQUUsMENBQXVCLENBQUMsWUFBWSxDQUFDO2dCQUMzQyxTQUFTLEVBQUUsMENBQXVCLENBQUMsWUFBWTthQUNoRCxDQUFDO1NBQ0gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVELDJFQUEyRTtJQUNuRSxrQkFBa0IsQ0FBQyxLQUF3QztRQUNqRSxJQUFLLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxDQUFDLFVBQVUsSUFBSSxLQUFLLENBQUMsVUFBVSxJQUFJLEVBQUUsQ0FBQyxFQUFHO1lBQ3hELE1BQU0sSUFBSSxLQUFLLENBQ2IseUNBQXlDLEtBQUssQ0FBQyxVQUFVLENBQUMsUUFBUSxFQUFFOzZEQUNmLENBQUMsQ0FBQztTQUMxRDtRQUNELElBQUksS0FBSyxDQUFDLFFBQVEsQ0FBQyxTQUFTLEVBQUUsR0FBRyxJQUFJLEVBQUU7WUFDckMsTUFBTSxJQUFJLEtBQUssQ0FDYix1Q0FBdUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxRQUFRLEVBQUU7a0ZBQ1UsQ0FBQyxDQUFDO1NBQy9FO0lBQ0gsQ0FBQztDQUNGO0FBdEdELG9FQXNHQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IER1cmF0aW9uLCBSZXNvdXJjZSB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQXdzQ3VzdG9tUmVzb3VyY2UsIEF3c0N1c3RvbVJlc291cmNlUG9saWN5LCBQaHlzaWNhbFJlc291cmNlSWQgfSBmcm9tICdAYXdzLWNkay9jdXN0b20tcmVzb3VyY2VzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgYXJuRm9yRGVwbG95bWVudENvbmZpZyB9IGZyb20gJy4uL3V0aWxzJztcbmltcG9ydCB7IElMYW1iZGFEZXBsb3ltZW50Q29uZmlnIH0gZnJvbSAnLi9kZXBsb3ltZW50LWNvbmZpZyc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIEN1c3RvbUxhbWJkYURlcGxveW1lbnRDb25maWdUeXBlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBDQU5BUlkgPSAnQ2FuYXJ5JyxcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIExJTkVBUiA9ICdMaW5lYXInXG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQ3VzdG9tTGFtYmRhRGVwbG95bWVudENvbmZpZ1Byb3BzIHtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdHlwZTogQ3VzdG9tTGFtYmRhRGVwbG95bWVudENvbmZpZ1R5cGU7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHBlcmNlbnRhZ2U6IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaW50ZXJ2YWw6IER1cmF0aW9uO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBkZXBsb3ltZW50Q29uZmlnTmFtZT86IHN0cmluZztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEN1c3RvbUxhbWJkYURlcGxveW1lbnRDb25maWcgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElMYW1iZGFEZXBsb3ltZW50Q29uZmlnIHtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyByZWFkb25seSBkZXBsb3ltZW50Q29uZmlnTmFtZTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgcmVhZG9ubHkgZGVwbG95bWVudENvbmZpZ0Fybjogc3RyaW5nO1xuXG4gIHB1YmxpYyBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ3VzdG9tTGFtYmRhRGVwbG95bWVudENvbmZpZ1Byb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICB0aGlzLnZhbGlkYXRlUGFyYW1ldGVycyhwcm9wcyk7XG5cbiAgICAvLyBJbiB0aGlzIHNlY3Rpb24gd2UgbWFrZSB0aGUgYXJndW1lbnQgZm9yIHRoZSBBV1MgQVBJIGNhbGxcbiAgICBjb25zdCBkZXBsb3ltZW50VHlwZSA9ICdUaW1lQmFzZWQnICsgcHJvcHMudHlwZS50b1N0cmluZygpO1xuICAgIGNvbnN0IGludGVydmFsTWludXRlcyA9IHByb3BzLmludGVydmFsLnRvTWludXRlcygpLnRvU3RyaW5nKCk7XG4gICAgY29uc3QgcGVyY2VudGFnZSA9IHByb3BzLnBlcmNlbnRhZ2UudG9TdHJpbmcoKTtcbiAgICBsZXQgcm91dGluZ0NvbmZpZzsgLy8gVGhlIGFyZ3VtZW50IHRvIHRoZSBBV1MgQVBJIGNhbGxcbiAgICBpZiAocHJvcHMudHlwZSA9PSBDdXN0b21MYW1iZGFEZXBsb3ltZW50Q29uZmlnVHlwZS5DQU5BUlkpIHtcbiAgICAgIHJvdXRpbmdDb25maWcgPSB7XG4gICAgICAgIHR5cGU6IGRlcGxveW1lbnRUeXBlLFxuICAgICAgICB0aW1lQmFzZWRDYW5hcnk6IHtcbiAgICAgICAgICBjYW5hcnlJbnRlcnZhbDogaW50ZXJ2YWxNaW51dGVzLFxuICAgICAgICAgIGNhbmFyeVBlcmNlbnRhZ2U6IHBlcmNlbnRhZ2UsXG4gICAgICAgIH0sXG4gICAgICB9O1xuICAgIH0gZWxzZSBpZiAocHJvcHMudHlwZSA9PSBDdXN0b21MYW1iZGFEZXBsb3ltZW50Q29uZmlnVHlwZS5MSU5FQVIpIHtcbiAgICAgIHJvdXRpbmdDb25maWcgPSB7XG4gICAgICAgIHR5cGU6IGRlcGxveW1lbnRUeXBlLFxuICAgICAgICB0aW1lQmFzZWRMaW5lYXI6IHtcbiAgICAgICAgICBsaW5lYXJJbnRlcnZhbDogaW50ZXJ2YWxNaW51dGVzLFxuICAgICAgICAgIGxpbmVhclBlcmNlbnRhZ2U6IHBlcmNlbnRhZ2UsXG4gICAgICAgIH0sXG4gICAgICB9O1xuICAgIH1cblxuICAgIC8vIEdlbmVyYXRlcyB0aGUgbmFtZSBvZiB0aGUgZGVwbG95bWVudCBjb25maWcuIEl0J3MgYWxzbyB3aGF0IHlvdSdsbCBzZWUgaW4gdGhlIEFXUyBjb25zb2xlXG4gICAgLy8gVGhlIG5hbWUgb2YgdGhlIGNvbmZpZyBpcyA8Y29uc3RydWN0IHVuaXF1ZSBpZD4uTGFtYmRhPGRlcGxveW1lbnQgdHlwZT48cGVyY2VudGFnZT5QZXJjZW50PGludGVydmFsPk1pbnV0ZXNcbiAgICAvLyBVbmxlc3MgdGhlIHVzZXIgcHJvdmlkZXMgYW4gZXhwbGljaXQgbmFtZVxuICAgIHRoaXMuZGVwbG95bWVudENvbmZpZ05hbWUgPSBwcm9wcy5kZXBsb3ltZW50Q29uZmlnTmFtZSAhPT0gdW5kZWZpbmVkXG4gICAgICA/IHByb3BzLmRlcGxveW1lbnRDb25maWdOYW1lXG4gICAgICA6IGAke3RoaXMubm9kZS51bmlxdWVJZH0uTGFtYmRhJHtwcm9wcy50eXBlfSR7cHJvcHMucGVyY2VudGFnZX1QZXJjZW50JHtwcm9wcy50eXBlID09PSBDdXN0b21MYW1iZGFEZXBsb3ltZW50Q29uZmlnVHlwZS5MSU5FQVJcbiAgICAgICAgPyAnRXZlcnknXG4gICAgICAgIDogJyd9JHtwcm9wcy5pbnRlcnZhbC50b01pbnV0ZXMoKX1NaW51dGVzYDtcbiAgICB0aGlzLmRlcGxveW1lbnRDb25maWdBcm4gPSBhcm5Gb3JEZXBsb3ltZW50Q29uZmlnKHRoaXMuZGVwbG95bWVudENvbmZpZ05hbWUpO1xuXG4gICAgLy8gVGhlIEFXUyBDdXN0b20gUmVzb3VyY2UgdGhhdCBjYWxscyBDb2RlRGVwbG95IHRvIGNyZWF0ZSBhbmQgZGVsZXRlIHRoZSByZXNvdXJjZVxuICAgIG5ldyBBd3NDdXN0b21SZXNvdXJjZSh0aGlzLCAnRGVwbG95bWVudENvbmZpZycsIHtcbiAgICAgIG9uQ3JlYXRlOiB7IC8vIFJ1biBvbiBjcmVhdGlvbiBvbmx5LCB0byBtYWtlIHRoZSByZXNvdXJjZVxuICAgICAgICBzZXJ2aWNlOiAnQ29kZURlcGxveScsXG4gICAgICAgIGFjdGlvbjogJ2NyZWF0ZURlcGxveW1lbnRDb25maWcnLFxuICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgZGVwbG95bWVudENvbmZpZ05hbWU6IHRoaXMuZGVwbG95bWVudENvbmZpZ05hbWUsXG4gICAgICAgICAgY29tcHV0ZVBsYXRmb3JtOiAnTGFtYmRhJyxcbiAgICAgICAgICB0cmFmZmljUm91dGluZ0NvbmZpZzogcm91dGluZ0NvbmZpZyxcbiAgICAgICAgfSxcbiAgICAgICAgLy8gVGhpcyBgcmVzb3VyY2VOYW1lYCBpcyB0aGUgaW5pdGlhbCBwaHlzaWNhbCBJRCBvZiB0aGUgY29uZmlnXG4gICAgICAgIHBoeXNpY2FsUmVzb3VyY2VJZDogUGh5c2ljYWxSZXNvdXJjZUlkLm9mKHRoaXMuZGVwbG95bWVudENvbmZpZ05hbWUpLFxuICAgICAgfSxcbiAgICAgIG9uVXBkYXRlOiB7IC8vIFJ1biBvbiBzdGFjayB1cGRhdGVcbiAgICAgICAgc2VydmljZTogJ0NvZGVEZXBsb3knLFxuICAgICAgICBhY3Rpb246ICdjcmVhdGVEZXBsb3ltZW50Q29uZmlnJyxcbiAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgIGRlcGxveW1lbnRDb25maWdOYW1lOiB0aGlzLmRlcGxveW1lbnRDb25maWdOYW1lLFxuICAgICAgICAgIGNvbXB1dGVQbGF0Zm9ybTogJ0xhbWJkYScsXG4gICAgICAgICAgdHJhZmZpY1JvdXRpbmdDb25maWc6IHJvdXRpbmdDb25maWcsXG4gICAgICAgIH0sXG4gICAgICAgIC8vIElmIGByZXNvdXJjZU5hbWVgIGlzIGRpZmZlcmVudCBmcm9tIHRoZSBsYXN0IHN0YWNrIHVwZGF0ZSAob3IgY3JlYXRpb24pLFxuICAgICAgICAvLyB0aGUgb2xkIGNvbmZpZyBnZXRzIGRlbGV0ZWQgYW5kIHRoZSBuZXcgb25lIGlzIGNyZWF0ZWRcbiAgICAgICAgcGh5c2ljYWxSZXNvdXJjZUlkOiBQaHlzaWNhbFJlc291cmNlSWQub2YodGhpcy5kZXBsb3ltZW50Q29uZmlnTmFtZSksXG4gICAgICB9LFxuICAgICAgb25EZWxldGU6IHsgLy8gUnVuIG9uIGRlbGV0aW9uLCBvciBvbiByZXNvdXJjZSByZXBsYWNlbWVudFxuICAgICAgICBzZXJ2aWNlOiAnQ29kZURlcGxveScsXG4gICAgICAgIGFjdGlvbjogJ2RlbGV0ZURlcGxveW1lbnRDb25maWcnLFxuICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgZGVwbG95bWVudENvbmZpZ05hbWU6IHRoaXMuZGVwbG95bWVudENvbmZpZ05hbWUsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgICAgcG9saWN5OiBBd3NDdXN0b21SZXNvdXJjZVBvbGljeS5mcm9tU2RrQ2FsbHMoe1xuICAgICAgICByZXNvdXJjZXM6IEF3c0N1c3RvbVJlc291cmNlUG9saWN5LkFOWV9SRVNPVVJDRSxcbiAgICAgIH0pLFxuICAgIH0pO1xuICB9XG5cbiAgLy8gVmFsaWRhdGUgdGhlIGlucHV0cy4gVGhlIHBlcmNlbnRhZ2UvaW50ZXJ2YWwgbGltaXRzIGNvbWUgZnJvbSBDb2RlRGVwbG95XG4gIHByaXZhdGUgdmFsaWRhdGVQYXJhbWV0ZXJzKHByb3BzOiBDdXN0b21MYW1iZGFEZXBsb3ltZW50Q29uZmlnUHJvcHMpOiB2b2lkIHtcbiAgICBpZiAoICEoMSA8PSBwcm9wcy5wZXJjZW50YWdlICYmIHByb3BzLnBlcmNlbnRhZ2UgPD0gOTkpICkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgSW52YWxpZCBkZXBsb3ltZW50IGNvbmZpZyBwZXJjZW50YWdlIFwiJHtwcm9wcy5wZXJjZW50YWdlLnRvU3RyaW5nKCl9XCIuIFxcXG4gICAgICAgIFN0ZXAgcGVyY2VudGFnZSBtdXN0IGJlIGFuIGludGVnZXIgYmV0d2VlbiAxIGFuZCA5OS5gKTtcbiAgICB9XG4gICAgaWYgKHByb3BzLmludGVydmFsLnRvTWludXRlcygpID4gMjg4MCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICBgSW52YWxpZCBkZXBsb3ltZW50IGNvbmZpZyBpbnRlcnZhbCBcIiR7cHJvcHMuaW50ZXJ2YWwudG9TdHJpbmcoKX1cIi4gXFxcbiAgICAgICAgVHJhZmZpYyBzaGlmdGluZyBpbnRlcnZhbHMgbXVzdCBiZSBwb3NpdGl2ZSBpbnRlZ2VycyB1cCB0byAyODgwICgyIGRheXMpLmApO1xuICAgIH1cbiAgfVxufVxuIl19