"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseAwsUrl = exports.AWS_URL_PATTERN = exports.normalizeComparisonString = exports.normalizeIdentifier = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
/**
 * Normalize string value to standard identifier form.
 *
 * Removes extraneous prefixes (Amazon, AWS), replaces all non-alphanumerics with underscore (_),
 * and converts to lower-case.
 * @param value - Value to normalize
 * @returns Returns normalized identifier string.
 */
function normalizeIdentifier(value) {
    value = value.replace(/^(Amazon|AWS)-?/i, ""); // remove prefix
    value = value.replace(/[^A-Za-z0-9]/g, "_"); // only alphanum + _
    value = value.replace(/_+/g, "_"); // de-dupe _
    return value.toLowerCase();
}
exports.normalizeIdentifier = normalizeIdentifier;
const ComparisonDictionary = {
    acl: "access_control_list",
    eni: "elastic_network_interface",
    eip: "elastic_ip_address",
};
/**
 * Normalizes string value for consistent comparison with variants between systems.
 * @param value - The value to normalize for comparison
 */
function normalizeComparisonString(value) {
    value = value.replace(/\.(png|svg)$/, "");
    value = normalizeIdentifier(value);
    Object.entries(ComparisonDictionary).forEach(([_key, _value]) => {
        _value = _value.replace(new RegExp(_key, "ig"), _value);
    });
    value = value.replace(/[_-]+(amazon|aws)[_-]+/gi, "");
    value = value.replace(/_(for|on|and|the|of|&)_/gi, "");
    value = value.replace(/v\d+/g, "");
    value = value.replace(/_/g, "");
    return value;
}
exports.normalizeComparisonString = normalizeComparisonString;
/**
 * RegEx pattern for https://aws.amazon.com based urls.
 */
exports.AWS_URL_PATTERN = /^https:\/\/(?<domain>(?:(?:docs)\.)?aws\.amazon\.com)\/(?<service>[^\/]+)(?:\/(?<resource>[^#]+))?\/?(?:#(?<dataType>.+))?$/i;
/** Parses https://aws.amazon.com based url into common definition. */
function parseAwsUrl(url) {
    url = url.replace(/\/(index\.html.*)?$/, "");
    const groups = url.match(exports.AWS_URL_PATTERN)
        ?.groups;
    if (groups) {
        const code = normalizeIdentifier(`${groups.service}_${groups.resource || ""}_${groups.dataType || ""}`);
        return {
            ...groups,
            code,
        };
    }
    throw new Error(`Failed to parse aws url: ${url}`);
}
exports.parseAwsUrl = parseAwsUrl;
//# sourceMappingURL=data:application/json;base64,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