"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractResolvedServiceName = exports.resolveServiceName = void 0;
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const assets_1 = require("../../generated/assets");
const utils_1 = require("../../utils");
const variants_1 = require("./variants");
/** @internal */
const SERVICE_NAME_LOOKUP = Object.fromEntries([
    ...Object.keys(assets_1.AwsAsset.Services).flatMap((key) => {
        return [
            // constant self lookup
            [key, key],
            [utils_1.normalizeComparisonString(key), key],
        ];
    }),
    ...Object.entries(variants_1.NORMALIZED_SERVICE_VARIANTS).flatMap(([key, variants]) => {
        if (variants == null)
            return [];
        return variants.map((variant) => [utils_1.normalizeComparisonString(variant), key]);
    }),
]);
/** @internal */
function resolveServiceName(value) {
    const comparableValue = utils_1.normalizeComparisonString(value);
    // constant lookup first for perf
    if (comparableValue in SERVICE_NAME_LOOKUP) {
        return SERVICE_NAME_LOOKUP[comparableValue];
    }
    // attempt know mappings
    if (comparableValue.startsWith("media") &&
        SERVICE_NAME_LOOKUP[`elemental${comparableValue}`]) {
        const serviceName = SERVICE_NAME_LOOKUP[`elemental${comparableValue}`];
        SERVICE_NAME_LOOKUP[comparableValue] = serviceName;
        return serviceName;
    }
    // prefixed lookup (WAFRegion => WAF, CodeGuruProfile => CodeGuru)
    for (const serviceName of Object.keys(assets_1.AwsAsset.Services)) {
        if (comparableValue.startsWith(utils_1.normalizeComparisonString(serviceName))) {
            // add to lookup for constant lookkup in subsequent calls;
            SERVICE_NAME_LOOKUP[comparableValue] = serviceName;
            return serviceName;
        }
    }
    throw new Error(`Failed to resolve serviceName ${value} (${comparableValue})`);
}
exports.resolveServiceName = resolveServiceName;
/**
 * Extract service name parts from value that is or contains a service name or variant of service name.
 * @internal
 */
function extractResolvedServiceName(value) {
    const parts = utils_1.normalizeIdentifier(value).split("_");
    for (let i = parts.length; i > 0; i--) {
        try {
            const match = parts.slice(0, i).join("_");
            const serviceName = resolveServiceName(match);
            let rest = value.replace(match, "");
            if (rest.length === 0) {
                rest = undefined;
            }
            const resolvedValue = `${serviceName}${rest}`;
            return { serviceName, match, rest, resolvedValue };
        }
        catch {
            continue;
        }
    }
    return;
}
exports.extractResolvedServiceName = extractResolvedServiceName;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaGVscGVycy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9pbnRlcm5hbC9zZXJ2aWNlcy9oZWxwZXJzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBOzs7Ozs7Ozs7Ozs7Ozt3SEFjd0g7QUFDeEgsbURBQWtEO0FBQ2xELHVDQUE2RTtBQUM3RSx5Q0FBeUQ7QUFFekQsZ0JBQWdCO0FBQ2hCLE1BQU0sbUJBQW1CLEdBQThCLE1BQU0sQ0FBQyxXQUFXLENBQUM7SUFDeEUsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLGlCQUFRLENBQUMsUUFBUSxDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsR0FBRyxFQUFFLEVBQUU7UUFDaEQsT0FBTztZQUNMLHVCQUF1QjtZQUN2QixDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUM7WUFDVixDQUFDLGlDQUF5QixDQUFDLEdBQUcsQ0FBQyxFQUFFLEdBQUcsQ0FBQztTQUN0QyxDQUFDO0lBQ0osQ0FBQyxDQUFDO0lBQ0YsR0FBRyxNQUFNLENBQUMsT0FBTyxDQUFDLHNDQUEyQixDQUFDLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUUsUUFBUSxDQUFDLEVBQUUsRUFBRTtRQUN6RSxJQUFJLFFBQVEsSUFBSSxJQUFJO1lBQUUsT0FBTyxFQUFFLENBQUM7UUFDaEMsT0FBTyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQyxDQUFDLGlDQUF5QixDQUFDLE9BQU8sQ0FBQyxFQUFFLEdBQUcsQ0FBQyxDQUFDLENBQUM7SUFDOUUsQ0FBQyxDQUFDO0NBQ0gsQ0FBQyxDQUFDO0FBRUgsZ0JBQWdCO0FBQ2hCLFNBQWdCLGtCQUFrQixDQUFDLEtBQWE7SUFDOUMsTUFBTSxlQUFlLEdBQUcsaUNBQXlCLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFekQsaUNBQWlDO0lBQ2pDLElBQUksZUFBZSxJQUFJLG1CQUFtQixFQUFFO1FBQzFDLE9BQU8sbUJBQW1CLENBQUMsZUFBZSxDQUFxQixDQUFDO0tBQ2pFO0lBRUQsd0JBQXdCO0lBQ3hCLElBQ0UsZUFBZSxDQUFDLFVBQVUsQ0FBQyxPQUFPLENBQUM7UUFDbkMsbUJBQW1CLENBQUMsWUFBWSxlQUFlLEVBQUUsQ0FBQyxFQUNsRDtRQUNBLE1BQU0sV0FBVyxHQUFHLG1CQUFtQixDQUFDLFlBQVksZUFBZSxFQUFFLENBQUMsQ0FBQztRQUN2RSxtQkFBbUIsQ0FBQyxlQUFlLENBQUMsR0FBRyxXQUFXLENBQUM7UUFDbkQsT0FBTyxXQUErQixDQUFDO0tBQ3hDO0lBRUQsa0VBQWtFO0lBQ2xFLEtBQUssTUFBTSxXQUFXLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxpQkFBUSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1FBQ3hELElBQUksZUFBZSxDQUFDLFVBQVUsQ0FBQyxpQ0FBeUIsQ0FBQyxXQUFXLENBQUMsQ0FBQyxFQUFFO1lBQ3RFLDBEQUEwRDtZQUMxRCxtQkFBbUIsQ0FBQyxlQUFlLENBQUMsR0FBRyxXQUFXLENBQUM7WUFDbkQsT0FBTyxXQUErQixDQUFDO1NBQ3hDO0tBQ0Y7SUFFRCxNQUFNLElBQUksS0FBSyxDQUNiLGlDQUFpQyxLQUFLLEtBQUssZUFBZSxHQUFHLENBQzlELENBQUM7QUFDSixDQUFDO0FBOUJELGdEQThCQztBQWFEOzs7R0FHRztBQUNILFNBQWdCLDBCQUEwQixDQUN4QyxLQUFhO0lBRWIsTUFBTSxLQUFLLEdBQUcsMkJBQW1CLENBQUMsS0FBSyxDQUFDLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBRXBELEtBQUssSUFBSSxDQUFDLEdBQUcsS0FBSyxDQUFDLE1BQU0sRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFO1FBQ3JDLElBQUk7WUFDRixNQUFNLEtBQUssR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDMUMsTUFBTSxXQUFXLEdBQUcsa0JBQWtCLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDOUMsSUFBSSxJQUFJLEdBQXVCLEtBQUssQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBQ3hELElBQUksSUFBSSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7Z0JBQ3JCLElBQUksR0FBRyxTQUFTLENBQUM7YUFDbEI7WUFDRCxNQUFNLGFBQWEsR0FBRyxHQUFHLFdBQVcsR0FBRyxJQUFJLEVBQUUsQ0FBQztZQUM5QyxPQUFPLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsYUFBYSxFQUFFLENBQUM7U0FDcEQ7UUFBQyxNQUFNO1lBQ04sU0FBUztTQUNWO0tBQ0Y7SUFFRCxPQUFPO0FBQ1QsQ0FBQztBQXJCRCxnRUFxQkMiLCJzb3VyY2VzQ29udGVudCI6WyIvKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqXG4gQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG5cbiBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLlxuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2Ugd2l0aCB0aGUgTGljZW5zZS5cbiBZb3UgbWF5IG9idGFpbiBhIGNvcHkgb2YgdGhlIExpY2Vuc2UgYXRcblxuIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuXG4gVW5sZXNzIHJlcXVpcmVkIGJ5IGFwcGxpY2FibGUgbGF3IG9yIGFncmVlZCB0byBpbiB3cml0aW5nLCBzb2Z0d2FyZVxuIGRpc3RyaWJ1dGVkIHVuZGVyIHRoZSBMaWNlbnNlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuIFwiQVMgSVNcIiBCQVNJUyxcbiBXSVRIT1VUIFdBUlJBTlRJRVMgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZWl0aGVyIGV4cHJlc3Mgb3IgaW1wbGllZC5cbiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnMgYW5kXG4gbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKioqKiogKi9cbmltcG9ydCB7IEF3c0Fzc2V0IH0gZnJvbSBcIi4uLy4uL2dlbmVyYXRlZC9hc3NldHNcIjtcbmltcG9ydCB7IG5vcm1hbGl6ZUNvbXBhcmlzb25TdHJpbmcsIG5vcm1hbGl6ZUlkZW50aWZpZXIgfSBmcm9tIFwiLi4vLi4vdXRpbHNcIjtcbmltcG9ydCB7IE5PUk1BTElaRURfU0VSVklDRV9WQVJJQU5UUyB9IGZyb20gXCIuL3ZhcmlhbnRzXCI7XG5cbi8qKiBAaW50ZXJuYWwgKi9cbmNvbnN0IFNFUlZJQ0VfTkFNRV9MT09LVVA6IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH0gPSBPYmplY3QuZnJvbUVudHJpZXMoW1xuICAuLi5PYmplY3Qua2V5cyhBd3NBc3NldC5TZXJ2aWNlcykuZmxhdE1hcCgoa2V5KSA9PiB7XG4gICAgcmV0dXJuIFtcbiAgICAgIC8vIGNvbnN0YW50IHNlbGYgbG9va3VwXG4gICAgICBba2V5LCBrZXldLFxuICAgICAgW25vcm1hbGl6ZUNvbXBhcmlzb25TdHJpbmcoa2V5KSwga2V5XSxcbiAgICBdO1xuICB9KSxcbiAgLi4uT2JqZWN0LmVudHJpZXMoTk9STUFMSVpFRF9TRVJWSUNFX1ZBUklBTlRTKS5mbGF0TWFwKChba2V5LCB2YXJpYW50c10pID0+IHtcbiAgICBpZiAodmFyaWFudHMgPT0gbnVsbCkgcmV0dXJuIFtdO1xuICAgIHJldHVybiB2YXJpYW50cy5tYXAoKHZhcmlhbnQpID0+IFtub3JtYWxpemVDb21wYXJpc29uU3RyaW5nKHZhcmlhbnQpLCBrZXldKTtcbiAgfSksXG5dKTtcblxuLyoqIEBpbnRlcm5hbCAqL1xuZXhwb3J0IGZ1bmN0aW9uIHJlc29sdmVTZXJ2aWNlTmFtZSh2YWx1ZTogc3RyaW5nKTogQXdzQXNzZXQuU2VydmljZSB7XG4gIGNvbnN0IGNvbXBhcmFibGVWYWx1ZSA9IG5vcm1hbGl6ZUNvbXBhcmlzb25TdHJpbmcodmFsdWUpO1xuXG4gIC8vIGNvbnN0YW50IGxvb2t1cCBmaXJzdCBmb3IgcGVyZlxuICBpZiAoY29tcGFyYWJsZVZhbHVlIGluIFNFUlZJQ0VfTkFNRV9MT09LVVApIHtcbiAgICByZXR1cm4gU0VSVklDRV9OQU1FX0xPT0tVUFtjb21wYXJhYmxlVmFsdWVdIGFzIEF3c0Fzc2V0LlNlcnZpY2U7XG4gIH1cblxuICAvLyBhdHRlbXB0IGtub3cgbWFwcGluZ3NcbiAgaWYgKFxuICAgIGNvbXBhcmFibGVWYWx1ZS5zdGFydHNXaXRoKFwibWVkaWFcIikgJiZcbiAgICBTRVJWSUNFX05BTUVfTE9PS1VQW2BlbGVtZW50YWwke2NvbXBhcmFibGVWYWx1ZX1gXVxuICApIHtcbiAgICBjb25zdCBzZXJ2aWNlTmFtZSA9IFNFUlZJQ0VfTkFNRV9MT09LVVBbYGVsZW1lbnRhbCR7Y29tcGFyYWJsZVZhbHVlfWBdO1xuICAgIFNFUlZJQ0VfTkFNRV9MT09LVVBbY29tcGFyYWJsZVZhbHVlXSA9IHNlcnZpY2VOYW1lO1xuICAgIHJldHVybiBzZXJ2aWNlTmFtZSBhcyBBd3NBc3NldC5TZXJ2aWNlO1xuICB9XG5cbiAgLy8gcHJlZml4ZWQgbG9va3VwIChXQUZSZWdpb24gPT4gV0FGLCBDb2RlR3VydVByb2ZpbGUgPT4gQ29kZUd1cnUpXG4gIGZvciAoY29uc3Qgc2VydmljZU5hbWUgb2YgT2JqZWN0LmtleXMoQXdzQXNzZXQuU2VydmljZXMpKSB7XG4gICAgaWYgKGNvbXBhcmFibGVWYWx1ZS5zdGFydHNXaXRoKG5vcm1hbGl6ZUNvbXBhcmlzb25TdHJpbmcoc2VydmljZU5hbWUpKSkge1xuICAgICAgLy8gYWRkIHRvIGxvb2t1cCBmb3IgY29uc3RhbnQgbG9va2t1cCBpbiBzdWJzZXF1ZW50IGNhbGxzO1xuICAgICAgU0VSVklDRV9OQU1FX0xPT0tVUFtjb21wYXJhYmxlVmFsdWVdID0gc2VydmljZU5hbWU7XG4gICAgICByZXR1cm4gc2VydmljZU5hbWUgYXMgQXdzQXNzZXQuU2VydmljZTtcbiAgICB9XG4gIH1cblxuICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgYEZhaWxlZCB0byByZXNvbHZlIHNlcnZpY2VOYW1lICR7dmFsdWV9ICgke2NvbXBhcmFibGVWYWx1ZX0pYFxuICApO1xufVxuXG4vKipcbiAqIEV4dHJhY3RlZCBzZXJ2aWNlIG5hbWUgZGVmaW5pdGlvblxuICogQGludGVybmFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRXh0cmFjdGVkU2VydmljZU5hbWUge1xuICByZWFkb25seSBzZXJ2aWNlTmFtZTogQXdzQXNzZXQuU2VydmljZTtcbiAgcmVhZG9ubHkgbWF0Y2g6IHN0cmluZztcbiAgcmVhZG9ubHkgcmVzdD86IHN0cmluZztcbiAgcmVhZG9ubHkgcmVzb2x2ZWRWYWx1ZTogc3RyaW5nO1xufVxuXG4vKipcbiAqIEV4dHJhY3Qgc2VydmljZSBuYW1lIHBhcnRzIGZyb20gdmFsdWUgdGhhdCBpcyBvciBjb250YWlucyBhIHNlcnZpY2UgbmFtZSBvciB2YXJpYW50IG9mIHNlcnZpY2UgbmFtZS5cbiAqIEBpbnRlcm5hbFxuICovXG5leHBvcnQgZnVuY3Rpb24gZXh0cmFjdFJlc29sdmVkU2VydmljZU5hbWUoXG4gIHZhbHVlOiBzdHJpbmdcbik6IEV4dHJhY3RlZFNlcnZpY2VOYW1lIHwgdW5kZWZpbmVkIHtcbiAgY29uc3QgcGFydHMgPSBub3JtYWxpemVJZGVudGlmaWVyKHZhbHVlKS5zcGxpdChcIl9cIik7XG5cbiAgZm9yIChsZXQgaSA9IHBhcnRzLmxlbmd0aDsgaSA+IDA7IGktLSkge1xuICAgIHRyeSB7XG4gICAgICBjb25zdCBtYXRjaCA9IHBhcnRzLnNsaWNlKDAsIGkpLmpvaW4oXCJfXCIpO1xuICAgICAgY29uc3Qgc2VydmljZU5hbWUgPSByZXNvbHZlU2VydmljZU5hbWUobWF0Y2gpO1xuICAgICAgbGV0IHJlc3Q6IHN0cmluZyB8IHVuZGVmaW5lZCA9IHZhbHVlLnJlcGxhY2UobWF0Y2gsIFwiXCIpO1xuICAgICAgaWYgKHJlc3QubGVuZ3RoID09PSAwKSB7XG4gICAgICAgIHJlc3QgPSB1bmRlZmluZWQ7XG4gICAgICB9XG4gICAgICBjb25zdCByZXNvbHZlZFZhbHVlID0gYCR7c2VydmljZU5hbWV9JHtyZXN0fWA7XG4gICAgICByZXR1cm4geyBzZXJ2aWNlTmFtZSwgbWF0Y2gsIHJlc3QsIHJlc29sdmVkVmFsdWUgfTtcbiAgICB9IGNhdGNoIHtcbiAgICAgIGNvbnRpbnVlO1xuICAgIH1cbiAgfVxuXG4gIHJldHVybjtcbn1cbiJdfQ==