"use strict";
var _a, _b, _c, _d, _e;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsArchitecture = exports.AwsResource = exports.AwsService = exports.AwsCategory = exports.AwsCategoryDrawioStyles = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const contants_1 = require("./contants");
const assets_1 = require("./generated/assets");
const drawio_spec_1 = require("./generated/drawio-spec");
const mappings_1 = require("./generated/mappings");
const helpers_1 = require("./internal/assets/helpers");
const definitions_1 = require("./internal/categories/definitions");
const types_1 = require("./internal/drawio/types");
const pricing_manifest_1 = require("./internal/pricing-manifest");
const themes = require("./themes");
/**
 * AwsCategoryDrawioStyles is a utility class for constructing drawio shape
 * styles for services and resources.
 */
class AwsCategoryDrawioStyles {
    constructor(categoryShape, base) {
        this.categoryShape = categoryShape;
        this.base = base;
    }
    /**
     * Get the drawio style for this category.
     */
    get categoryStyle() {
        return {
            ...this.base,
            shape: types_1.DrawioAws4ParentShapes.RESOURCE_ICON,
            resIcon: this.categoryShape,
        };
    }
    /**
     * Gets the drawio style for a service based on the category style.
     * @param serviceShape The service shape to style based on category
     * @returns {AwsDrawioResourceIconStyle} The style drawio style definition for the resource based on category style.
     */
    getServiceStyle(serviceShape) {
        return {
            ...this.categoryStyle,
            resIcon: serviceShape,
        };
    }
    /**
     * Gets the drawio style for a resource based on the category style.
     * @param resourceShape The resource shape to style based on category
     * @returns {AwsDrawioResourceIconStyle} The style drawio style definition for the resource based on category style.
     */
    getResourceStyle(resourceShape) {
        const { resIcon, ...categoryStyle } = this.categoryStyle;
        return {
            ...categoryStyle,
            shape: resourceShape,
            pointerEvent: 1,
            strokeColor: "none",
        };
    }
}
exports.AwsCategoryDrawioStyles = AwsCategoryDrawioStyles;
_a = JSII_RTTI_SYMBOL_1;
AwsCategoryDrawioStyles[_a] = { fqn: "@aws-prototyping-sdk/aws-arch.AwsCategoryDrawioStyles", version: "0.12.9" };
/**
 * AwsCategory class provides an interface for normalizing category metadata
 * between mapped systems.
 */
class AwsCategory {
    /** @internal */
    constructor(definition) {
        this.id = definition.id;
        this.name = definition.name;
        this.variants = [...(definition.variants || [])];
        this.fillColor = definition.fillColor;
        this.gradientColor = definition.gradientColor;
        this.fontColor = definition.fontColor || definitions_1.CATEGORY_DEFAULT_FONT_COLOR;
        if (this.id !== definitions_1.GENERAL_CATEGORY_ID) {
            if (this.id in assets_1.AwsAsset.Categories) {
                this._assetIcon = assets_1.AwsAsset.Categories[this.id];
            }
            else {
                throw new Error(`AwsCategory id "${this.id}" does not match AwsAsset.Category`);
            }
        }
        const drawioShape = [this.id, ...this.variants].find((v) => v.toUpperCase() in drawio_spec_1.DrawioSpec.Aws4.ShapeNames);
        if (drawioShape) {
            this.drawioStyles = new AwsCategoryDrawioStyles(drawioShape, {
                ...types_1.DRAWIO_RESOURCE_STYLE_BASE,
                fillColor: this.fillColor,
                gradientColor: this.gradientColor,
                fontColor: this.fontColor,
            });
        }
    }
    /** @internal */
    static _register(definition) {
        const instance = new AwsCategory(definition);
        this._instanceMap.set(definition.id, instance);
        return instance;
    }
    /**
     * Get {@link AwsCategory} based on {@link AwsCategoryId}
     * @param id {AwsCategoryId} The id of the category to retrieve.
     * @returns Returns the category with the id
     */
    static getCategory(id) {
        if (this._instanceMap.has(id)) {
            return this._instanceMap.get(id);
        }
        throw new Error(`AwsCategory ${id} is not registered`);
    }
    /**
     * Get record of all categories keyed by category id.
     */
    static get categories() {
        return Object.fromEntries(this._instanceMap.entries());
    }
    /**
     * Retrieves a well-formatted relative path to the icon for this given
     * category in the specified format.
     */
    icon(format, theme) {
        if (this._assetIcon == null)
            return;
        try {
            return AwsArchitecture.formatAssetPath(this._assetIcon, format, theme);
        }
        catch { }
        return;
    }
    /**
     * Gets a list of all services within this category.
     */
    categoryServices() {
        if (this._services == null) {
            this._services = Object.values(AwsService.services).filter((service) => service.category === this);
        }
        return this._services;
    }
}
exports.AwsCategory = AwsCategory;
_b = JSII_RTTI_SYMBOL_1;
AwsCategory[_b] = { fqn: "@aws-prototyping-sdk/aws-arch.AwsCategory", version: "0.12.9" };
/** @internal */
AwsCategory._instanceMap = new Map();
// Register all categories
for (const categoryDefinition of Object.values(definitions_1.AwsCategoryDefinitions)) {
    AwsCategory._register(categoryDefinition);
}
/**
 * AwsService class provides an interface for normalizing service metadata
 * between mapped systems.
 */
class AwsService {
    /** @internal */
    constructor(cfnKey, cfnMapped) {
        this.cfnService = cfnKey;
        this.cfnProvider = cfnMapped.provider;
        if (cfnMapped.assetKey) {
            this._assetKey = cfnMapped.assetKey;
            const _parsed = helpers_1.parseAssetPath(assets_1.AwsAsset.Services[cfnMapped.assetKey]);
            this._assetIcon = _parsed.service && assets_1.AwsAsset.Services[_parsed.service];
            this.category = AwsCategory.getCategory(_parsed.category);
        }
        this.pricingServiceCode = cfnMapped.pricingServiceCode;
        this.drawioShape = cfnMapped.drawioShape;
    }
    /** @internal */
    static _register(cfnKey, cfnMapped) {
        const instance = new AwsService(cfnKey, cfnMapped);
        this._instanceMap.set(cfnKey, instance);
        return instance;
    }
    /**
     * Get {@link AwsService} by CloudFormation "service" name, where service name is expressed
     * as `<provider>::<service>::<resource>`.
     * @param cfnService The service name to retrieve {@link AwsService} for.
     * @returns Returns the {@link AwsService} associated with the `cfnService` provided
     * @throws Error is service not found
     */
    static getService(cfnService) {
        if (this._instanceMap.has(cfnService)) {
            return this._instanceMap.get(cfnService);
        }
        throw new Error(`AwsService ${cfnService} is not registered`);
    }
    /**
     * Finds the {@link AwsService} associated with a given value.
     * @param value Value to match {@link AwsService}, which can be `id`, `assetKey`, `fullName`, etc.
     * @returns Returns matching {@link AwsService} or `undefined` if not found
     * @throws Error if service not found
     */
    static findService(value) {
        if (this._instanceMap.has(value)) {
            return this._instanceMap.get(value);
        }
        for (const instance of this._instanceMap.values()) {
            if (instance._assetKey === value || instance.fullName === value) {
                return instance;
            }
        }
        return;
    }
    /**
     * Get record of all {@link AwsService}s keyed by `id`
     */
    static get services() {
        return Object.fromEntries(this._instanceMap.entries());
    }
    /**
     * The proper full name of the service.
     *
     * @example "AWS Glue", "Amazon S3"
     */
    get fullName() {
        if (this.pricingMetadata) {
            return this.pricingMetadata.name;
        }
        if (this._assetKey) {
            return assets_1.AwsAsset.AssetFullNameLookup[this._assetKey];
        }
        return this.cfnService;
    }
    /**
     * Get relative asset icon for the service for a given format and optional theme.
     * @param {IconFormats} format - The format of icon.
     * @param {ThemesEnum} [theme] - Optional theme
     * @returns Returns relative asset icon path
     */
    icon(format, theme) {
        if (!this._assetIcon)
            return undefined;
        try {
            return AwsArchitecture.formatAssetPath(this._assetIcon, format, theme);
        }
        catch { }
        return;
    }
    /**
     * Get drawio style for this service
     */
    drawioStyle() {
        // compute on first access
        if (this._drawioStyle === undefined) {
            if (this.category && this.category.drawioStyles && this.drawioShape) {
                this._drawioStyle = this.category.drawioStyles.getServiceStyle(this.drawioShape);
            }
            else {
                // prevent recomputation
                this._drawioStyle = null;
            }
        }
        return this._drawioStyle || undefined;
    }
    /**
     * List all resources of this service
     */
    serviceResources() {
        if (this._resources == null) {
            this._resources = Object.values(AwsResource.resources).filter((resource) => resource.service === this);
        }
        return this._resources;
    }
    /**
     * Get service pricing metadata.
     */
    get pricingMetadata() {
        return (this.pricingServiceCode &&
            pricing_manifest_1.PricingManifest.Services[this.pricingServiceCode]);
    }
}
exports.AwsService = AwsService;
_c = JSII_RTTI_SYMBOL_1;
AwsService[_c] = { fqn: "@aws-prototyping-sdk/aws-arch.AwsService", version: "0.12.9" };
/** @internal */
AwsService._instanceMap = new Map();
// Register all services
for (const [cfnKey, mapping] of Object.entries(mappings_1.AwsServiceMapping)) {
    AwsService._register(cfnKey, mapping);
}
/**
 * AwsResource class provides an interface for normalizing resource metadata
 * between mapped systems.
 */
class AwsResource {
    /** @internal */
    constructor(cfnResourceType, cfnMapped) {
        this.cfnResourceType = cfnResourceType;
        this.service = AwsService.getService(cfnMapped.service);
        if (cfnMapped.assetKey) {
            this._assetKey = cfnMapped.assetKey;
            this._assetIcon = assets_1.AwsAsset.Resources[cfnMapped.assetKey];
            this.fullName = assets_1.AwsAsset.AssetFullNameLookup[cfnMapped.assetKey];
        }
        if (cfnMapped.serviceAssetKey) {
            this._serviceIcon = assets_1.AwsAsset.Services[cfnMapped.serviceAssetKey];
        }
        if (cfnMapped.generalIconKey) {
            this._generalIcon = assets_1.AwsAsset.GeneralIcons[cfnMapped.generalIconKey];
        }
        this.drawioShape = cfnMapped.drawioShape || cfnMapped.drawioGeneralShape;
    }
    /** @internal */
    static _register(cfnResourceType, cfnMapped) {
        const instance = new AwsResource(cfnResourceType, cfnMapped);
        this._instanceMap.set(cfnResourceType, instance);
        return instance;
    }
    /**
     * Get {@link AwsResource} by CloudFormation resource type.
     * @param cfnResourceType - Fully qualifief CloudFormation resource type (eg: AWS:S3:Bucket)
     * @throws Error is no resource found
     */
    static getResource(cfnResourceType) {
        if (this._instanceMap.has(cfnResourceType)) {
            return this._instanceMap.get(cfnResourceType);
        }
        throw new Error(`AwsResource ${cfnResourceType} is not registered`);
    }
    /**
     * Find {@link AwsResource} associated with given value.
     * @param value - The value to match {@link AwsResource}; can be id, asset key, full name, etc.
     * @throws Error is no resource found
     */
    static findResource(value) {
        if (this._instanceMap.has(value)) {
            return this._instanceMap.get(value);
        }
        for (const instance of this._instanceMap.values()) {
            if (instance._assetKey === value || instance.fullName === value) {
                return instance;
            }
        }
        throw new Error(`AwsService ${value} is not registered`);
    }
    /**
     * Get record of all resources keyed by resource id.
     */
    static get resources() {
        return Object.fromEntries(this._instanceMap.entries());
    }
    /**
     * Gets the service icon for the resource.
     *
     * This maybe different than {@link AwsResource.service.icon} based on mappings overrides, which
     * if do not exist will fallback to {@link AwsResource.service.icon}.
     *
     * @param {IconFormats} format - The format of icon.
     * @param {ThemesEnum} [theme] - Optional theme
     * @returns Returns relative asset icon path
     *
     * @see {@link AwsService.icon}
     */
    getServiceIcon(format, theme) {
        if (this._serviceIcon) {
            try {
                return AwsArchitecture.formatAssetPath(this._serviceIcon, format, theme);
            }
            catch { }
        }
        return this.service.icon(format, theme);
    }
    /**
     * Gets the resource specific icon for the resource.
     * @param {IconFormats} format - The format of icon.
     * @param {ThemesEnum} [theme] - Optional theme
     * @returns Returns relative asset icon path
     */
    getResourceIcon(format, theme) {
        if (!this._assetIcon)
            return undefined;
        try {
            return AwsArchitecture.formatAssetPath(this._assetIcon, format, theme);
        }
        catch { }
        return;
    }
    /** @internal */
    _getGeneralIcon(format, theme) {
        if (!this._generalIcon)
            return undefined;
        try {
            return AwsArchitecture.formatAssetPath(this._generalIcon, format, theme);
        }
        catch { }
        return;
    }
    /**
     * Gets the best icon match for the resource following the order of:
     * 1. explicit resource icon
     * 2. general icon
     * 3. service icon
     * @param {IconFormats} format - The format of icon.
     * @param {ThemesEnum} [theme] - Optional theme
     * @returns Returns relative asset icon path
     */
    icon(format, theme) {
        return (this.getResourceIcon(format, theme) ||
            this._getGeneralIcon(format, theme) ||
            this.getServiceIcon(format, theme));
    }
    /**
     * Gets the draiwio style for the resource.
     */
    drawioStyle() {
        // compute on first access
        if (this._drawioStyle === undefined) {
            if (this.service.category &&
                this.service.category.drawioStyles &&
                this.drawioShape) {
                this._drawioStyle = this.service.category.drawioStyles.getResourceStyle(this.drawioShape);
            }
            else {
                // prevent recomputation
                this._drawioStyle = null;
            }
        }
        return this._drawioStyle || undefined;
    }
}
exports.AwsResource = AwsResource;
_d = JSII_RTTI_SYMBOL_1;
AwsResource[_d] = { fqn: "@aws-prototyping-sdk/aws-arch.AwsResource", version: "0.12.9" };
/** @internal */
AwsResource._instanceMap = new Map();
// Register all resources
for (const [cfnKey, mapping] of Object.entries(mappings_1.AwsResourceMapping)) {
    AwsResource._register(cfnKey, mapping);
}
/**
 * AwsArchitecture provides an interface for retrieving the inferred normalization between [@aws-cdk/cfnspec](https://github.com/aws/aws-cdk/blob/main/packages/%40aws-cdk/cfnspec)
 * and [AWS Architecture Icons](https://aws.amazon.com/architecture/icons/) systems
 * for all CloudFormation "services" and "resources".
 */
class AwsArchitecture {
    constructor() { }
    /**
     * The absolute directory where [AWS Architecture Icons](https://aws.amazon.com/architecture/icons/)
     * are stored and retrieved.
     */
    static get assetDirectory() {
        if (this._assetDirectory == null) {
            this._assetDirectory = path.join(__dirname, "..", contants_1.ASSET_DIRNAME);
        }
        return this._assetDirectory;
    }
    /**
     * Get all categories
     * @see {@link AwsCategory.categories}
     */
    static get categories() {
        return AwsCategory.categories;
    }
    /**
     * Get all services
     * @see {@link AwsService.services}
     */
    static get services() {
        return AwsService.services;
    }
    /**
     * Get all resources
     * @see {@link AwsResource.resources}
     */
    static get resources() {
        return AwsResource.resources;
    }
    /**
     * Get specific category based on id
     * @see {@link AwsCategory.getCategory}
     */
    static getCategory(category) {
        return AwsCategory.getCategory(category);
    }
    /**
     * Get specific service based on identifier (eg: S3, AWS::S3, AWS::S3::Bucket)
     * @see {@link AwsSerfice.getService}
     */
    static getService(identifier) {
        if (identifier.includes("::")) {
            identifier = identifier.split("::")[1];
        }
        return AwsService.getService(identifier);
    }
    /**
     * Get resource based on Cfn Resource Type (eg: AWS::S3::Bucket)
     * @see {@link AwsResource.getResource}
     */
    static getResource(cfnType) {
        return AwsResource.getResource(cfnType);
    }
    /**
     * Get icon for EC2 instance type.
     * @param instanceType - The {@link AwsAsset.InstanceType} to get icon for
     * @param {IconFormats} format - The format of icon.
     * @param {ThemesEnum} [theme] - Optional theme
     * @returns Returns relative asset icon path
     */
    static getInstanceTypeIcon(instanceType, format = "png", theme) {
        return this.formatAssetPath(assets_1.AwsAsset.InstanceTypes[instanceType], format, theme);
    }
    /**
     * Resolve relative asset path to absolute asset path.
     * @param assetPath - The relative asset path to resolve.
     * @returns {string} Absolute asset path
     */
    static resolveAssetPath(assetPath) {
        if (assetPath == null)
            throw new Error("Failed to resolve undefined asset path");
        return path.join(AwsArchitecture.assetDirectory, assetPath);
    }
    /**
     * Gets formatted asset path including extension and theme.
     * @param qualifiedAssetKey The qualified asset key (eg: compute/ec2/service_icon, storage/s3/bucket)
     * @param format {IconFormats} The format to return (eg: png, svg)
     * @param theme {ThemesEnum} - (Optional) The theme to use, if not specific or now matching asset for the them, the default theme is used
     * @returns Relative asset file path
     */
    static formatAssetPath(qualifiedAssetKey, format, theme) {
        if (theme && theme !== themes.DefaultThemeId) {
            const themedIcon = `${qualifiedAssetKey}.${theme}.${format}`;
            if (assets_1.AwsAsset.AssetFiles.has(themedIcon)) {
                return themedIcon;
            }
        }
        const icon = `${qualifiedAssetKey}.${format}`;
        if (assets_1.AwsAsset.AssetFiles.has(icon)) {
            return icon;
        }
        throw new Error(`Invalid asset key "${qualifiedAssetKey}"`);
    }
}
exports.AwsArchitecture = AwsArchitecture;
_e = JSII_RTTI_SYMBOL_1;
AwsArchitecture[_e] = { fqn: "@aws-prototyping-sdk/aws-arch.AwsArchitecture", version: "0.12.9" };
//# sourceMappingURL=data:application/json;base64,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