"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Portfolio = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const association_manager_1 = require("./private/association-manager");
const util_1 = require("./private/util");
const validation_1 = require("./private/validation");
const aws_servicecatalog_1 = require("aws-cdk-lib/aws-servicecatalog");
class PortfolioBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.associatedPrincipals = new Set();
    }
    /**
     * (experimental) Associate portfolio with an IAM Role.
     *
     * @experimental
     */
    giveAccessToRole(role) {
        this.associatePrincipal(role.roleArn, role.node.addr);
    }
    /**
     * (experimental) Associate portfolio with an IAM User.
     *
     * @experimental
     */
    giveAccessToUser(user) {
        this.associatePrincipal(user.userArn, user.node.addr);
    }
    /**
     * (experimental) Associate portfolio with an IAM Group.
     *
     * @experimental
     */
    giveAccessToGroup(group) {
        this.associatePrincipal(group.groupArn, group.node.addr);
    }
    /**
     * (experimental) Associate portfolio with the given product.
     *
     * @experimental
     */
    addProduct(product) {
        association_manager_1.AssociationManager.associateProductWithPortfolio(this, product, undefined);
    }
    /**
     * (experimental) Initiate a portfolio share with another account.
     *
     * @experimental
     */
    shareWithAccount(accountId, options = {}) {
        const hashId = this.generateUniqueHash(accountId);
        new aws_servicecatalog_1.CfnPortfolioShare(this, `PortfolioShare${hashId}`, {
            portfolioId: this.portfolioId,
            accountId: accountId,
            shareTagOptions: options.shareTagOptions,
            acceptLanguage: options.messageLanguage,
        });
    }
    /**
     * (experimental) Associate Tag Options.
     *
     * A TagOption is a key-value pair managed in AWS Service Catalog.
     * It is not an AWS tag, but serves as a template for creating an AWS tag based on the TagOption.
     *
     * @experimental
     */
    associateTagOptions(tagOptions) {
        association_manager_1.AssociationManager.associateTagOptions(this, tagOptions);
    }
    /**
     * (experimental) Add a Resource Update Constraint.
     *
     * @experimental
     */
    constrainTagUpdates(product, options = {}) {
        association_manager_1.AssociationManager.constrainTagUpdates(this, product, options);
    }
    /**
     * (experimental) Add notifications for supplied topics on the provisioned product.
     *
     * @experimental
     */
    notifyOnStackEvents(product, topic, options = {}) {
        association_manager_1.AssociationManager.notifyOnStackEvents(this, product, topic, options);
    }
    /**
     * (experimental) Set provisioning rules for the product.
     *
     * @experimental
     */
    constrainCloudFormationParameters(product, options) {
        association_manager_1.AssociationManager.constrainCloudFormationParameters(this, product, options);
    }
    /**
     * (experimental) Force users to assume a certain role when launching a product.
     *
     * @experimental
     */
    setLaunchRole(product, launchRole, options = {}) {
        association_manager_1.AssociationManager.setLaunchRole(this, product, launchRole, options);
    }
    /**
     * (experimental) Configure deployment options using AWS Cloudformaiton StackSets.
     *
     * @experimental
     */
    deployWithStackSets(product, options) {
        association_manager_1.AssociationManager.deployWithStackSets(this, product, options);
    }
    /**
     * Associate a principal with the portfolio.
     * If the principal is already associated, it will skip.
     */
    associatePrincipal(principalArn, principalId) {
        if (!this.associatedPrincipals.has(principalArn)) {
            const hashId = this.generateUniqueHash(principalId);
            new aws_servicecatalog_1.CfnPortfolioPrincipalAssociation(this, `PortolioPrincipalAssociation${hashId}`, {
                portfolioId: this.portfolioId,
                principalArn: principalArn,
                principalType: 'IAM',
            });
            this.associatedPrincipals.add(principalArn);
        }
    }
}
/**
 * (experimental) A Service Catalog portfolio.
 *
 * @experimental
 */
class Portfolio extends PortfolioBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.validatePortfolioProps(props);
        this.portfolio = new aws_servicecatalog_1.CfnPortfolio(this, 'Resource', {
            displayName: props.displayName,
            providerName: props.providerName,
            description: props.description,
            acceptLanguage: props.messageLanguage,
        });
        this.portfolioId = this.portfolio.ref;
        this.portfolioArn = cdk.Stack.of(this).formatArn({
            service: 'servicecatalog',
            resource: 'portfolio',
            resourceName: this.portfolioId,
        });
        if (props.tagOptions !== undefined) {
            this.associateTagOptions(props.tagOptions);
        }
    }
    /**
     * (experimental) Creates a Portfolio construct that represents an external portfolio.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param portfolioArn the Amazon Resource Name of the existing portfolio.
     * @experimental
     */
    static fromPortfolioArn(scope, id, portfolioArn) {
        const arn = cdk.Stack.of(scope).splitArn(portfolioArn, cdk.ArnFormat.SLASH_RESOURCE_NAME);
        const portfolioId = arn.resourceName;
        if (!portfolioId) {
            throw new Error('Missing required Portfolio ID from Portfolio ARN: ' + portfolioArn);
        }
        class Import extends PortfolioBase {
            constructor() {
                super(...arguments);
                this.portfolioArn = portfolioArn;
                this.portfolioId = portfolioId;
            }
            generateUniqueHash(value) {
                return util_1.hashValues(this.portfolioArn, value);
            }
        }
        return new Import(scope, id, {
            environmentFromArn: portfolioArn,
        });
    }
    /**
     * (experimental) Create a unique id based off the L1 CfnPortfolio or the arn of an imported portfolio.
     *
     * @experimental
     */
    generateUniqueHash(value) {
        return util_1.hashValues(cdk.Names.nodeUniqueId(this.portfolio.node), value);
    }
    validatePortfolioProps(props) {
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio display name', 1, 100, props.displayName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio provider name', 1, 50, props.providerName);
        validation_1.InputValidator.validateLength(this.node.path, 'portfolio description', 0, 2000, props.description);
    }
}
exports.Portfolio = Portfolio;
_a = JSII_RTTI_SYMBOL_1;
Portfolio[_a] = { fqn: "@aws-cdk/aws-servicecatalog-alpha.Portfolio", version: "2.0.0-alpha.4" };
//# sourceMappingURL=data:application/json;base64,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