"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphqlApi = exports.IamResource = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appsync_1 = require("aws-cdk-lib/aws-appsync");
const graphqlapi_base_1 = require("./graphqlapi-base");
const schema_1 = require("./schema");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
    /**
     * Lambda authorization type
     */
    AuthorizationType["LAMBDA"] = "AWS_LAMBDA";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * A class used to generate resource arns for AppSync
 */
class IamResource {
    constructor(arns) {
        this.arns = arns;
    }
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns) {
        if (arns.length === 0) {
            throw new Error('At least 1 custom ARN must be provided.');
        }
        return new IamResource(arns);
    }
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type, ...fields) {
        const arns = fields.length ? fields.map((field) => `types/${type}/fields/${field}`) : [`types/${type}/*`];
        return new IamResource(arns);
    }
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all() {
        return new IamResource(['*']);
    }
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_GraphqlApi(api);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.resourceArns);
            }
            throw error;
        }
        return this.arns.map((arn) => aws_cdk_lib_1.Stack.of(api).formatArn({
            service: 'appsync',
            resource: `apis/${api.apiId}`,
            arnFormat: aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME,
            resourceName: `${arn}`,
        }));
    }
}
exports.IamResource = IamResource;
_a = JSII_RTTI_SYMBOL_1;
IamResource[_a] = { fqn: "@aws-cdk/aws-appsync-alpha.IamResource", version: "2.28.0-alpha.0" };
/**
 * An AppSync GraphQL API
 *
 * @resource AWS::AppSync::GraphQLApi
 */
class GraphqlApi extends graphqlapi_base_1.GraphqlApiBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_GraphqlApiProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, GraphqlApi);
            }
            throw error;
        }
        const defaultMode = props.authorizationConfig?.defaultAuthorization ??
            { authorizationType: AuthorizationType.API_KEY };
        const additionalModes = props.authorizationConfig?.additionalAuthorizationModes ?? [];
        const modes = [defaultMode, ...additionalModes];
        this.modes = modes.map((mode) => mode.authorizationType);
        this.validateAuthorizationProps(modes);
        this.api = new aws_appsync_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultMode.authorizationType,
            logConfig: this.setupLogConfig(props.logConfig),
            openIdConnectConfig: this.setupOpenIdConnectConfig(defaultMode.openIdConnectConfig),
            userPoolConfig: this.setupUserPoolConfig(defaultMode.userPoolConfig),
            lambdaAuthorizerConfig: this.setupLambdaAuthorizerConfig(defaultMode.lambdaAuthorizerConfig),
            additionalAuthenticationProviders: this.setupAdditionalAuthorizationModes(additionalModes),
            xrayEnabled: props.xrayEnabled,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphqlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        this.schema = props.schema ?? new schema_1.Schema();
        this.schemaResource = this.schema.bind(this);
        if (props.domainName) {
            new aws_appsync_1.CfnDomainName(this, 'DomainName', {
                domainName: props.domainName.domainName,
                certificateArn: props.domainName.certificate.certificateArn,
                description: `domain for ${this.name} at ${this.graphqlUrl}`,
            });
            new aws_appsync_1.CfnDomainNameApiAssociation(this, 'DomainAssociation', {
                domainName: props.domainName.domainName,
                apiId: this.apiId,
            });
        }
        if (modes.some((mode) => mode.authorizationType === AuthorizationType.API_KEY)) {
            const config = modes.find((mode) => {
                return mode.authorizationType === AuthorizationType.API_KEY && mode.apiKeyConfig;
            })?.apiKeyConfig;
            this.apiKeyResource = this.createAPIKey(config);
            this.apiKeyResource.addDependsOn(this.schemaResource);
            this.apiKey = this.apiKeyResource.attrApiKey;
        }
        if (modes.some((mode) => mode.authorizationType === AuthorizationType.LAMBDA)) {
            const config = modes.find((mode) => {
                return mode.authorizationType === AuthorizationType.LAMBDA && mode.lambdaAuthorizerConfig;
            })?.lambdaAuthorizerConfig;
            config?.handler.addPermission('appsync', {
                principal: new aws_iam_1.ServicePrincipal('appsync.amazonaws.com'),
                action: 'lambda:InvokeFunction',
            });
        }
    }
    /**
     * Import a GraphQL API through this function
     *
     * @param scope scope
     * @param id id
     * @param attrs GraphQL API Attributes of an API
     */
    static fromGraphqlApiAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_GraphqlApiAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromGraphqlApiAttributes);
            }
            throw error;
        }
        const arn = attrs.graphqlApiArn ?? aws_cdk_lib_1.Stack.of(scope).formatArn({
            service: 'appsync',
            resource: `apis/${attrs.graphqlApiId}`,
        });
        class Import extends graphqlapi_base_1.GraphqlApiBase {
            constructor(s, i) {
                super(s, i);
                this.apiId = attrs.graphqlApiId;
                this.arn = arn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee, resources, ...actions) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_IamResource(resources);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grant);
            }
            throw error;
        }
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: resources.resourceArns(this),
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Mutation', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Query', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Subscription', ...fields), 'appsync:GraphQL');
    }
    validateAuthorizationProps(modes) {
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.LAMBDA).length > 1) {
            throw new Error('You can only have a single AWS Lambda function configured to authorize your API.');
        }
        modes.map((mode) => {
            if (mode.authorizationType === AuthorizationType.OIDC && !mode.openIdConnectConfig) {
                throw new Error('Missing OIDC Configuration');
            }
            if (mode.authorizationType === AuthorizationType.USER_POOL && !mode.userPoolConfig) {
                throw new Error('Missing User Pool Configuration');
            }
            if (mode.authorizationType === AuthorizationType.LAMBDA && !mode.lambdaAuthorizerConfig) {
                throw new Error('Missing Lambda Configuration');
            }
        });
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.API_KEY).length > 1) {
            throw new Error('You can\'t duplicate API_KEY configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.IAM).length > 1) {
            throw new Error('You can\'t duplicate IAM configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
    }
    /**
     * Add schema dependency to a given construct
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct) {
        construct.addDependsOn(this.schemaResource);
        return true;
    }
    setupLogConfig(config) {
        if (!config)
            return undefined;
        const logsRoleArn = config.role?.roleArn ?? new aws_iam_1.Role(this, 'ApiLogsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('appsync.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'),
            ],
        }).roleArn;
        return {
            cloudWatchLogsRoleArn: logsRoleArn,
            excludeVerboseContent: config.excludeVerboseContent,
            fieldLogLevel: config.fieldLogLevel,
        };
    }
    setupOpenIdConnectConfig(config) {
        if (!config)
            return undefined;
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    setupUserPoolConfig(config) {
        if (!config)
            return undefined;
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.env.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || UserPoolDefaultAction.ALLOW,
        };
    }
    setupLambdaAuthorizerConfig(config) {
        if (!config)
            return undefined;
        return {
            authorizerResultTtlInSeconds: config.resultsCacheTtl?.toSeconds(),
            authorizerUri: config.handler.functionArn,
            identityValidationExpression: config.validationRegex,
        };
    }
    setupAdditionalAuthorizationModes(modes) {
        if (!modes || modes.length === 0)
            return undefined;
        return modes.reduce((acc, mode) => [
            ...acc, {
                authenticationType: mode.authorizationType,
                userPoolConfig: this.setupUserPoolConfig(mode.userPoolConfig),
                openIdConnectConfig: this.setupOpenIdConnectConfig(mode.openIdConnectConfig),
                lambdaAuthorizerConfig: this.setupLambdaAuthorizerConfig(mode.lambdaAuthorizerConfig),
            },
        ], []);
    }
    createAPIKey(config) {
        if (config?.expires?.isBefore(aws_cdk_lib_1.Duration.days(1)) || config?.expires?.isAfter(aws_cdk_lib_1.Duration.days(365))) {
            throw Error('API key expiration must be between 1 and 365 days.');
        }
        const expires = config?.expires ? config?.expires.toEpoch() : undefined;
        return new aws_appsync_1.CfnApiKey(this, `${config?.name || 'Default'}ApiKey`, {
            expires,
            description: config?.description,
            apiId: this.apiId,
        });
    }
    /**
     * Escape hatch to append to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     */
    addToSchema(addition, delimiter) {
        this.schema.addToSchema(addition, delimiter);
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     */
    addType(type) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_IIntermediateType(type);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addType);
            }
            throw error;
        }
        return this.schema.addType(type);
    }
    /**
     * Add a query field to the schema's Query. CDK will create an
     * Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName, field) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_ResolvableField(field);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addQuery);
            }
            throw error;
        }
        return this.schema.addQuery(fieldName, field);
    }
    /**
     * Add a mutation field to the schema's Mutation. CDK will create an
     * Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName, field) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_ResolvableField(field);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addMutation);
            }
            throw error;
        }
        return this.schema.addMutation(fieldName, field);
    }
    /**
     * Add a subscription field to the schema's Subscription. CDK will create an
     * Object Type called 'Subscription'. For example,
     *
     * type Subscription {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription
     * @param field the resolvable field to for this Subscription
     */
    addSubscription(fieldName, field) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appsync_alpha_ResolvableField(field);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addSubscription);
            }
            throw error;
        }
        return this.schema.addSubscription(fieldName, field);
    }
}
exports.GraphqlApi = GraphqlApi;
_b = JSII_RTTI_SYMBOL_1;
GraphqlApi[_b] = { fqn: "@aws-cdk/aws-appsync-alpha.GraphqlApi", version: "2.28.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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