"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseQuery = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const customresources = require("@aws-cdk/custom-resources");
const cluster_1 = require("../cluster");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
class DatabaseQuery extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const adminUser = this.getAdminUser(props);
        const handler = new lambda.SingletonFunction(this, 'Handler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'database-query-provider'), {
                exclude: ['*.ts'],
            }),
            runtime: lambda.Runtime.NODEJS_14_X,
            handler: 'index.handler',
            timeout: cdk.Duration.minutes(1),
            uuid: '3de5bea7-27da-4796-8662-5efb56431b5f',
            lambdaPurpose: 'Query Redshift Database',
        });
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['redshift-data:DescribeStatement', 'redshift-data:ExecuteStatement'],
            resources: ['*'],
        }));
        adminUser.grantRead(handler);
        const provider = new customresources.Provider(this, 'Provider', {
            onEventHandler: handler,
        });
        const queryHandlerProps = {
            handler: props.handler,
            clusterName: props.cluster.clusterName,
            adminUserArn: adminUser.secretArn,
            databaseName: props.databaseName,
            ...props.properties,
        };
        this.resource = new cdk.CustomResource(this, 'Resource', {
            resourceType: 'Custom::RedshiftDatabaseQuery',
            serviceToken: provider.serviceToken,
            removalPolicy: props.removalPolicy,
            properties: queryHandlerProps,
        });
        this.grantPrincipal = handler.grantPrincipal;
        this.ref = this.resource.ref;
    }
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    getAtt(attributeName) {
        return this.resource.getAtt(attributeName);
    }
    getAttString(attributeName) {
        return this.resource.getAttString(attributeName);
    }
    getAdminUser(props) {
        const cluster = props.cluster;
        let adminUser = props.adminUser;
        if (!adminUser) {
            if (cluster instanceof cluster_1.Cluster) {
                if (cluster.secret) {
                    adminUser = cluster.secret;
                }
                else {
                    throw new Error('Administrative access to the Redshift cluster is required but an admin user secret was not provided and the cluster did not generate admin user credentials (they were provided explicitly)');
                }
            }
            else {
                throw new Error('Administrative access to the Redshift cluster is required but an admin user secret was not provided and the cluster was imported');
            }
        }
        return adminUser;
    }
}
exports.DatabaseQuery = DatabaseQuery;
//# sourceMappingURL=data:application/json;base64,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