"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationMethod = exports.Certificate = exports.CertificateValidation = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("@aws-cdk/core");
const certificate_base_1 = require("./certificate-base");
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * How to validate a certificate
 */
class CertificateValidation {
    /** @param props Certification validation properties */
    constructor(props) {
        this.props = props;
        this.method = props.method ?? ValidationMethod.EMAIL;
    }
    /**
     * Validate the certificate with DNS
     *
     * IMPORTANT: If `hostedZone` is not specified, DNS records must be added
     * manually and the stack will not complete creating until the records are
     * added.
     *
     * @param hostedZone the hosted zone where DNS records must be created
     */
    static fromDns(hostedZone) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZone,
        });
    }
    /**
     * Validate the certificate with automatically created DNS records in multiple
     * Amazon Route 53 hosted zones.
     *
     * @param hostedZones a map of hosted zones where DNS records must be created
     * for the domains in the certificate
     */
    static fromDnsMultiZone(hostedZones) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZones,
        });
    }
    /**
     * Validate the certificate with Email
     *
     * IMPORTANT: if you are creating a certificate as part of your stack, the stack
     * will not complete creating until you read and follow the instructions in the
     * email that you will receive.
     *
     * ACM will send validation emails to the following addresses:
     *
     *  admin@domain.com
     *  administrator@domain.com
     *  hostmaster@domain.com
     *  postmaster@domain.com
     *  webmaster@domain.com
     *
     * For every domain that you register.
     *
     * @param validationDomains a map of validation domains to use for domains in the certificate
     */
    static fromEmail(validationDomains) {
        return new CertificateValidation({
            method: ValidationMethod.EMAIL,
            validationDomains,
        });
    }
}
exports.CertificateValidation = CertificateValidation;
_a = JSII_RTTI_SYMBOL_1;
CertificateValidation[_a] = { fqn: "@aws-cdk/aws-certificatemanager.CertificateValidation", version: "1.181.0" };
/**
 * A certificate managed by AWS Certificate Manager
 */
class Certificate extends certificate_base_1.CertificateBase {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_certificatemanager_CertificateProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Certificate);
            }
            throw error;
        }
        let validation;
        if (props.validation) {
            validation = props.validation;
        }
        else { // Deprecated props
            if (props.validationMethod === ValidationMethod.DNS) {
                validation = CertificateValidation.fromDns();
            }
            else {
                validation = CertificateValidation.fromEmail(props.validationDomains);
            }
        }
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: renderDomainValidation(validation, allDomainNames),
            validationMethod: validation.method,
        });
        this.certificateArn = cert.ref;
    }
    /**
     * Import a certificate
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends certificate_base_1.CertificateBase {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
_b = JSII_RTTI_SYMBOL_1;
Certificate[_b] = { fqn: "@aws-cdk/aws-certificatemanager.Certificate", version: "1.181.0" };
/**
 * Method used to assert ownership of the domain
 */
var ValidationMethod;
(function (ValidationMethod) {
    /**
     * Send email to a number of email addresses associated with the domain
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-email.html
     */
    ValidationMethod["EMAIL"] = "EMAIL";
    /**
     * Validate ownership by adding appropriate DNS records
     *
     * @see https://docs.aws.amazon.com/acm/latest/userguide/gs-acm-validate-dns.html
     */
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
// eslint-disable-next-line max-len
function renderDomainValidation(validation, domainNames) {
    const domainValidation = [];
    switch (validation.method) {
        case ValidationMethod.DNS:
            for (const domainName of getUniqueDnsDomainNames(domainNames)) {
                const hostedZone = validation.props.hostedZones?.[domainName] ?? validation.props.hostedZone;
                if (hostedZone) {
                    domainValidation.push({ domainName, hostedZoneId: hostedZone.hostedZoneId });
                }
            }
            break;
        case ValidationMethod.EMAIL:
            for (const domainName of domainNames) {
                const validationDomain = validation.props.validationDomains?.[domainName];
                if (!validationDomain && core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                domainValidation.push({ domainName, validationDomain: validationDomain ?? util_1.apexDomain(domainName) });
            }
            break;
        default:
            throw new Error(`Unknown validation method ${validation.method}`);
    }
    return domainValidation.length !== 0 ? domainValidation : undefined;
}
/**
 * Removes wildcard domains (*.example.com) where the base domain (example.com) is present.
 * This is because the DNS validation treats them as the same thing, and the automated CloudFormation
 * DNS validation errors out with the duplicate records.
 */
function getUniqueDnsDomainNames(domainNames) {
    return domainNames.filter(domain => {
        return core_1.Token.isUnresolved(domain) || !domain.startsWith('*.') || !domainNames.includes(domain.replace('*.', ''));
    });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2VydGlmaWNhdGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjZXJ0aWZpY2F0ZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFFQSx3Q0FBaUQ7QUFFakQseURBQXFEO0FBQ3JELGlGQUFnRTtBQUNoRSxpQ0FBb0M7QUF1R3BDOztHQUVHO0FBQ0gsTUFBYSxxQkFBcUI7SUE4RGhDLHVEQUF1RDtJQUN2RCxZQUFvQyxLQUFtQztRQUFuQyxVQUFLLEdBQUwsS0FBSyxDQUE4QjtRQUNyRSxJQUFJLENBQUMsTUFBTSxHQUFHLEtBQUssQ0FBQyxNQUFNLElBQUksZ0JBQWdCLENBQUMsS0FBSyxDQUFDO0tBQ3REO0lBaEVEOzs7Ozs7OztPQVFHO0lBQ0ksTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFnQztRQUNwRCxPQUFPLElBQUkscUJBQXFCLENBQUM7WUFDL0IsTUFBTSxFQUFFLGdCQUFnQixDQUFDLEdBQUc7WUFDNUIsVUFBVTtTQUNYLENBQUMsQ0FBQztLQUNKO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLGdCQUFnQixDQUFDLFdBQTBEO1FBQ3ZGLE9BQU8sSUFBSSxxQkFBcUIsQ0FBQztZQUMvQixNQUFNLEVBQUUsZ0JBQWdCLENBQUMsR0FBRztZQUM1QixXQUFXO1NBQ1osQ0FBQyxDQUFDO0tBQ0o7SUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Ba0JHO0lBQ0ksTUFBTSxDQUFDLFNBQVMsQ0FBQyxpQkFBb0Q7UUFDMUUsT0FBTyxJQUFJLHFCQUFxQixDQUFDO1lBQy9CLE1BQU0sRUFBRSxnQkFBZ0IsQ0FBQyxLQUFLO1lBQzlCLGlCQUFpQjtTQUNsQixDQUFDLENBQUM7S0FDSjs7QUF2REgsc0RBa0VDOzs7QUFFRDs7R0FFRztBQUNILE1BQWEsV0FBWSxTQUFRLGtDQUFlO0lBaUI5QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXVCO1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7Ozs7OzsrQ0FsQlIsV0FBVzs7OztRQW9CcEIsSUFBSSxVQUFpQyxDQUFDO1FBQ3RDLElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRTtZQUNwQixVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztTQUMvQjthQUFNLEVBQUUsbUJBQW1CO1lBQzFCLElBQUksS0FBSyxDQUFDLGdCQUFnQixLQUFLLGdCQUFnQixDQUFDLEdBQUcsRUFBRTtnQkFDbkQsVUFBVSxHQUFHLHFCQUFxQixDQUFDLE9BQU8sRUFBRSxDQUFDO2FBQzlDO2lCQUFNO2dCQUNMLFVBQVUsR0FBRyxxQkFBcUIsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUM7YUFDdkU7U0FDRjtRQUVELE1BQU0sY0FBYyxHQUFHLENBQUMsS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsdUJBQXVCLElBQUksRUFBRSxDQUFDLENBQUM7UUFFdEYsTUFBTSxJQUFJLEdBQUcsSUFBSSw2Q0FBYyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDaEQsVUFBVSxFQUFFLEtBQUssQ0FBQyxVQUFVO1lBQzVCLHVCQUF1QixFQUFFLEtBQUssQ0FBQyx1QkFBdUI7WUFDdEQsdUJBQXVCLEVBQUUsc0JBQXNCLENBQUMsVUFBVSxFQUFFLGNBQWMsQ0FBQztZQUMzRSxnQkFBZ0IsRUFBRSxVQUFVLENBQUMsTUFBTTtTQUNwQyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksQ0FBQyxHQUFHLENBQUM7S0FDaEM7SUF4Q0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsa0JBQWtCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsY0FBc0I7UUFDbkYsTUFBTSxNQUFPLFNBQVEsa0NBQWU7WUFBcEM7O2dCQUNrQixtQkFBYyxHQUFHLGNBQWMsQ0FBQztZQUNsRCxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztLQUM5Qjs7QUFWSCxrQ0EwQ0M7OztBQUVEOztHQUVHO0FBQ0gsSUFBWSxnQkFjWDtBQWRELFdBQVksZ0JBQWdCO0lBQzFCOzs7O09BSUc7SUFDSCxtQ0FBZSxDQUFBO0lBRWY7Ozs7T0FJRztJQUNILCtCQUFXLENBQUE7QUFDYixDQUFDLEVBZFcsZ0JBQWdCLEdBQWhCLHdCQUFnQixLQUFoQix3QkFBZ0IsUUFjM0I7QUFFRCxtQ0FBbUM7QUFDbkMsU0FBUyxzQkFBc0IsQ0FBQyxVQUFpQyxFQUFFLFdBQXFCO0lBQ3RGLE1BQU0sZ0JBQWdCLEdBQW9ELEVBQUUsQ0FBQztJQUU3RSxRQUFRLFVBQVUsQ0FBQyxNQUFNLEVBQUU7UUFDekIsS0FBSyxnQkFBZ0IsQ0FBQyxHQUFHO1lBQ3ZCLEtBQUssTUFBTSxVQUFVLElBQUksdUJBQXVCLENBQUMsV0FBVyxDQUFDLEVBQUU7Z0JBQzdELE1BQU0sVUFBVSxHQUFHLFVBQVUsQ0FBQyxLQUFLLENBQUMsV0FBVyxFQUFFLENBQUMsVUFBVSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUM7Z0JBQzdGLElBQUksVUFBVSxFQUFFO29CQUNkLGdCQUFnQixDQUFDLElBQUksQ0FBQyxFQUFFLFVBQVUsRUFBRSxZQUFZLEVBQUUsVUFBVSxDQUFDLFlBQVksRUFBRSxDQUFDLENBQUM7aUJBQzlFO2FBQ0Y7WUFDRCxNQUFNO1FBQ1IsS0FBSyxnQkFBZ0IsQ0FBQyxLQUFLO1lBQ3pCLEtBQUssTUFBTSxVQUFVLElBQUksV0FBVyxFQUFFO2dCQUNwQyxNQUFNLGdCQUFnQixHQUFHLFVBQVUsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxVQUFVLENBQUMsQ0FBQztnQkFDMUUsSUFBSSxDQUFDLGdCQUFnQixJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLEVBQUU7b0JBQ3ZELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0ZBQWdGLENBQUMsQ0FBQztpQkFDbkc7Z0JBQ0QsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLEVBQUUsVUFBVSxFQUFFLGdCQUFnQixFQUFFLGdCQUFnQixJQUFJLGlCQUFVLENBQUMsVUFBVSxDQUFDLEVBQUUsQ0FBQyxDQUFDO2FBQ3JHO1lBQ0QsTUFBTTtRQUNSO1lBQ0UsTUFBTSxJQUFJLEtBQUssQ0FBQyw2QkFBNkIsVUFBVSxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7S0FDckU7SUFFRCxPQUFPLGdCQUFnQixDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7QUFDdEUsQ0FBQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFTLHVCQUF1QixDQUFDLFdBQXFCO0lBQ3BELE9BQU8sV0FBVyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsRUFBRTtRQUNqQyxPQUFPLFlBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ25ILENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNsb3Vkd2F0Y2ggZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3Vkd2F0Y2gnO1xuaW1wb3J0ICogYXMgcm91dGU1MyBmcm9tICdAYXdzLWNkay9hd3Mtcm91dGU1Myc7XG5pbXBvcnQgeyBJUmVzb3VyY2UsIFRva2VuIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IENlcnRpZmljYXRlQmFzZSB9IGZyb20gJy4vY2VydGlmaWNhdGUtYmFzZSc7XG5pbXBvcnQgeyBDZm5DZXJ0aWZpY2F0ZSB9IGZyb20gJy4vY2VydGlmaWNhdGVtYW5hZ2VyLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBhcGV4RG9tYWluIH0gZnJvbSAnLi91dGlsJztcblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgY2VydGlmaWNhdGUgaW4gQVdTIENlcnRpZmljYXRlIE1hbmFnZXJcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJQ2VydGlmaWNhdGUgZXh0ZW5kcyBJUmVzb3VyY2Uge1xuICAvKipcbiAgICogVGhlIGNlcnRpZmljYXRlJ3MgQVJOXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGNlcnRpZmljYXRlQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFJldHVybiB0aGUgRGF5c1RvRXhwaXJ5IG1ldHJpYyBmb3IgdGhpcyBBV1MgQ2VydGlmaWNhdGUgTWFuYWdlclxuICAgKiBDZXJ0aWZpY2F0ZS4gQnkgZGVmYXVsdCwgdGhpcyBpcyB0aGUgbWluaW11bSB2YWx1ZSBvdmVyIDEgZGF5LlxuICAgKlxuICAgKiBUaGlzIG1ldHJpYyBpcyBubyBsb25nZXIgZW1pdHRlZCBvbmNlIHRoZSBjZXJ0aWZpY2F0ZSBoYXMgZWZmZWN0aXZlbHlcbiAgICogZXhwaXJlZCwgc28gYWxhcm1zIGNvbmZpZ3VyZWQgb24gdGhpcyBtZXRyaWMgc2hvdWxkIHByb2JhYmx5IHRyZWF0IG1pc3NpbmdcbiAgICogZGF0YSBhcyBcImJyZWFjaGluZ1wiLlxuICAgKi9cbiAgbWV0cmljRGF5c1RvRXhwaXJ5KHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWM7XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgeW91ciBjZXJ0aWZpY2F0ZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIENlcnRpZmljYXRlUHJvcHMge1xuICAvKipcbiAgICogRnVsbHktcXVhbGlmaWVkIGRvbWFpbiBuYW1lIHRvIHJlcXVlc3QgYSBjZXJ0aWZpY2F0ZSBmb3IuXG4gICAqXG4gICAqIE1heSBjb250YWluIHdpbGRjYXJkcywgc3VjaCBhcyBgYCouZG9tYWluLmNvbWBgLlxuICAgKi9cbiAgcmVhZG9ubHkgZG9tYWluTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBBbHRlcm5hdGl2ZSBkb21haW4gbmFtZXMgb24geW91ciBjZXJ0aWZpY2F0ZS5cbiAgICpcbiAgICogVXNlIHRoaXMgdG8gcmVnaXN0ZXIgYWx0ZXJuYXRpdmUgZG9tYWluIG5hbWVzIHRoYXQgcmVwcmVzZW50IHRoZSBzYW1lIHNpdGUuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYWRkaXRpb25hbCBGUUROcyB3aWxsIGJlIGluY2x1ZGVkIGFzIGFsdGVybmF0aXZlIGRvbWFpbiBuYW1lcy5cbiAgICovXG4gIHJlYWRvbmx5IHN1YmplY3RBbHRlcm5hdGl2ZU5hbWVzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFdoYXQgdmFsaWRhdGlvbiBkb21haW4gdG8gdXNlIGZvciBldmVyeSByZXF1ZXN0ZWQgZG9tYWluLlxuICAgKlxuICAgKiBIYXMgdG8gYmUgYSBzdXBlcmRvbWFpbiBvZiB0aGUgcmVxdWVzdGVkIGRvbWFpbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBcGV4IGRvbWFpbiBpcyB1c2VkIGZvciBldmVyeSBkb21haW4gdGhhdCdzIG5vdCBvdmVycmlkZGVuLlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYHZhbGlkYXRpb25gIGluc3RlYWQuXG4gICAqL1xuICByZWFkb25seSB2YWxpZGF0aW9uRG9tYWlucz86IHtbZG9tYWluTmFtZTogc3RyaW5nXTogc3RyaW5nfTtcblxuICAvKipcbiAgICogVmFsaWRhdGlvbiBtZXRob2QgdXNlZCB0byBhc3NlcnQgZG9tYWluIG93bmVyc2hpcFxuICAgKlxuICAgKiBAZGVmYXVsdCBWYWxpZGF0aW9uTWV0aG9kLkVNQUlMXG4gICAqIEBkZXByZWNhdGVkIHVzZSBgdmFsaWRhdGlvbmAgaW5zdGVhZC5cbiAgICovXG4gIHJlYWRvbmx5IHZhbGlkYXRpb25NZXRob2Q/OiBWYWxpZGF0aW9uTWV0aG9kO1xuXG4gIC8qKlxuICAgKiBIb3cgdG8gdmFsaWRhdGUgdGhpcyBjZXJ0aWZpY2F0ZVxuICAgKlxuICAgKiBAZGVmYXVsdCBDZXJ0aWZpY2F0ZVZhbGlkYXRpb24uZnJvbUVtYWlsKClcbiAgICovXG4gIHJlYWRvbmx5IHZhbGlkYXRpb24/OiBDZXJ0aWZpY2F0ZVZhbGlkYXRpb247XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgY2VydGlmaWNhdGUgdmFsaWRhdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIENlcnRpZmljYXRpb25WYWxpZGF0aW9uUHJvcHMge1xuICAvKipcbiAgICogVmFsaWRhdGlvbiBtZXRob2RcbiAgICpcbiAgICogQGRlZmF1bHQgVmFsaWRhdGlvbk1ldGhvZC5FTUFJTFxuICAgKi9cbiAgcmVhZG9ubHkgbWV0aG9kPzogVmFsaWRhdGlvbk1ldGhvZDtcblxuICAvKipcbiAgICogSG9zdGVkIHpvbmUgdG8gdXNlIGZvciBETlMgdmFsaWRhdGlvblxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHVzZSBlbWFpbCB2YWxpZGF0aW9uXG4gICAqL1xuICByZWFkb25seSBob3N0ZWRab25lPzogcm91dGU1My5JSG9zdGVkWm9uZTtcblxuICAvKipcbiAgICogQSBtYXAgb2YgaG9zdGVkIHpvbmVzIHRvIHVzZSBmb3IgRE5TIHZhbGlkYXRpb25cbiAgICpcbiAgICogQGRlZmF1bHQgLSB1c2UgYGhvc3RlZFpvbmVgXG4gICAqL1xuICByZWFkb25seSBob3N0ZWRab25lcz86IHsgW2RvbWFpbk5hbWU6IHN0cmluZ106IHJvdXRlNTMuSUhvc3RlZFpvbmUgfTtcblxuICAvKipcbiAgICogVmFsaWRhdGlvbiBkb21haW5zIHRvIHVzZSBmb3IgZW1haWwgdmFsaWRhdGlvblxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFwZXggZG9tYWluXG4gICAqL1xuICByZWFkb25seSB2YWxpZGF0aW9uRG9tYWlucz86IHsgW2RvbWFpbk5hbWU6IHN0cmluZ106IHN0cmluZyB9O1xufVxuXG4vKipcbiAqIEhvdyB0byB2YWxpZGF0ZSBhIGNlcnRpZmljYXRlXG4gKi9cbmV4cG9ydCBjbGFzcyBDZXJ0aWZpY2F0ZVZhbGlkYXRpb24ge1xuICAvKipcbiAgICogVmFsaWRhdGUgdGhlIGNlcnRpZmljYXRlIHdpdGggRE5TXG4gICAqXG4gICAqIElNUE9SVEFOVDogSWYgYGhvc3RlZFpvbmVgIGlzIG5vdCBzcGVjaWZpZWQsIEROUyByZWNvcmRzIG11c3QgYmUgYWRkZWRcbiAgICogbWFudWFsbHkgYW5kIHRoZSBzdGFjayB3aWxsIG5vdCBjb21wbGV0ZSBjcmVhdGluZyB1bnRpbCB0aGUgcmVjb3JkcyBhcmVcbiAgICogYWRkZWQuXG4gICAqXG4gICAqIEBwYXJhbSBob3N0ZWRab25lIHRoZSBob3N0ZWQgem9uZSB3aGVyZSBETlMgcmVjb3JkcyBtdXN0IGJlIGNyZWF0ZWRcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbURucyhob3N0ZWRab25lPzogcm91dGU1My5JSG9zdGVkWm9uZSkge1xuICAgIHJldHVybiBuZXcgQ2VydGlmaWNhdGVWYWxpZGF0aW9uKHtcbiAgICAgIG1ldGhvZDogVmFsaWRhdGlvbk1ldGhvZC5ETlMsXG4gICAgICBob3N0ZWRab25lLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFZhbGlkYXRlIHRoZSBjZXJ0aWZpY2F0ZSB3aXRoIGF1dG9tYXRpY2FsbHkgY3JlYXRlZCBETlMgcmVjb3JkcyBpbiBtdWx0aXBsZVxuICAgKiBBbWF6b24gUm91dGUgNTMgaG9zdGVkIHpvbmVzLlxuICAgKlxuICAgKiBAcGFyYW0gaG9zdGVkWm9uZXMgYSBtYXAgb2YgaG9zdGVkIHpvbmVzIHdoZXJlIEROUyByZWNvcmRzIG11c3QgYmUgY3JlYXRlZFxuICAgKiBmb3IgdGhlIGRvbWFpbnMgaW4gdGhlIGNlcnRpZmljYXRlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21EbnNNdWx0aVpvbmUoaG9zdGVkWm9uZXM6IHsgW2RvbWFpbk5hbWU6IHN0cmluZ106IHJvdXRlNTMuSUhvc3RlZFpvbmUgfSkge1xuICAgIHJldHVybiBuZXcgQ2VydGlmaWNhdGVWYWxpZGF0aW9uKHtcbiAgICAgIG1ldGhvZDogVmFsaWRhdGlvbk1ldGhvZC5ETlMsXG4gICAgICBob3N0ZWRab25lcyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBWYWxpZGF0ZSB0aGUgY2VydGlmaWNhdGUgd2l0aCBFbWFpbFxuICAgKlxuICAgKiBJTVBPUlRBTlQ6IGlmIHlvdSBhcmUgY3JlYXRpbmcgYSBjZXJ0aWZpY2F0ZSBhcyBwYXJ0IG9mIHlvdXIgc3RhY2ssIHRoZSBzdGFja1xuICAgKiB3aWxsIG5vdCBjb21wbGV0ZSBjcmVhdGluZyB1bnRpbCB5b3UgcmVhZCBhbmQgZm9sbG93IHRoZSBpbnN0cnVjdGlvbnMgaW4gdGhlXG4gICAqIGVtYWlsIHRoYXQgeW91IHdpbGwgcmVjZWl2ZS5cbiAgICpcbiAgICogQUNNIHdpbGwgc2VuZCB2YWxpZGF0aW9uIGVtYWlscyB0byB0aGUgZm9sbG93aW5nIGFkZHJlc3NlczpcbiAgICpcbiAgICogIGFkbWluQGRvbWFpbi5jb21cbiAgICogIGFkbWluaXN0cmF0b3JAZG9tYWluLmNvbVxuICAgKiAgaG9zdG1hc3RlckBkb21haW4uY29tXG4gICAqICBwb3N0bWFzdGVyQGRvbWFpbi5jb21cbiAgICogIHdlYm1hc3RlckBkb21haW4uY29tXG4gICAqXG4gICAqIEZvciBldmVyeSBkb21haW4gdGhhdCB5b3UgcmVnaXN0ZXIuXG4gICAqXG4gICAqIEBwYXJhbSB2YWxpZGF0aW9uRG9tYWlucyBhIG1hcCBvZiB2YWxpZGF0aW9uIGRvbWFpbnMgdG8gdXNlIGZvciBkb21haW5zIGluIHRoZSBjZXJ0aWZpY2F0ZVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tRW1haWwodmFsaWRhdGlvbkRvbWFpbnM/OiB7IFtkb21haW5OYW1lOiBzdHJpbmddOiBzdHJpbmcgfSkge1xuICAgIHJldHVybiBuZXcgQ2VydGlmaWNhdGVWYWxpZGF0aW9uKHtcbiAgICAgIG1ldGhvZDogVmFsaWRhdGlvbk1ldGhvZC5FTUFJTCxcbiAgICAgIHZhbGlkYXRpb25Eb21haW5zLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSB2YWxpZGF0aW9uIG1ldGhvZFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IG1ldGhvZDogVmFsaWRhdGlvbk1ldGhvZDtcblxuICAvKiogQHBhcmFtIHByb3BzIENlcnRpZmljYXRpb24gdmFsaWRhdGlvbiBwcm9wZXJ0aWVzICovXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IHByb3BzOiBDZXJ0aWZpY2F0aW9uVmFsaWRhdGlvblByb3BzKSB7XG4gICAgdGhpcy5tZXRob2QgPSBwcm9wcy5tZXRob2QgPz8gVmFsaWRhdGlvbk1ldGhvZC5FTUFJTDtcbiAgfVxufVxuXG4vKipcbiAqIEEgY2VydGlmaWNhdGUgbWFuYWdlZCBieSBBV1MgQ2VydGlmaWNhdGUgTWFuYWdlclxuICovXG5leHBvcnQgY2xhc3MgQ2VydGlmaWNhdGUgZXh0ZW5kcyBDZXJ0aWZpY2F0ZUJhc2UgaW1wbGVtZW50cyBJQ2VydGlmaWNhdGUge1xuICAvKipcbiAgICogSW1wb3J0IGEgY2VydGlmaWNhdGVcbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUNlcnRpZmljYXRlQXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGNlcnRpZmljYXRlQXJuOiBzdHJpbmcpOiBJQ2VydGlmaWNhdGUge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIENlcnRpZmljYXRlQmFzZSB7XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgY2VydGlmaWNhdGVBcm4gPSBjZXJ0aWZpY2F0ZUFybjtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBjZXJ0aWZpY2F0ZSdzIEFSTlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGNlcnRpZmljYXRlQXJuOiBzdHJpbmc7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENlcnRpZmljYXRlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgbGV0IHZhbGlkYXRpb246IENlcnRpZmljYXRlVmFsaWRhdGlvbjtcbiAgICBpZiAocHJvcHMudmFsaWRhdGlvbikge1xuICAgICAgdmFsaWRhdGlvbiA9IHByb3BzLnZhbGlkYXRpb247XG4gICAgfSBlbHNlIHsgLy8gRGVwcmVjYXRlZCBwcm9wc1xuICAgICAgaWYgKHByb3BzLnZhbGlkYXRpb25NZXRob2QgPT09IFZhbGlkYXRpb25NZXRob2QuRE5TKSB7XG4gICAgICAgIHZhbGlkYXRpb24gPSBDZXJ0aWZpY2F0ZVZhbGlkYXRpb24uZnJvbURucygpO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdmFsaWRhdGlvbiA9IENlcnRpZmljYXRlVmFsaWRhdGlvbi5mcm9tRW1haWwocHJvcHMudmFsaWRhdGlvbkRvbWFpbnMpO1xuICAgICAgfVxuICAgIH1cblxuICAgIGNvbnN0IGFsbERvbWFpbk5hbWVzID0gW3Byb3BzLmRvbWFpbk5hbWVdLmNvbmNhdChwcm9wcy5zdWJqZWN0QWx0ZXJuYXRpdmVOYW1lcyB8fCBbXSk7XG5cbiAgICBjb25zdCBjZXJ0ID0gbmV3IENmbkNlcnRpZmljYXRlKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGRvbWFpbk5hbWU6IHByb3BzLmRvbWFpbk5hbWUsXG4gICAgICBzdWJqZWN0QWx0ZXJuYXRpdmVOYW1lczogcHJvcHMuc3ViamVjdEFsdGVybmF0aXZlTmFtZXMsXG4gICAgICBkb21haW5WYWxpZGF0aW9uT3B0aW9uczogcmVuZGVyRG9tYWluVmFsaWRhdGlvbih2YWxpZGF0aW9uLCBhbGxEb21haW5OYW1lcyksXG4gICAgICB2YWxpZGF0aW9uTWV0aG9kOiB2YWxpZGF0aW9uLm1ldGhvZCxcbiAgICB9KTtcblxuICAgIHRoaXMuY2VydGlmaWNhdGVBcm4gPSBjZXJ0LnJlZjtcbiAgfVxufVxuXG4vKipcbiAqIE1ldGhvZCB1c2VkIHRvIGFzc2VydCBvd25lcnNoaXAgb2YgdGhlIGRvbWFpblxuICovXG5leHBvcnQgZW51bSBWYWxpZGF0aW9uTWV0aG9kIHtcbiAgLyoqXG4gICAqIFNlbmQgZW1haWwgdG8gYSBudW1iZXIgb2YgZW1haWwgYWRkcmVzc2VzIGFzc29jaWF0ZWQgd2l0aCB0aGUgZG9tYWluXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2FjbS9sYXRlc3QvdXNlcmd1aWRlL2dzLWFjbS12YWxpZGF0ZS1lbWFpbC5odG1sXG4gICAqL1xuICBFTUFJTCA9ICdFTUFJTCcsXG5cbiAgLyoqXG4gICAqIFZhbGlkYXRlIG93bmVyc2hpcCBieSBhZGRpbmcgYXBwcm9wcmlhdGUgRE5TIHJlY29yZHNcbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYWNtL2xhdGVzdC91c2VyZ3VpZGUvZ3MtYWNtLXZhbGlkYXRlLWRucy5odG1sXG4gICAqL1xuICBETlMgPSAnRE5TJyxcbn1cblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG1heC1sZW5cbmZ1bmN0aW9uIHJlbmRlckRvbWFpblZhbGlkYXRpb24odmFsaWRhdGlvbjogQ2VydGlmaWNhdGVWYWxpZGF0aW9uLCBkb21haW5OYW1lczogc3RyaW5nW10pOiBDZm5DZXJ0aWZpY2F0ZS5Eb21haW5WYWxpZGF0aW9uT3B0aW9uUHJvcGVydHlbXSB8IHVuZGVmaW5lZCB7XG4gIGNvbnN0IGRvbWFpblZhbGlkYXRpb246IENmbkNlcnRpZmljYXRlLkRvbWFpblZhbGlkYXRpb25PcHRpb25Qcm9wZXJ0eVtdID0gW107XG5cbiAgc3dpdGNoICh2YWxpZGF0aW9uLm1ldGhvZCkge1xuICAgIGNhc2UgVmFsaWRhdGlvbk1ldGhvZC5ETlM6XG4gICAgICBmb3IgKGNvbnN0IGRvbWFpbk5hbWUgb2YgZ2V0VW5pcXVlRG5zRG9tYWluTmFtZXMoZG9tYWluTmFtZXMpKSB7XG4gICAgICAgIGNvbnN0IGhvc3RlZFpvbmUgPSB2YWxpZGF0aW9uLnByb3BzLmhvc3RlZFpvbmVzPy5bZG9tYWluTmFtZV0gPz8gdmFsaWRhdGlvbi5wcm9wcy5ob3N0ZWRab25lO1xuICAgICAgICBpZiAoaG9zdGVkWm9uZSkge1xuICAgICAgICAgIGRvbWFpblZhbGlkYXRpb24ucHVzaCh7IGRvbWFpbk5hbWUsIGhvc3RlZFpvbmVJZDogaG9zdGVkWm9uZS5ob3N0ZWRab25lSWQgfSk7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIGJyZWFrO1xuICAgIGNhc2UgVmFsaWRhdGlvbk1ldGhvZC5FTUFJTDpcbiAgICAgIGZvciAoY29uc3QgZG9tYWluTmFtZSBvZiBkb21haW5OYW1lcykge1xuICAgICAgICBjb25zdCB2YWxpZGF0aW9uRG9tYWluID0gdmFsaWRhdGlvbi5wcm9wcy52YWxpZGF0aW9uRG9tYWlucz8uW2RvbWFpbk5hbWVdO1xuICAgICAgICBpZiAoIXZhbGlkYXRpb25Eb21haW4gJiYgVG9rZW4uaXNVbnJlc29sdmVkKGRvbWFpbk5hbWUpKSB7XG4gICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdXaGVuIHVzaW5nIFRva2VucyBmb3IgZG9tYWluIG5hbWVzLCBcXCd2YWxpZGF0aW9uRG9tYWluc1xcJyBuZWVkcyB0byBiZSBzdXBwbGllZCcpO1xuICAgICAgICB9XG4gICAgICAgIGRvbWFpblZhbGlkYXRpb24ucHVzaCh7IGRvbWFpbk5hbWUsIHZhbGlkYXRpb25Eb21haW46IHZhbGlkYXRpb25Eb21haW4gPz8gYXBleERvbWFpbihkb21haW5OYW1lKSB9KTtcbiAgICAgIH1cbiAgICAgIGJyZWFrO1xuICAgIGRlZmF1bHQ6XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYFVua25vd24gdmFsaWRhdGlvbiBtZXRob2QgJHt2YWxpZGF0aW9uLm1ldGhvZH1gKTtcbiAgfVxuXG4gIHJldHVybiBkb21haW5WYWxpZGF0aW9uLmxlbmd0aCAhPT0gMCA/IGRvbWFpblZhbGlkYXRpb24gOiB1bmRlZmluZWQ7XG59XG5cbi8qKlxuICogUmVtb3ZlcyB3aWxkY2FyZCBkb21haW5zICgqLmV4YW1wbGUuY29tKSB3aGVyZSB0aGUgYmFzZSBkb21haW4gKGV4YW1wbGUuY29tKSBpcyBwcmVzZW50LlxuICogVGhpcyBpcyBiZWNhdXNlIHRoZSBETlMgdmFsaWRhdGlvbiB0cmVhdHMgdGhlbSBhcyB0aGUgc2FtZSB0aGluZywgYW5kIHRoZSBhdXRvbWF0ZWQgQ2xvdWRGb3JtYXRpb25cbiAqIEROUyB2YWxpZGF0aW9uIGVycm9ycyBvdXQgd2l0aCB0aGUgZHVwbGljYXRlIHJlY29yZHMuXG4gKi9cbmZ1bmN0aW9uIGdldFVuaXF1ZURuc0RvbWFpbk5hbWVzKGRvbWFpbk5hbWVzOiBzdHJpbmdbXSkge1xuICByZXR1cm4gZG9tYWluTmFtZXMuZmlsdGVyKGRvbWFpbiA9PiB7XG4gICAgcmV0dXJuIFRva2VuLmlzVW5yZXNvbHZlZChkb21haW4pIHx8ICFkb21haW4uc3RhcnRzV2l0aCgnKi4nKSB8fCAhZG9tYWluTmFtZXMuaW5jbHVkZXMoZG9tYWluLnJlcGxhY2UoJyouJywgJycpKTtcbiAgfSk7XG59XG4iXX0=