'''
  apkg
  ~~~~

  A package manager for Agda.

'''

# ----------------------------------------------------------------------------

import click
from os                 import access, W_OK

from pony.orm           import db_session, select

from ..__version__       import __version__
from .init               import init
from ..config            import REPO, AGDA_DEFAULTS_PATH, AGDA_LIBRARIES_PATH
from ..service.logging   import logger, clog
from ..service.database  import Library , LibraryVersion

# ----------------------------------------------------------------------------

@db_session
def getLibraries():
  return select(l for l in LibraryVersion if l.installed)[:]

@db_session
def getDefaultLibraries():
  return select(l for l in Library if l.installed and l.default)[:]

@click.group()
def write_defaults(): pass

@write_defaults.command()
@click.option('--yes'
             , type=bool
             , is_flag=True 
             , help='Yes for everything.')
@clog.simple_verbosity_option(logger)
@click.pass_context
def write_defaults(ctx, yes):
  """Create/update 'defaults' and 'libraries' files for Agda."""
  
  no_write_permission_msg = '[!] The current user has no permission to modify:'
  confirm_overwrite_msg   = 'already exists\nDo you want to overwrite it?'
  header = "-- File generated by Agda-Pkg v{}\n".format(__version__)
  overwrite_confirm = lambda f : click.confirm("[!] {} {}".format(f.as_posix(),confirm_overwrite_msg))
  
  write_lpath,  write_dpath = True, True
  lpath = AGDA_LIBRARIES_PATH 
  dpath = AGDA_DEFAULTS_PATH
  
  if not yes and (lpath.exists() and not overwrite_confirm(lpath)):
    write_lpath = False
  lpath.touch(exist_ok=True)
  
  if write_lpath and access(lpath, W_OK):
    try:
      lpath.write_text(header + \
        '\n'.join([v.agdaLibFilePath.as_posix() for v in getLibraries()]) + '\n')
      click.echo("Updated ({})".format(lpath.as_posix()))
    except Exception as e:
      logger.error(e)
  elif write_lpath:
    click.echo(no_write_permission_msg)
    click.echo("     {}".format(lpath.as_posix()))
      
  if not yes and (dpath.exists() and not overwrite_confirm(dpath)):
    write_dpath = False
  dpath.touch(exist_ok=True)
  if write_dpath and access(dpath, W_OK):
    try:
      dpath.write_text(header + \
        '\n'.join(lib.name for lib in getDefaultLibraries()) + '\n')
      click.echo("Updated ({})".format(dpath.as_posix()))
    except Exception as e:
      logger.error(e)
  elif write_dpath:
    click.echo(no_write_permission_msg)
    click.echo("     {}".format(dpath.as_posix()))