"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReplaceKey = exports.BucketAccessControl = exports.EventType = exports.BucketEncryption = exports.Bucket = exports.ObjectOwnership = exports.InventoryObjectVersion = exports.InventoryFrequency = exports.InventoryFormat = exports.RedirectProtocol = exports.HttpMethods = exports.BlockPublicAccess = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const path = require("path");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const bucket_policy_1 = require("./bucket-policy");
const notifications_resource_1 = require("./notifications-resource");
const perms = require("./perms");
const s3_generated_1 = require("./s3.generated");
const util_1 = require("./util");
const AUTO_DELETE_OBJECTS_RESOURCE_TYPE = 'Custom::S3AutoDeleteObjects';
/**
 * Represents an S3 Bucket.
 *
 * Buckets can be either defined within this stack:
 *
 *   new Bucket(this, 'MyBucket', { props });
 *
 * Or imported from an existing bucket:
 *
 *   Bucket.import(this, 'MyImportedBucket', { bucketArn: ... });
 *
 * You can also export a bucket and import it into another stack:
 *
 *   const ref = myBucket.export();
 *   Bucket.import(this, 'MyImportedBucket', ref);
 *
 */
class BucketBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Indicates if a bucket resource policy should automatically created upon
         * the first call to `addToResourcePolicy`.
         */
        this.autoCreatePolicy = false;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailEvent(id, options = {}) {
        var _d, _e;
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.s3'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                resources: {
                    ARN: (_e = (_d = options.paths) === null || _d === void 0 ? void 0 : _d.map(p => this.arnForObjects(p))) !== null && _e !== void 0 ? _e : [this.bucketArn],
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event that triggers when an object is uploaded to the specified paths (keys) in this bucket using the PutObject API call.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using `onCloudTrailWriteObject` may be preferable.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailPutObject(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutObject'],
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event that triggers when an object at the specified paths (keys) in this bucket are written to.
     *
     * This includes
     * the events PutObject, CopyObject, and CompleteMultipartUpload.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using this method may be preferable to `onCloudTrailPutObject`.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailWriteObject(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: [
                    'CompleteMultipartUpload',
                    'CopyObject',
                    'PutObject',
                ],
                requestParameters: {
                    bucketName: [this.bucketName],
                    key: options.paths,
                },
            },
        });
        return rule;
    }
    /**
     * Adds a statement to the resource policy for a principal (i.e. account/role/service) to perform actions on this bucket and/or it's contents. Use `bucketArn` and `arnForObjects(keys)` to obtain ARNs for this bucket or objects.
     *
     * @stability stable
     */
    addToResourcePolicy(permission) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new bucket_policy_1.BucketPolicy(this, 'Policy', { bucket: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(permission);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        var _d;
        const errors = super.validate();
        errors.push(...((_d = this.policy) === null || _d === void 0 ? void 0 : _d.document.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * The https URL of an S3 object.
     *
     * Specify `regional: false` at the options
     * for non-regional URLs. For example:
     *
     * @param key The S3 key of the object.
     * @returns an ObjectS3Url token
     * @stability stable
     * @example
     *
     * https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey
     */
    urlForObject(key) {
        const stack = core_1.Stack.of(this);
        const prefix = `https://s3.${stack.region}.${stack.urlSuffix}/`;
        if (typeof key !== 'string') {
            return this.urlJoin(prefix, this.bucketName);
        }
        return this.urlJoin(prefix, this.bucketName, key);
    }
    /**
     * The virtual hosted-style URL of an S3 object.
     *
     * Specify `regional: false` at
     * the options for non-regional URL. For example:
     *
     * @param key The S3 key of the object.
     * @param options Options for generating URL.
     * @returns an ObjectS3Url token
     * @stability stable
     * @example
     *
     * https://china-bucket.s3.cn-north-1.amazonaws.com.cn/mykey
     */
    virtualHostedUrlForObject(key, options) {
        var _d;
        const domainName = ((_d = options === null || options === void 0 ? void 0 : options.regional) !== null && _d !== void 0 ? _d : true) ? this.bucketRegionalDomainName : this.bucketDomainName;
        const prefix = `https://${domainName}`;
        if (typeof key !== 'string') {
            return prefix;
        }
        return this.urlJoin(prefix, key);
    }
    /**
     * The S3 URL of an S3 object.
     *
     * For example:
     *
     * @param key The S3 key of the object.
     * @returns an ObjectS3Url token
     * @stability stable
     * @example
     *
     * s3://bucket/key
     */
    s3UrlForObject(key) {
        const prefix = 's3://';
        if (typeof key !== 'string') {
            return this.urlJoin(prefix, this.bucketName);
        }
        return this.urlJoin(prefix, this.bucketName, key);
    }
    /**
     * Returns an ARN that represents all objects within the bucket that match the key pattern specified.
     *
     * To represent all keys, specify ``"*"``.
     *
     * If you need to specify a keyPattern with multiple components, concatenate them into a single string, e.g.:
     *
     *    arnForObjects(`home/${team}/${user}/*`)
     *
     * @stability stable
     */
    arnForObjects(keyPattern) {
        return `${this.bucketArn}/${keyPattern}`;
    }
    /**
     * Grant read permissions for this bucket and it's contents to an IAM principal (Role/Group/User).
     *
     * If encryption is used, permission to use the key to decrypt the contents
     * of the bucket will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantRead(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_READ_ACTIONS, perms.KEY_READ_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grant write permissions to this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * Before CDK version 1.85.0, this method granted the `s3:PutObject*` permission that included `s3:PutObjectAcl`,
     * which could be used to grant read/write object access to IAM principals in other accounts.
     * If you want to get rid of that behavior, update your CDK version to 1.85.0 or later,
     * and make sure the `@aws-cdk/aws-s3:grantWriteWithoutAcl` feature flag is set to `true`
     * in the `context` key of your cdk.json file.
     * If you've already updated, but still need the principal to have permissions to modify the ACLs,
     * use the {@link grantPutAcl} method.
     *
     * @stability stable
     */
    grantWrite(identity, objectsKeyPattern = '*') {
        return this.grant(identity, this.writeActions, perms.KEY_WRITE_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:PutObject* and s3:Abort* permissions for this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantPut(identity, objectsKeyPattern = '*') {
        return this.grant(identity, this.putActions, perms.KEY_WRITE_ACTIONS, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grant the given IAM identity permissions to modify the ACLs of objects in the given Bucket.
     *
     * If your application has the '@aws-cdk/aws-s3:grantWriteWithoutAcl' feature flag set,
     * calling {@link grantWrite} or {@link grantReadWrite} no longer grants permissions to modify the ACLs of the objects;
     * in this case, if you need to modify object ACLs, call this method explicitly.
     *
     * @stability stable
     */
    grantPutAcl(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_PUT_ACL_ACTIONS, [], this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:DeleteObject* permission to an IAM pricipal for objects in this bucket.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantDelete(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_DELETE_ACTIONS, [], this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants read/write permissions for this bucket and it's contents to an IAM principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     *
     * Before CDK version 1.85.0, this method granted the `s3:PutObject*` permission that included `s3:PutObjectAcl`,
     * which could be used to grant read/write object access to IAM principals in other accounts.
     * If you want to get rid of that behavior, update your CDK version to 1.85.0 or later,
     * and make sure the `@aws-cdk/aws-s3:grantWriteWithoutAcl` feature flag is set to `true`
     * in the `context` key of your cdk.json file.
     * If you've already updated, but still need the principal to have permissions to modify the ACLs,
     * use the {@link grantPutAcl} method.
     *
     * @stability stable
     */
    grantReadWrite(identity, objectsKeyPattern = '*') {
        const bucketActions = perms.BUCKET_READ_ACTIONS.concat(this.writeActions);
        // we need unique permissions because some permissions are common between read and write key actions
        const keyActions = [...new Set([...perms.KEY_READ_ACTIONS, ...perms.KEY_WRITE_ACTIONS])];
        return this.grant(identity, bucketActions, keyActions, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Allows unrestricted access to objects from this bucket.
     *
     * IMPORTANT: This permission allows anyone to perform actions on S3 objects
     * in this bucket, which is useful for when you configure your bucket as a
     * website and want everyone to be able to read objects in the bucket without
     * needing to authenticate.
     *
     * Without arguments, this method will grant read ("s3:GetObject") access to
     * all objects ("*") in the bucket.
     *
     * The method returns the `iam.Grant` object, which can then be modified
     * as needed. For example, you can add a condition that will restrict access only
     * to an IPv4 range like this:
     *
     *      const grant = bucket.grantPublicAccess();
     *      grant.resourceStatement!.addCondition(‘IpAddress’, { “aws:SourceIp”: “54.240.143.0/24” });
     *
     * @param keyPrefix the prefix of S3 object keys (e.g. `home/*`). Default is "*".
     * @param allowedActions the set of S3 actions to allow.
     * @stability stable
     */
    grantPublicAccess(keyPrefix = '*', ...allowedActions) {
        if (this.disallowPublicAccess) {
            throw new Error("Cannot grant public access when 'blockPublicPolicy' is enabled");
        }
        allowedActions = allowedActions.length > 0 ? allowedActions : ['s3:GetObject'];
        return iam.Grant.addToPrincipalOrResource({
            actions: allowedActions,
            resourceArns: [this.arnForObjects(keyPrefix)],
            grantee: new iam.Anyone(),
            resource: this,
        });
    }
    get writeActions() {
        return [
            ...perms.BUCKET_DELETE_ACTIONS,
            ...this.putActions,
        ];
    }
    get putActions() {
        return core_1.FeatureFlags.of(this).isEnabled(cxapi.S3_GRANT_WRITE_WITHOUT_ACL)
            ? perms.BUCKET_PUT_ACTIONS
            : perms.LEGACY_BUCKET_PUT_ACTIONS;
    }
    urlJoin(...components) {
        return components.reduce((result, component) => {
            if (result.endsWith('/')) {
                result = result.slice(0, -1);
            }
            if (component.startsWith('/')) {
                component = component.slice(1);
            }
            return `${result}/${component}`;
        });
    }
    grant(grantee, bucketActions, keyActions, resourceArn, ...otherResourceArns) {
        const resources = [resourceArn, ...otherResourceArns];
        const ret = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: bucketActions,
            resourceArns: resources,
            resource: this,
        });
        if (this.encryptionKey && keyActions && keyActions.length !== 0) {
            this.encryptionKey.grant(grantee, ...keyActions);
        }
        return ret;
    }
}
/**
 * @stability stable
 */
class BlockPublicAccess {
    /**
     * @stability stable
     */
    constructor(options) {
        this.blockPublicAcls = options.blockPublicAcls;
        this.blockPublicPolicy = options.blockPublicPolicy;
        this.ignorePublicAcls = options.ignorePublicAcls;
        this.restrictPublicBuckets = options.restrictPublicBuckets;
    }
}
exports.BlockPublicAccess = BlockPublicAccess;
_a = JSII_RTTI_SYMBOL_1;
BlockPublicAccess[_a] = { fqn: "@aws-cdk/aws-s3.BlockPublicAccess", version: "1.93.0" };
/**
 * @stability stable
 */
BlockPublicAccess.BLOCK_ALL = new BlockPublicAccess({
    blockPublicAcls: true,
    blockPublicPolicy: true,
    ignorePublicAcls: true,
    restrictPublicBuckets: true,
});
/**
 * @stability stable
 */
BlockPublicAccess.BLOCK_ACLS = new BlockPublicAccess({
    blockPublicAcls: true,
    ignorePublicAcls: true,
});
/**
 * All http request methods.
 *
 * @stability stable
 */
var HttpMethods;
(function (HttpMethods) {
    HttpMethods["GET"] = "GET";
    HttpMethods["PUT"] = "PUT";
    HttpMethods["HEAD"] = "HEAD";
    HttpMethods["POST"] = "POST";
    HttpMethods["DELETE"] = "DELETE";
})(HttpMethods = exports.HttpMethods || (exports.HttpMethods = {}));
/**
 * All http request methods.
 *
 * @stability stable
 */
var RedirectProtocol;
(function (RedirectProtocol) {
    RedirectProtocol["HTTP"] = "http";
    RedirectProtocol["HTTPS"] = "https";
})(RedirectProtocol = exports.RedirectProtocol || (exports.RedirectProtocol = {}));
/**
 * All supported inventory list formats.
 *
 * @stability stable
 */
var InventoryFormat;
(function (InventoryFormat) {
    InventoryFormat["CSV"] = "CSV";
    InventoryFormat["PARQUET"] = "Parquet";
    InventoryFormat["ORC"] = "ORC";
})(InventoryFormat = exports.InventoryFormat || (exports.InventoryFormat = {}));
/**
 * All supported inventory frequencies.
 *
 * @stability stable
 */
var InventoryFrequency;
(function (InventoryFrequency) {
    InventoryFrequency["DAILY"] = "Daily";
    InventoryFrequency["WEEKLY"] = "Weekly";
})(InventoryFrequency = exports.InventoryFrequency || (exports.InventoryFrequency = {}));
/**
 * Inventory version support.
 *
 * @stability stable
 */
var InventoryObjectVersion;
(function (InventoryObjectVersion) {
    InventoryObjectVersion["ALL"] = "All";
    InventoryObjectVersion["CURRENT"] = "Current";
})(InventoryObjectVersion = exports.InventoryObjectVersion || (exports.InventoryObjectVersion = {}));
/**
 * The ObjectOwnership of the bucket.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/about-object-ownership.html
 * @stability stable
 */
var ObjectOwnership;
(function (ObjectOwnership) {
    ObjectOwnership["BUCKET_OWNER_PREFERRED"] = "BucketOwnerPreferred";
    ObjectOwnership["OBJECT_WRITER"] = "ObjectWriter";
})(ObjectOwnership = exports.ObjectOwnership || (exports.ObjectOwnership = {}));
/**
 * An S3 bucket with associated policy objects.
 *
 * This bucket does not yet have all features that exposed by the underlying
 * BucketResource.
 *
 * @stability stable
 */
class Bucket extends BucketBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _d;
        super(scope, id, {
            physicalName: props.bucketName,
        });
        /**
         * Indicates if a bucket resource policy should automatically created upon the first call to `addToResourcePolicy`.
         *
         * @stability stable
         */
        this.autoCreatePolicy = true;
        this.lifecycleRules = [];
        this.metrics = [];
        this.cors = [];
        this.inventories = [];
        const { bucketEncryption, encryptionKey } = this.parseEncryption(props);
        this.validateBucketName(this.physicalName);
        const websiteConfiguration = this.renderWebsiteConfiguration(props);
        this.isWebsite = (websiteConfiguration !== undefined);
        const resource = new s3_generated_1.CfnBucket(this, 'Resource', {
            bucketName: this.physicalName,
            bucketEncryption,
            versioningConfiguration: props.versioned ? { status: 'Enabled' } : undefined,
            lifecycleConfiguration: core_1.Lazy.any({ produce: () => this.parseLifecycleConfiguration() }),
            websiteConfiguration,
            publicAccessBlockConfiguration: props.blockPublicAccess,
            metricsConfigurations: core_1.Lazy.any({ produce: () => this.parseMetricConfiguration() }),
            corsConfiguration: core_1.Lazy.any({ produce: () => this.parseCorsConfiguration() }),
            accessControl: core_1.Lazy.string({ produce: () => this.accessControl }),
            loggingConfiguration: this.parseServerAccessLogs(props),
            inventoryConfigurations: core_1.Lazy.any({ produce: () => this.parseInventoryConfiguration() }),
            ownershipControls: this.parseOwnershipControls(props),
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.versioned = props.versioned;
        this.encryptionKey = encryptionKey;
        this.bucketName = this.getResourceNameAttribute(resource.ref);
        this.bucketArn = this.getResourceArnAttribute(resource.attrArn, {
            region: '',
            account: '',
            service: 's3',
            resource: this.physicalName,
        });
        this.bucketDomainName = resource.attrDomainName;
        this.bucketWebsiteUrl = resource.attrWebsiteUrl;
        this.bucketWebsiteDomainName = core_1.Fn.select(2, core_1.Fn.split('/', this.bucketWebsiteUrl));
        this.bucketDualStackDomainName = resource.attrDualStackDomainName;
        this.bucketRegionalDomainName = resource.attrRegionalDomainName;
        this.disallowPublicAccess = props.blockPublicAccess && props.blockPublicAccess.blockPublicPolicy;
        this.accessControl = props.accessControl;
        // Enforce AWS Foundational Security Best Practice
        if (props.enforceSSL) {
            this.enforceSSLStatement();
        }
        if (props.serverAccessLogsBucket instanceof Bucket) {
            props.serverAccessLogsBucket.allowLogDelivery();
        }
        for (const inventory of (_d = props.inventories) !== null && _d !== void 0 ? _d : []) {
            this.addInventory(inventory);
        }
        // Add all bucket metric configurations rules
        (props.metrics || []).forEach(this.addMetric.bind(this));
        // Add all cors configuration rules
        (props.cors || []).forEach(this.addCorsRule.bind(this));
        // Add all lifecycle rules
        (props.lifecycleRules || []).forEach(this.addLifecycleRule.bind(this));
        // defines a BucketNotifications construct. Notice that an actual resource will only
        // be added if there are notifications added, so we don't need to condition this.
        this.notifications = new notifications_resource_1.BucketNotifications(this, 'Notifications', { bucket: this });
        if (props.publicReadAccess) {
            this.grantPublicAccess();
        }
        if (props.autoDeleteObjects) {
            if (props.removalPolicy !== core_1.RemovalPolicy.DESTROY) {
                throw new Error('Cannot use \'autoDeleteObjects\' property on a bucket without setting removal policy to \'DESTROY\'.');
            }
            this.enableAutoDeleteObjects();
        }
    }
    /**
     * @stability stable
     */
    static fromBucketArn(scope, id, bucketArn) {
        return Bucket.fromBucketAttributes(scope, id, { bucketArn });
    }
    /**
     * @stability stable
     */
    static fromBucketName(scope, id, bucketName) {
        return Bucket.fromBucketAttributes(scope, id, { bucketName });
    }
    /**
     * Creates a Bucket construct that represents an external bucket.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `BucketAttributes` object.
     * @stability stable
     */
    static fromBucketAttributes(scope, id, attrs) {
        var _d;
        const stack = core_1.Stack.of(scope);
        const region = (_d = attrs.region) !== null && _d !== void 0 ? _d : stack.region;
        const urlSuffix = stack.urlSuffix;
        const bucketName = util_1.parseBucketName(scope, attrs);
        if (!bucketName) {
            throw new Error('Bucket name is required');
        }
        const newUrlFormat = attrs.bucketWebsiteNewUrlFormat === undefined
            ? false
            : attrs.bucketWebsiteNewUrlFormat;
        const websiteDomain = newUrlFormat
            ? `${bucketName}.s3-website.${region}.${urlSuffix}`
            : `${bucketName}.s3-website-${region}.${urlSuffix}`;
        class Import extends BucketBase {
            constructor() {
                var _d;
                super(...arguments);
                this.bucketName = bucketName;
                this.bucketArn = util_1.parseBucketArn(scope, attrs);
                this.bucketDomainName = attrs.bucketDomainName || `${bucketName}.s3.${urlSuffix}`;
                this.bucketWebsiteUrl = attrs.bucketWebsiteUrl || `http://${websiteDomain}`;
                this.bucketWebsiteDomainName = attrs.bucketWebsiteUrl ? core_1.Fn.select(2, core_1.Fn.split('/', attrs.bucketWebsiteUrl)) : websiteDomain;
                this.bucketRegionalDomainName = attrs.bucketRegionalDomainName || `${bucketName}.s3.${region}.${urlSuffix}`;
                this.bucketDualStackDomainName = attrs.bucketDualStackDomainName || `${bucketName}.s3.dualstack.${region}.${urlSuffix}`;
                this.bucketWebsiteNewUrlFormat = newUrlFormat;
                this.encryptionKey = attrs.encryptionKey;
                this.isWebsite = (_d = attrs.isWebsite) !== null && _d !== void 0 ? _d : false;
                this.policy = undefined;
                this.autoCreatePolicy = false;
                this.disallowPublicAccess = false;
            }
            /**
             * Exports this bucket from the stack.
             */
            export() {
                return attrs;
            }
        }
        return new Import(scope, id, {
            account: attrs.account,
            region: attrs.region,
        });
    }
    /**
     * Add a lifecycle rule to the bucket.
     *
     * @param rule The rule to add.
     * @stability stable
     */
    addLifecycleRule(rule) {
        if ((rule.noncurrentVersionExpiration !== undefined
            || (rule.noncurrentVersionTransitions && rule.noncurrentVersionTransitions.length > 0))
            && !this.versioned) {
            throw new Error("Cannot use 'noncurrent' rules on a nonversioned bucket");
        }
        this.lifecycleRules.push(rule);
    }
    /**
     * Adds a metrics configuration for the CloudWatch request metrics from the bucket.
     *
     * @param metric The metric configuration to add.
     * @stability stable
     */
    addMetric(metric) {
        this.metrics.push(metric);
    }
    /**
     * Adds a cross-origin access configuration for objects in an Amazon S3 bucket.
     *
     * @param rule The CORS configuration rule to add.
     * @stability stable
     */
    addCorsRule(rule) {
        this.cors.push(rule);
    }
    /**
     * Adds a bucket notification event destination.
     *
     * @param event The event to trigger the notification.
     * @param dest The notification destination (Lambda, SNS Topic or SQS Queue).
     * @param filters S3 object key filter rules to determine which objects trigger this event.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html
     * @stability stable
     * @example
     *
     *    bucket.addEventNotification(EventType.OnObjectCreated, myLambda, 'home/myusername/*')
     */
    addEventNotification(event, dest, ...filters) {
        this.notifications.addNotification(event, dest, ...filters);
    }
    /**
     * Subscribes a destination to receive notifications when an object is created in the bucket.
     *
     * This is identical to calling
     * `onEvent(EventType.ObjectCreated)`.
     *
     * @param dest The notification destination (see onEvent).
     * @param filters Filters (see onEvent).
     * @stability stable
     */
    addObjectCreatedNotification(dest, ...filters) {
        return this.addEventNotification(EventType.OBJECT_CREATED, dest, ...filters);
    }
    /**
     * Subscribes a destination to receive notifications when an object is removed from the bucket.
     *
     * This is identical to calling
     * `onEvent(EventType.ObjectRemoved)`.
     *
     * @param dest The notification destination (see onEvent).
     * @param filters Filters (see onEvent).
     * @stability stable
     */
    addObjectRemovedNotification(dest, ...filters) {
        return this.addEventNotification(EventType.OBJECT_REMOVED, dest, ...filters);
    }
    /**
     * Add an inventory configuration.
     *
     * @param inventory configuration to add.
     * @stability stable
     */
    addInventory(inventory) {
        this.inventories.push(inventory);
    }
    /**
     * Adds an iam statement to enforce SSL requests only.
     */
    enforceSSLStatement() {
        const statement = new iam.PolicyStatement({
            actions: ['s3:*'],
            conditions: {
                Bool: { 'aws:SecureTransport': 'false' },
            },
            effect: iam.Effect.DENY,
            resources: [this.arnForObjects('*')],
            principals: [new iam.AnyPrincipal()],
        });
        this.addToResourcePolicy(statement);
    }
    validateBucketName(physicalName) {
        const bucketName = physicalName;
        if (!bucketName || core_1.Token.isUnresolved(bucketName)) {
            // the name is a late-bound value, not a defined string,
            // so skip validation
            return;
        }
        const errors = [];
        // Rules codified from https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html
        if (bucketName.length < 3 || bucketName.length > 63) {
            errors.push('Bucket name must be at least 3 and no more than 63 characters');
        }
        const charsetMatch = bucketName.match(/[^a-z0-9.-]/);
        if (charsetMatch) {
            errors.push('Bucket name must only contain lowercase characters and the symbols, period (.) and dash (-) '
                + `(offset: ${charsetMatch.index})`);
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(0))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + '(offset: 0)');
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(bucketName.length - 1))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + `(offset: ${bucketName.length - 1})`);
        }
        const consecSymbolMatch = bucketName.match(/\.-|-\.|\.\./);
        if (consecSymbolMatch) {
            errors.push('Bucket name must not have dash next to period, or period next to dash, or consecutive periods '
                + `(offset: ${consecSymbolMatch.index})`);
        }
        if (/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/.test(bucketName)) {
            errors.push('Bucket name must not resemble an IP address');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid S3 bucket name (value: ${bucketName})${os_1.EOL}${errors.join(os_1.EOL)}`);
        }
    }
    /**
     * Set up key properties and return the Bucket encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        // default based on whether encryptionKey is specified
        let encryptionType = props.encryption;
        if (encryptionType === undefined) {
            encryptionType = props.encryptionKey ? BucketEncryption.KMS : BucketEncryption.UNENCRYPTED;
        }
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== BucketEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        // if bucketKeyEnabled is set, encryption must be set to KMS.
        if (props.bucketKeyEnabled && encryptionType !== BucketEncryption.KMS) {
            throw new Error(`bucketKeyEnabled is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === BucketEncryption.UNENCRYPTED) {
            return { bucketEncryption: undefined, encryptionKey: undefined };
        }
        if (encryptionType === BucketEncryption.KMS) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`,
            });
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    {
                        bucketKeyEnabled: props.bucketKeyEnabled,
                        serverSideEncryptionByDefault: {
                            sseAlgorithm: 'aws:kms',
                            kmsMasterKeyId: encryptionKey.keyArn,
                        },
                    },
                ],
            };
            return { encryptionKey, bucketEncryption };
        }
        if (encryptionType === BucketEncryption.S3_MANAGED) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    { serverSideEncryptionByDefault: { sseAlgorithm: 'AES256' } },
                ],
            };
            return { bucketEncryption };
        }
        if (encryptionType === BucketEncryption.KMS_MANAGED) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    { serverSideEncryptionByDefault: { sseAlgorithm: 'aws:kms' } },
                ],
            };
            return { bucketEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
    /**
     * Parse the lifecycle configuration out of the uucket props
     * @param props Par
     */
    parseLifecycleConfiguration() {
        if (!this.lifecycleRules || this.lifecycleRules.length === 0) {
            return undefined;
        }
        const self = this;
        return { rules: this.lifecycleRules.map(parseLifecycleRule) };
        function parseLifecycleRule(rule) {
            var _d, _e;
            const enabled = (_d = rule.enabled) !== null && _d !== void 0 ? _d : true;
            const x = {
                // eslint-disable-next-line max-len
                abortIncompleteMultipartUpload: rule.abortIncompleteMultipartUploadAfter !== undefined ? { daysAfterInitiation: rule.abortIncompleteMultipartUploadAfter.toDays() } : undefined,
                expirationDate: rule.expirationDate,
                expirationInDays: (_e = rule.expiration) === null || _e === void 0 ? void 0 : _e.toDays(),
                id: rule.id,
                noncurrentVersionExpirationInDays: rule.noncurrentVersionExpiration && rule.noncurrentVersionExpiration.toDays(),
                noncurrentVersionTransitions: mapOrUndefined(rule.noncurrentVersionTransitions, t => ({
                    storageClass: t.storageClass.value,
                    transitionInDays: t.transitionAfter.toDays(),
                })),
                prefix: rule.prefix,
                status: enabled ? 'Enabled' : 'Disabled',
                transitions: mapOrUndefined(rule.transitions, t => ({
                    storageClass: t.storageClass.value,
                    transitionDate: t.transitionDate,
                    transitionInDays: t.transitionAfter && t.transitionAfter.toDays(),
                })),
                tagFilters: self.parseTagFilters(rule.tagFilters),
            };
            return x;
        }
    }
    parseServerAccessLogs(props) {
        var _d;
        if (!props.serverAccessLogsBucket && !props.serverAccessLogsPrefix) {
            return undefined;
        }
        return {
            destinationBucketName: (_d = props.serverAccessLogsBucket) === null || _d === void 0 ? void 0 : _d.bucketName,
            logFilePrefix: props.serverAccessLogsPrefix,
        };
    }
    parseMetricConfiguration() {
        if (!this.metrics || this.metrics.length === 0) {
            return undefined;
        }
        const self = this;
        return this.metrics.map(parseMetric);
        function parseMetric(metric) {
            return {
                id: metric.id,
                prefix: metric.prefix,
                tagFilters: self.parseTagFilters(metric.tagFilters),
            };
        }
    }
    parseCorsConfiguration() {
        if (!this.cors || this.cors.length === 0) {
            return undefined;
        }
        return { corsRules: this.cors.map(parseCors) };
        function parseCors(rule) {
            return {
                id: rule.id,
                maxAge: rule.maxAge,
                allowedHeaders: rule.allowedHeaders,
                allowedMethods: rule.allowedMethods,
                allowedOrigins: rule.allowedOrigins,
                exposedHeaders: rule.exposedHeaders,
            };
        }
    }
    parseTagFilters(tagFilters) {
        if (!tagFilters || tagFilters.length === 0) {
            return undefined;
        }
        return Object.keys(tagFilters).map(tag => ({
            key: tag,
            value: tagFilters[tag],
        }));
    }
    parseOwnershipControls({ objectOwnership }) {
        if (!objectOwnership) {
            return undefined;
        }
        return {
            rules: [{
                    objectOwnership,
                }],
        };
    }
    renderWebsiteConfiguration(props) {
        if (!props.websiteErrorDocument && !props.websiteIndexDocument && !props.websiteRedirect && !props.websiteRoutingRules) {
            return undefined;
        }
        if (props.websiteErrorDocument && !props.websiteIndexDocument) {
            throw new Error('"websiteIndexDocument" is required if "websiteErrorDocument" is set');
        }
        if (props.websiteRedirect && (props.websiteErrorDocument || props.websiteIndexDocument || props.websiteRoutingRules)) {
            throw new Error('"websiteIndexDocument", "websiteErrorDocument" and, "websiteRoutingRules" cannot be set if "websiteRedirect" is used');
        }
        const routingRules = props.websiteRoutingRules ? props.websiteRoutingRules.map((rule) => {
            if (rule.condition && !rule.condition.httpErrorCodeReturnedEquals && !rule.condition.keyPrefixEquals) {
                throw new Error('The condition property cannot be an empty object');
            }
            return {
                redirectRule: {
                    hostName: rule.hostName,
                    httpRedirectCode: rule.httpRedirectCode,
                    protocol: rule.protocol,
                    replaceKeyWith: rule.replaceKey && rule.replaceKey.withKey,
                    replaceKeyPrefixWith: rule.replaceKey && rule.replaceKey.prefixWithKey,
                },
                routingRuleCondition: rule.condition,
            };
        }) : undefined;
        return {
            indexDocument: props.websiteIndexDocument,
            errorDocument: props.websiteErrorDocument,
            redirectAllRequestsTo: props.websiteRedirect,
            routingRules,
        };
    }
    /**
     * Allows the LogDelivery group to write, fails if ACL was set differently.
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl
     */
    allowLogDelivery() {
        if (this.accessControl && this.accessControl !== BucketAccessControl.LOG_DELIVERY_WRITE) {
            throw new Error("Cannot enable log delivery to this bucket because the bucket's ACL has been set and can't be changed");
        }
        this.accessControl = BucketAccessControl.LOG_DELIVERY_WRITE;
    }
    parseInventoryConfiguration() {
        if (!this.inventories || this.inventories.length === 0) {
            return undefined;
        }
        return this.inventories.map((inventory, index) => {
            var _d, _e, _f, _g, _h, _j;
            const format = (_d = inventory.format) !== null && _d !== void 0 ? _d : InventoryFormat.CSV;
            const frequency = (_e = inventory.frequency) !== null && _e !== void 0 ? _e : InventoryFrequency.WEEKLY;
            const id = (_f = inventory.inventoryId) !== null && _f !== void 0 ? _f : `${this.node.id}Inventory${index}`;
            if (inventory.destination.bucket instanceof Bucket) {
                inventory.destination.bucket.addToResourcePolicy(new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['s3:PutObject'],
                    resources: [
                        inventory.destination.bucket.bucketArn,
                        inventory.destination.bucket.arnForObjects(`${(_g = inventory.destination.prefix) !== null && _g !== void 0 ? _g : ''}*`),
                    ],
                    principals: [new iam.ServicePrincipal('s3.amazonaws.com')],
                    conditions: {
                        ArnLike: {
                            'aws:SourceArn': this.bucketArn,
                        },
                    },
                }));
            }
            return {
                id,
                destination: {
                    bucketArn: inventory.destination.bucket.bucketArn,
                    bucketAccountId: inventory.destination.bucketOwner,
                    prefix: inventory.destination.prefix,
                    format,
                },
                enabled: (_h = inventory.enabled) !== null && _h !== void 0 ? _h : true,
                includedObjectVersions: (_j = inventory.includeObjectVersions) !== null && _j !== void 0 ? _j : InventoryObjectVersion.ALL,
                scheduleFrequency: frequency,
                optionalFields: inventory.optionalFields,
                prefix: inventory.objectsPrefix,
            };
        });
    }
    enableAutoDeleteObjects() {
        const provider = core_1.CustomResourceProvider.getOrCreateProvider(this, AUTO_DELETE_OBJECTS_RESOURCE_TYPE, {
            codeDirectory: path.join(__dirname, 'auto-delete-objects-handler'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12,
            description: `Lambda function for auto-deleting objects in ${this.bucketName} S3 bucket.`,
        });
        // Use a bucket policy to allow the custom resource to delete
        // objects in the bucket
        this.addToResourcePolicy(new iam.PolicyStatement({
            actions: [
                ...perms.BUCKET_READ_ACTIONS,
                ...perms.BUCKET_DELETE_ACTIONS,
            ],
            resources: [
                this.bucketArn,
                this.arnForObjects('*'),
            ],
            principals: [new iam.ArnPrincipal(provider.roleArn)],
        }));
        const customResource = new core_1.CustomResource(this, 'AutoDeleteObjectsCustomResource', {
            resourceType: AUTO_DELETE_OBJECTS_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                BucketName: this.bucketName,
            },
        });
        // Ensure bucket policy is deleted AFTER the custom resource otherwise
        // we don't have permissions to list and delete in the bucket.
        // (add a `if` to make TS happy)
        if (this.policy) {
            customResource.node.addDependency(this.policy);
        }
    }
}
exports.Bucket = Bucket;
_b = JSII_RTTI_SYMBOL_1;
Bucket[_b] = { fqn: "@aws-cdk/aws-s3.Bucket", version: "1.93.0" };
/**
 * What kind of server-side encryption to apply to this bucket.
 *
 * @stability stable
 */
var BucketEncryption;
(function (BucketEncryption) {
    BucketEncryption["UNENCRYPTED"] = "NONE";
    BucketEncryption["KMS_MANAGED"] = "MANAGED";
    BucketEncryption["S3_MANAGED"] = "S3MANAGED";
    BucketEncryption["KMS"] = "KMS";
})(BucketEncryption = exports.BucketEncryption || (exports.BucketEncryption = {}));
/**
 * Notification event types.
 *
 * @stability stable
 */
var EventType;
(function (EventType) {
    EventType["OBJECT_CREATED"] = "s3:ObjectCreated:*";
    EventType["OBJECT_CREATED_PUT"] = "s3:ObjectCreated:Put";
    EventType["OBJECT_CREATED_POST"] = "s3:ObjectCreated:Post";
    EventType["OBJECT_CREATED_COPY"] = "s3:ObjectCreated:Copy";
    EventType["OBJECT_CREATED_COMPLETE_MULTIPART_UPLOAD"] = "s3:ObjectCreated:CompleteMultipartUpload";
    EventType["OBJECT_REMOVED"] = "s3:ObjectRemoved:*";
    EventType["OBJECT_REMOVED_DELETE"] = "s3:ObjectRemoved:Delete";
    EventType["OBJECT_REMOVED_DELETE_MARKER_CREATED"] = "s3:ObjectRemoved:DeleteMarkerCreated";
    EventType["OBJECT_RESTORE_POST"] = "s3:ObjectRestore:Post";
    EventType["OBJECT_RESTORE_COMPLETED"] = "s3:ObjectRestore:Completed";
    EventType["REDUCED_REDUNDANCY_LOST_OBJECT"] = "s3:ReducedRedundancyLostObject";
    EventType["REPLICATION_OPERATION_FAILED_REPLICATION"] = "s3:Replication:OperationFailedReplication";
    EventType["REPLICATION_OPERATION_MISSED_THRESHOLD"] = "s3:Replication:OperationMissedThreshold";
    EventType["REPLICATION_OPERATION_REPLICATED_AFTER_THRESHOLD"] = "s3:Replication:OperationReplicatedAfterThreshold";
    EventType["REPLICATION_OPERATION_NOT_TRACKED"] = "s3:Replication:OperationNotTracked";
})(EventType = exports.EventType || (exports.EventType = {}));
/**
 * Default bucket access control types.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html
 * @stability stable
 */
var BucketAccessControl;
(function (BucketAccessControl) {
    BucketAccessControl["PRIVATE"] = "Private";
    BucketAccessControl["PUBLIC_READ"] = "PublicRead";
    BucketAccessControl["PUBLIC_READ_WRITE"] = "PublicReadWrite";
    BucketAccessControl["AUTHENTICATED_READ"] = "AuthenticatedRead";
    BucketAccessControl["LOG_DELIVERY_WRITE"] = "LogDeliveryWrite";
    BucketAccessControl["BUCKET_OWNER_READ"] = "BucketOwnerRead";
    BucketAccessControl["BUCKET_OWNER_FULL_CONTROL"] = "BucketOwnerFullControl";
    BucketAccessControl["AWS_EXEC_READ"] = "AwsExecRead";
})(BucketAccessControl = exports.BucketAccessControl || (exports.BucketAccessControl = {}));
/**
 * @stability stable
 */
class ReplaceKey {
    constructor(withKey, prefixWithKey) {
        this.withKey = withKey;
        this.prefixWithKey = prefixWithKey;
    }
    /**
     * The specific object key to use in the redirect request.
     *
     * @stability stable
     */
    static with(keyReplacement) {
        return new this(keyReplacement);
    }
    /**
     * The object key prefix to use in the redirect request.
     *
     * @stability stable
     */
    static prefixWith(keyReplacement) {
        return new this(undefined, keyReplacement);
    }
}
exports.ReplaceKey = ReplaceKey;
_c = JSII_RTTI_SYMBOL_1;
ReplaceKey[_c] = { fqn: "@aws-cdk/aws-s3.ReplaceKey", version: "1.93.0" };
function mapOrUndefined(list, callback) {
    if (!list || list.length === 0) {
        return undefined;
    }
    return list.map(callback);
}
//# sourceMappingURL=data:application/json;base64,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