"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateGraph = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * A collection of connected states.
 *
 * A StateGraph is used to keep track of all states that are connected (have
 * transitions between them). It does not include the substatemachines in
 * a Parallel's branches: those are their own StateGraphs, but the graphs
 * themselves have a hierarchical relationship as well.
 *
 * By assigning states to a definitive StateGraph, we verify that no state
 * machines are constructed. In particular:
 *
 * - Every state object can only ever be in 1 StateGraph, and not inadvertently
 *    be used in two graphs.
 * - Every stateId must be unique across all states in the entire state
 *    machine.
 *
 * All policy statements in all states in all substatemachines are bubbled so
 * that the top-level StateMachine instantiation can read them all and add
 * them to the IAM Role.
 *
 * You do not need to instantiate this class; it is used internally.
 *
 * @stability stable
 */
class StateGraph {
    /**
     * @param startState state that gets executed when the state machine is launched.
     * @param graphDescription description of the state machine.
     * @stability stable
     */
    constructor(startState, graphDescription) {
        this.startState = startState;
        this.graphDescription = graphDescription;
        /**
         * The accumulated policy statements.
         *
         * @stability stable
         */
        this.policyStatements = new Array();
        /**
         * All states in this graph
         */
        this.allStates = new Set();
        /**
         * A mapping of stateId -> Graph for all states in this graph and subgraphs
         */
        this.allContainedStates = new Map();
        this.allStates.add(startState);
        startState.bindToGraph(this);
    }
    /**
     * Register a state as part of this graph.
     *
     * Called by State.bindToGraph().
     *
     * @stability stable
     */
    registerState(state) {
        this.registerContainedState(state.stateId, this);
        this.allStates.add(state);
    }
    /**
     * Register a Policy Statement used by states in this graph.
     *
     * @stability stable
     */
    registerPolicyStatement(statement) {
        if (this.superGraph) {
            this.superGraph.registerPolicyStatement(statement);
        }
        else {
            this.policyStatements.push(statement);
        }
    }
    /**
     * Register this graph as a child of the given graph.
     *
     * Resource changes will be bubbled up to the given graph.
     *
     * @stability stable
     */
    registerSuperGraph(graph) {
        if (this.superGraph === graph) {
            return;
        }
        if (this.superGraph) {
            throw new Error('Every StateGraph can only be registered into one other StateGraph');
        }
        this.superGraph = graph;
        this.pushContainedStatesUp(graph);
        this.pushPolicyStatementsUp(graph);
    }
    /**
     * Return the Amazon States Language JSON for this graph.
     *
     * @stability stable
     */
    toGraphJson() {
        const states = {};
        for (const state of this.allStates) {
            states[state.stateId] = state.toStateJson();
        }
        return {
            StartAt: this.startState.stateId,
            States: states,
            TimeoutSeconds: this.timeout && this.timeout.toSeconds(),
        };
    }
    /**
     * Return a string description of this graph.
     *
     * @stability stable
     */
    toString() {
        const someNodes = Array.from(this.allStates).slice(0, 3).map(x => x.stateId);
        if (this.allStates.size > 3) {
            someNodes.push('...');
        }
        return `${this.graphDescription} (${someNodes.join(', ')})`;
    }
    /**
     * Register a stateId and graph where it was registered
     */
    registerContainedState(stateId, graph) {
        if (this.superGraph) {
            this.superGraph.registerContainedState(stateId, graph);
        }
        else {
            const existingGraph = this.allContainedStates.get(stateId);
            if (existingGraph) {
                throw new Error(`State with name '${stateId}' occurs in both ${graph} and ${existingGraph}. All states must have unique names.`);
            }
            this.allContainedStates.set(stateId, graph);
        }
    }
    /**
     * Push all contained state info up to the given super graph
     */
    pushContainedStatesUp(superGraph) {
        for (const [stateId, graph] of this.allContainedStates) {
            superGraph.registerContainedState(stateId, graph);
        }
    }
    /**
     * Push all policy statements to into the given super graph
     */
    pushPolicyStatementsUp(superGraph) {
        for (const policyStatement of this.policyStatements) {
            superGraph.registerPolicyStatement(policyStatement);
        }
    }
}
exports.StateGraph = StateGraph;
_a = JSII_RTTI_SYMBOL_1;
StateGraph[_a] = { fqn: "@aws-cdk/aws-stepfunctions.StateGraph", version: "1.129.0" };
//# sourceMappingURL=data:application/json;base64,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