"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const s3 = require("@aws-cdk/aws-s3");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test Case 2 - Test the getter methods
// --------------------------------------------------------------
test('Test properties', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    };
    const app = new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-kinesis-firehose-kinesis-analytics', props);
    // Assertions
    expect(app.kinesisAnalytics !== null);
    expect(app.kinesisFirehose !== null);
    expect(app.kinesisFirehoseRole !== null);
    expect(app.kinesisFirehoseLogGroup !== null);
    expect(app.s3Bucket !== null);
    expect(app.s3LoggingBucket !== null);
});
// --------------------------------------------------------------
// Test Case 3 - Override kinesisFirehose props
// --------------------------------------------------------------
test('test kinesisFirehose override ', () => {
    const stack = new core_1.Stack();
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-firehose-s3', {
        kinesisFirehoseProps: {
            extendedS3DestinationConfiguration: {
                bufferingHints: {
                    intervalInSeconds: 600,
                    sizeInMBs: 55
                },
            }
        },
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        ExtendedS3DestinationConfiguration: {
            BufferingHints: {
                IntervalInSeconds: 600,
                SizeInMBs: 55
            }
        }
    });
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Test bad call with existingBucket and bucketProps", () => {
    // Stack
    const stack = new core_1.Stack();
    const testBucket = new s3.Bucket(stack, 'test-bucket', {});
    const app = () => {
        // Helper declaration
        new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, "bad-s3-args", {
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: core_1.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// s3 bucket with bucket, loggingBucket, and auto delete objects
// --------------------------------------------------------------
test('s3 bucket with bucket, loggingBucket, and auto delete objects', () => {
    const stack = new core_1.Stack();
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'kinsisfirehose-s3-analytics', {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        },
        loggingBucketProps: {
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        AccessControl: "LogDeliveryWrite"
    });
    expect(stack).toHaveResource("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "kinsisfirehoses3analyticsKinesisFirehoseToS3S3LoggingBucket6EE455EF"
        }
    });
});
// --------------------------------------------------------------
// s3 bucket with one content bucket and no logging bucket
// --------------------------------------------------------------
test('s3 bucket with one content bucket and no logging bucket', () => {
    const stack = new core_1.Stack();
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'kinsisfirehose-s3-analytics', {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        },
        kinesisFirehoseProps: {
            deliveryStreamType: 'KinesisStreamAsSource'
        },
        bucketProps: {
            removalPolicy: core_1.RemovalPolicy.DESTROY,
        },
        logS3AccessLogs: false
    });
    expect(stack).toCountResources("AWS::S3::Bucket", 1);
});
//# sourceMappingURL=data:application/json;base64,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