"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test Case 1 - Pattern deployment w/ default properties
// --------------------------------------------------------------
test('Pattern deployment w/ default properties', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ticker_symbol',
                                sqlType: 'VARCHAR(4)',
                                mapping: '$.ticker_symbol'
                            }, {
                                name: 'sector',
                                sqlType: 'VARCHAR(16)',
                                mapping: '$.sector'
                            }, {
                                name: 'change',
                                sqlType: 'REAL',
                                mapping: '$.change'
                            }, {
                                name: 'price',
                                sqlType: 'REAL',
                                mapping: '$.price'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    };
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-firehose-s3-and-analytics-stack', props);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test Case 2 - Test the getter methods
// --------------------------------------------------------------
test('Test properties', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    };
    const app = new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-kinesis-firehose-kinesis-analytics', props);
    // Assertions
    expect(app.kinesisAnalytics !== null);
    expect(app.kinesisFirehose !== null);
    expect(app.kinesisFirehoseRole !== null);
    expect(app.kinesisFirehoseLogGroup !== null);
    expect(app.s3Bucket !== null);
    expect(app.s3LoggingBucket !== null);
});
// --------------------------------------------------------------
// Test Case 3 - Override kinesisFirehose props
// --------------------------------------------------------------
test('test kinesisFirehose override ', () => {
    const stack = new core_1.Stack();
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-firehose-s3', {
        kinesisFirehoseProps: {
            extendedS3DestinationConfiguration: {
                bufferingHints: {
                    intervalInSeconds: 600,
                    sizeInMBs: 55
                },
            }
        },
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        ExtendedS3DestinationConfiguration: {
            BufferingHints: {
                IntervalInSeconds: 600,
                SizeInMBs: 55
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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