"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToSqs = void 0;
// Imports
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
const cdk = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayToSqs class.
 */
class ApiGatewayToSqs extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the ApiGatewayToSqs class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {ApiGatewayToSqsProps} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the dead letter queue, if applicable
        let dlqi;
        if (!props.deployDeadLetterQueue || props.deployDeadLetterQueue === true) {
            const dlq = defaults.buildQueue(this, 'deadLetterQueue', {
                queueProps: props.queueProps
            });
            dlqi = defaults.buildDeadLetterQueue({
                deadLetterQueue: dlq,
                maxReceiveCount: (props.maxReceiveCount) ? props.maxReceiveCount : 3
            });
        }
        // Setup the queue
        this.sqsQueue = defaults.buildQueue(this, 'queue', {
            queueProps: props.queueProps,
            deadLetterQueue: dlqi
        });
        // Setup the API Gateway
        this.apiGateway = defaults.GlobalRestApi(this);
        // Setup the API Gateway role
        this.apiGatewayRole = new iam.Role(this, 'api-gateway-role', {
            assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
        });
        // Setup the API Gateway resource
        const apiGatewayResource = this.apiGateway.root.addResource('message');
        // Setup API Gateway methods
        // Create
        if (props.allowCreateOperation && props.allowCreateOperation === true && props.createRequestTemplate) {
            const createRequestTemplate = "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")";
            this.addActionToPolicy("sqs:SendMessage");
            this.addMethod(this.apiGateway.root, createRequestTemplate, "POST");
        }
        // Read
        if (!props.allowReadOperation || props.allowReadOperation === true) {
            const getRequestTemplate = "Action=ReceiveMessage";
            this.addActionToPolicy("sqs:ReceiveMessage");
            this.addMethod(this.apiGateway.root, getRequestTemplate, "GET");
        }
        // Delete
        if (props.allowDeleteOperation && props.allowDeleteOperation === true) {
            const deleteRequestTemplate = "Action=DeleteMessage&ReceiptHandle=$util.urlEncode($input.params('receiptHandle'))";
            this.addActionToPolicy("sqs:DeleteMessage");
            this.addMethod(apiGatewayResource, deleteRequestTemplate, "DELETE");
        }
    }
    addActionToPolicy(action) {
        this.apiGatewayRole.addToPolicy(new iam.PolicyStatement({
            resources: [
                this.sqsQueue.queueArn
            ],
            actions: [`${action}`]
        }));
    }
    addMethod(apiResource, requestTemplate, apiMethod) {
        // Add the integration
        const apiGatewayIntegration = new api.AwsIntegration({
            service: "sqs",
            path: `${cdk.Aws.ACCOUNT_ID}/${this.sqsQueue.queueName}`,
            integrationHttpMethod: "POST",
            options: {
                passthroughBehavior: api.PassthroughBehavior.NEVER,
                credentialsRole: this.apiGatewayRole,
                requestParameters: {
                    "integration.request.header.Content-Type": "'application/x-www-form-urlencoded'"
                },
                requestTemplates: {
                    "application/json": requestTemplate
                },
                integrationResponses: [
                    {
                        statusCode: "200"
                    },
                    {
                        statusCode: "500",
                        responseTemplates: {
                            "text/html": "Error"
                        },
                        selectionPattern: "500"
                    }
                ]
            }
        });
        // Add the method to the resource
        apiResource.addMethod(apiMethod, apiGatewayIntegration, {
            authorizationType: api.AuthorizationType.IAM,
            methodResponses: [
                {
                    statusCode: "200",
                    responseParameters: {
                        "method.response.header.Content-Type": true
                    }
                },
                {
                    statusCode: "500",
                    responseParameters: {
                        "method.response.header.Content-Type": true
                    },
                }
            ]
        });
    }
}
exports.ApiGatewayToSqs = ApiGatewayToSqs;
//# sourceMappingURL=data:application/json;base64,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