"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const json_path_1 = require("./json-path");
/**
 * Extract a field from the State Machine data that gets passed around between states
 */
class Data {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateDataPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal string list, get the value from a JSON path
     */
    static listAt(path) {
        validateDataPath(path);
        return core_1.Token.asList(new json_path_1.JsonPathToken(path));
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateDataPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Use the entire data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entirePayload() {
        return new json_path_1.JsonPathToken('$').toString();
    }
    /**
     * Determines if the indicated string is an encoded JSON path
     *
     * @param value string to be evaluated
     */
    static isJsonPathString(value) {
        return !!json_path_1.jsonPathString(value);
    }
    constructor() {
    }
}
exports.Data = Data;
/**
 * Extract a field from the State Machine Context data
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html#wait-token-contextobject
 */
class Context {
    /**
     * Instead of using a literal string, get the value from a JSON path
     */
    static stringAt(path) {
        validateContextPath(path);
        return new json_path_1.JsonPathToken(path).toString();
    }
    /**
     * Instead of using a literal number, get the value from a JSON path
     */
    static numberAt(path) {
        validateContextPath(path);
        return core_1.Token.asNumber(new json_path_1.JsonPathToken(path));
    }
    /**
     * Return the Task Token field
     *
     * External actions will need this token to report step completion
     * back to StepFunctions using the `SendTaskSuccess` or `SendTaskFailure`
     * calls.
     */
    static get taskToken() {
        return new json_path_1.JsonPathToken('$$.Task.Token').toString();
    }
    /**
     * Use the entire context data structure
     *
     * Will be an object at invocation time, but is represented in the CDK
     * application as a string.
     */
    static get entireContext() {
        return new json_path_1.JsonPathToken('$$').toString();
    }
    constructor() {
    }
}
exports.Context = Context;
/**
 * Helper functions to work with structures containing fields
 */
class FieldUtils {
    /**
     * Render a JSON structure containing fields to the right StepFunctions structure
     */
    static renderObject(obj) {
        return json_path_1.renderObject(obj);
    }
    /**
     * Return all JSON paths used in the given structure
     */
    static findReferencedPaths(obj) {
        return Array.from(json_path_1.findReferencedPaths(obj)).sort();
    }
    /**
     * Returns whether the given task structure contains the TaskToken field anywhere
     *
     * The field is considered included if the field itself or one of its containing
     * fields occurs anywhere in the payload.
     */
    static containsTaskToken(obj) {
        const paths = json_path_1.findReferencedPaths(obj);
        return paths.has('$$.Task.Token') || paths.has('$$.Task') || paths.has('$$');
    }
    constructor() {
    }
}
exports.FieldUtils = FieldUtils;
function validateDataPath(path) {
    if (path !== '$' && !path.startsWith('$.')) {
        throw new Error("Data JSON path values must either be exactly equal to '$' or start with '$.'");
    }
}
function validateContextPath(path) {
    if (path !== '$$' && !path.startsWith('$$.')) {
        throw new Error("Context JSON path values must either be exactly equal to '$$' or start with '$$.'");
    }
}
//# sourceMappingURL=data:application/json;base64,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