"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateTransitionMetric = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
/**
 * Metrics on the rate limiting performed on state machine execution.
 *
 * These rate limits are shared across all state machines.
 *
 * @stability stable
 */
class StateTransitionMetric {
    /**
     * Return the given named metric for the service's state transition metrics.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    static metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: { ServiceMetric: 'StateTransition' },
            ...props,
        });
    }
    /**
     * Metric for the number of available state transitions.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    static metricProvisionedBucketSize(props) {
        return StateTransitionMetric.metric('ProvisionedBucketSize', props);
    }
    /**
     * Metric for the provisioned steady-state execution rate.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    static metricProvisionedRefillRate(props) {
        return StateTransitionMetric.metric('ProvisionedRefillRate', props);
    }
    /**
     * Metric for the number of available state transitions per second.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    static metricConsumedCapacity(props) {
        return StateTransitionMetric.metric('ConsumedCapacity', props);
    }
    /**
     * Metric for the number of throttled state transitions.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricThrottledEvents(props) {
        return StateTransitionMetric.metric('ThrottledEvents', { statistic: 'sum', ...props });
    }
}
exports.StateTransitionMetric = StateTransitionMetric;
//# sourceMappingURL=data:application/json;base64,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