"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseQuery = void 0;
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const customresources = require("@aws-cdk/custom-resources");
const cluster_1 = require("../cluster");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_1 = require("@aws-cdk/core");
class DatabaseQuery extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const adminUser = this.getAdminUser(props);
        const handler = new lambda.SingletonFunction(this, 'Handler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'database-query-provider')),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            timeout: cdk.Duration.minutes(1),
            uuid: '3de5bea7-27da-4796-8662-5efb56431b5f',
            lambdaPurpose: 'Query Redshift Database',
        });
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['redshift-data:DescribeStatement', 'redshift-data:ExecuteStatement'],
            resources: ['*'],
        }));
        adminUser.grantRead(handler);
        const provider = new customresources.Provider(this, 'Provider', {
            onEventHandler: handler,
        });
        const queryHandlerProps = {
            handler: props.handler,
            clusterName: props.cluster.clusterName,
            adminUserArn: adminUser.secretArn,
            databaseName: props.databaseName,
            ...props.properties,
        };
        this.resource = new cdk.CustomResource(this, 'Resource', {
            resourceType: 'Custom::RedshiftDatabaseQuery',
            serviceToken: provider.serviceToken,
            removalPolicy: props.removalPolicy,
            properties: queryHandlerProps,
        });
        this.grantPrincipal = handler.grantPrincipal;
        this.ref = this.resource.ref;
    }
    applyRemovalPolicy(policy) {
        this.resource.applyRemovalPolicy(policy);
    }
    getAtt(attributeName) {
        return this.resource.getAtt(attributeName);
    }
    getAttString(attributeName) {
        return this.resource.getAttString(attributeName);
    }
    getAdminUser(props) {
        const cluster = props.cluster;
        let adminUser = props.adminUser;
        if (!adminUser) {
            if (cluster instanceof cluster_1.Cluster) {
                if (cluster.secret) {
                    adminUser = cluster.secret;
                }
                else {
                    throw new Error('Administrative access to the Redshift cluster is required but an admin user secret was not provided and the cluster did not generate admin user credentials (they were provided explicitly)');
                }
            }
            else {
                throw new Error('Administrative access to the Redshift cluster is required but an admin user secret was not provided and the cluster was imported');
            }
        }
        return adminUser;
    }
}
exports.DatabaseQuery = DatabaseQuery;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGF0YWJhc2UtcXVlcnkuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJkYXRhYmFzZS1xdWVyeS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFDN0Isd0NBQXdDO0FBQ3hDLDhDQUE4QztBQUU5QyxxQ0FBcUM7QUFDckMsNkRBQTZEO0FBRTdELHdDQUFxQztBQUlyQyxpR0FBaUc7QUFDakcsOERBQThEO0FBQzlELHdDQUEyRDtBQWEzRCxNQUFhLGFBQTRCLFNBQVEsZ0JBQWE7SUFNNUQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1QztRQUMvRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDM0MsTUFBTSxPQUFPLEdBQUcsSUFBSSxNQUFNLENBQUMsaUJBQWlCLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUM1RCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUseUJBQXlCLENBQUMsQ0FBQztZQUM1RSxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLE9BQU8sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDaEMsSUFBSSxFQUFFLHNDQUFzQztZQUM1QyxhQUFhLEVBQUUseUJBQXlCO1NBQ3pDLENBQUMsQ0FBQztRQUNILE9BQU8sQ0FBQyxlQUFlLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQzlDLE9BQU8sRUFBRSxDQUFDLGlDQUFpQyxFQUFFLGdDQUFnQyxDQUFDO1lBQzlFLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNqQixDQUFDLENBQUMsQ0FBQztRQUNKLFNBQVMsQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUM7UUFFN0IsTUFBTSxRQUFRLEdBQUcsSUFBSSxlQUFlLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDOUQsY0FBYyxFQUFFLE9BQU87U0FDeEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxpQkFBaUIsR0FBNkM7WUFDbEUsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ3RCLFdBQVcsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLFdBQVc7WUFDdEMsWUFBWSxFQUFFLFNBQVMsQ0FBQyxTQUFTO1lBQ2pDLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtZQUNoQyxHQUFHLEtBQUssQ0FBQyxVQUFVO1NBQ3BCLENBQUM7UUFDRixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksR0FBRyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3ZELFlBQVksRUFBRSwrQkFBK0I7WUFDN0MsWUFBWSxFQUFFLFFBQVEsQ0FBQyxZQUFZO1lBQ25DLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtZQUNsQyxVQUFVLEVBQUUsaUJBQWlCO1NBQzlCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxjQUFjLEdBQUcsT0FBTyxDQUFDLGNBQWMsQ0FBQztRQUM3QyxJQUFJLENBQUMsR0FBRyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDO0lBQy9CLENBQUM7SUFFTSxrQkFBa0IsQ0FBQyxNQUF5QjtRQUNqRCxJQUFJLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQzNDLENBQUM7SUFFTSxNQUFNLENBQUMsYUFBcUI7UUFDakMsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxhQUFhLENBQUMsQ0FBQztJQUM3QyxDQUFDO0lBRU0sWUFBWSxDQUFDLGFBQXFCO1FBQ3ZDLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVPLFlBQVksQ0FBQyxLQUFzQjtRQUN6QyxNQUFNLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzlCLElBQUksU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7UUFDaEMsSUFBSSxDQUFDLFNBQVMsRUFBRTtZQUNkLElBQUksT0FBTyxZQUFZLGlCQUFPLEVBQUU7Z0JBQzlCLElBQUksT0FBTyxDQUFDLE1BQU0sRUFBRTtvQkFDbEIsU0FBUyxHQUFHLE9BQU8sQ0FBQyxNQUFNLENBQUM7aUJBQzVCO3FCQUFNO29CQUNMLE1BQU0sSUFBSSxLQUFLLENBQ2IsNkxBQTZMLENBQzlMLENBQUM7aUJBQ0g7YUFDRjtpQkFBTTtnQkFDTCxNQUFNLElBQUksS0FBSyxDQUNiLGtJQUFrSSxDQUNuSSxDQUFDO2FBQ0g7U0FDRjtRQUNELE9BQU8sU0FBUyxDQUFDO0lBQ25CLENBQUM7Q0FDRjtBQTlFRCxzQ0E4RUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0ICogYXMgc2VjcmV0c21hbmFnZXIgZnJvbSAnQGF3cy1jZGsvYXdzLXNlY3JldHNtYW5hZ2VyJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGN1c3RvbXJlc291cmNlcyBmcm9tICdAYXdzLWNkay9jdXN0b20tcmVzb3VyY2VzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQ2x1c3RlciB9IGZyb20gJy4uL2NsdXN0ZXInO1xuaW1wb3J0IHsgRGF0YWJhc2VPcHRpb25zIH0gZnJvbSAnLi4vZGF0YWJhc2Utb3B0aW9ucyc7XG5pbXBvcnQgeyBEYXRhYmFzZVF1ZXJ5SGFuZGxlclByb3BzIH0gZnJvbSAnLi9oYW5kbGVyLXByb3BzJztcblxuLy8ga2VlcCB0aGlzIGltcG9ydCBzZXBhcmF0ZSBmcm9tIG90aGVyIGltcG9ydHMgdG8gcmVkdWNlIGNoYW5jZSBmb3IgbWVyZ2UgY29uZmxpY3RzIHdpdGggdjItbWFpblxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG5vLWR1cGxpY2F0ZS1pbXBvcnRzLCBpbXBvcnQvb3JkZXJcbmltcG9ydCB7IENvbnN0cnVjdCBhcyBDb3JlQ29uc3RydWN0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgRGF0YWJhc2VRdWVyeVByb3BzPEhhbmRsZXJQcm9wcz4gZXh0ZW5kcyBEYXRhYmFzZU9wdGlvbnMge1xuICByZWFkb25seSBoYW5kbGVyOiBzdHJpbmc7XG4gIHJlYWRvbmx5IHByb3BlcnRpZXM6IEhhbmRsZXJQcm9wcztcbiAgLyoqXG4gICAqIFRoZSBwb2xpY3kgdG8gYXBwbHkgd2hlbiB0aGlzIHJlc291cmNlIGlzIHJlbW92ZWQgZnJvbSB0aGUgYXBwbGljYXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IGNkay5SZW1vdmFsUG9saWN5LkRlc3Ryb3lcbiAgICovXG4gIHJlYWRvbmx5IHJlbW92YWxQb2xpY3k/OiBjZGsuUmVtb3ZhbFBvbGljeTtcbn1cblxuZXhwb3J0IGNsYXNzIERhdGFiYXNlUXVlcnk8SGFuZGxlclByb3BzPiBleHRlbmRzIENvcmVDb25zdHJ1Y3QgaW1wbGVtZW50cyBpYW0uSUdyYW50YWJsZSB7XG4gIHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBpYW0uSVByaW5jaXBhbDtcbiAgcmVhZG9ubHkgcmVmOiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSByZWFkb25seSByZXNvdXJjZTogY2RrLkN1c3RvbVJlc291cmNlO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBEYXRhYmFzZVF1ZXJ5UHJvcHM8SGFuZGxlclByb3BzPikge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBhZG1pblVzZXIgPSB0aGlzLmdldEFkbWluVXNlcihwcm9wcyk7XG4gICAgY29uc3QgaGFuZGxlciA9IG5ldyBsYW1iZGEuU2luZ2xldG9uRnVuY3Rpb24odGhpcywgJ0hhbmRsZXInLCB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgJ2RhdGFiYXNlLXF1ZXJ5LXByb3ZpZGVyJykpLFxuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEyX1gsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICB0aW1lb3V0OiBjZGsuRHVyYXRpb24ubWludXRlcygxKSxcbiAgICAgIHV1aWQ6ICczZGU1YmVhNy0yN2RhLTQ3OTYtODY2Mi01ZWZiNTY0MzFiNWYnLFxuICAgICAgbGFtYmRhUHVycG9zZTogJ1F1ZXJ5IFJlZHNoaWZ0IERhdGFiYXNlJyxcbiAgICB9KTtcbiAgICBoYW5kbGVyLmFkZFRvUm9sZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBbJ3JlZHNoaWZ0LWRhdGE6RGVzY3JpYmVTdGF0ZW1lbnQnLCAncmVkc2hpZnQtZGF0YTpFeGVjdXRlU3RhdGVtZW50J10sXG4gICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgIH0pKTtcbiAgICBhZG1pblVzZXIuZ3JhbnRSZWFkKGhhbmRsZXIpO1xuXG4gICAgY29uc3QgcHJvdmlkZXIgPSBuZXcgY3VzdG9tcmVzb3VyY2VzLlByb3ZpZGVyKHRoaXMsICdQcm92aWRlcicsIHtcbiAgICAgIG9uRXZlbnRIYW5kbGVyOiBoYW5kbGVyLFxuICAgIH0pO1xuXG4gICAgY29uc3QgcXVlcnlIYW5kbGVyUHJvcHM6IERhdGFiYXNlUXVlcnlIYW5kbGVyUHJvcHMgJiBIYW5kbGVyUHJvcHMgPSB7XG4gICAgICBoYW5kbGVyOiBwcm9wcy5oYW5kbGVyLFxuICAgICAgY2x1c3Rlck5hbWU6IHByb3BzLmNsdXN0ZXIuY2x1c3Rlck5hbWUsXG4gICAgICBhZG1pblVzZXJBcm46IGFkbWluVXNlci5zZWNyZXRBcm4sXG4gICAgICBkYXRhYmFzZU5hbWU6IHByb3BzLmRhdGFiYXNlTmFtZSxcbiAgICAgIC4uLnByb3BzLnByb3BlcnRpZXMsXG4gICAgfTtcbiAgICB0aGlzLnJlc291cmNlID0gbmV3IGNkay5DdXN0b21SZXNvdXJjZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICByZXNvdXJjZVR5cGU6ICdDdXN0b206OlJlZHNoaWZ0RGF0YWJhc2VRdWVyeScsXG4gICAgICBzZXJ2aWNlVG9rZW46IHByb3ZpZGVyLnNlcnZpY2VUb2tlbixcbiAgICAgIHJlbW92YWxQb2xpY3k6IHByb3BzLnJlbW92YWxQb2xpY3ksXG4gICAgICBwcm9wZXJ0aWVzOiBxdWVyeUhhbmRsZXJQcm9wcyxcbiAgICB9KTtcblxuICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwgPSBoYW5kbGVyLmdyYW50UHJpbmNpcGFsO1xuICAgIHRoaXMucmVmID0gdGhpcy5yZXNvdXJjZS5yZWY7XG4gIH1cblxuICBwdWJsaWMgYXBwbHlSZW1vdmFsUG9saWN5KHBvbGljeTogY2RrLlJlbW92YWxQb2xpY3kpOiB2b2lkIHtcbiAgICB0aGlzLnJlc291cmNlLmFwcGx5UmVtb3ZhbFBvbGljeShwb2xpY3kpO1xuICB9XG5cbiAgcHVibGljIGdldEF0dChhdHRyaWJ1dGVOYW1lOiBzdHJpbmcpOiBjZGsuUmVmZXJlbmNlIHtcbiAgICByZXR1cm4gdGhpcy5yZXNvdXJjZS5nZXRBdHQoYXR0cmlidXRlTmFtZSk7XG4gIH1cblxuICBwdWJsaWMgZ2V0QXR0U3RyaW5nKGF0dHJpYnV0ZU5hbWU6IHN0cmluZyk6IHN0cmluZyB7XG4gICAgcmV0dXJuIHRoaXMucmVzb3VyY2UuZ2V0QXR0U3RyaW5nKGF0dHJpYnV0ZU5hbWUpO1xuICB9XG5cbiAgcHJpdmF0ZSBnZXRBZG1pblVzZXIocHJvcHM6IERhdGFiYXNlT3B0aW9ucyk6IHNlY3JldHNtYW5hZ2VyLklTZWNyZXQge1xuICAgIGNvbnN0IGNsdXN0ZXIgPSBwcm9wcy5jbHVzdGVyO1xuICAgIGxldCBhZG1pblVzZXIgPSBwcm9wcy5hZG1pblVzZXI7XG4gICAgaWYgKCFhZG1pblVzZXIpIHtcbiAgICAgIGlmIChjbHVzdGVyIGluc3RhbmNlb2YgQ2x1c3Rlcikge1xuICAgICAgICBpZiAoY2x1c3Rlci5zZWNyZXQpIHtcbiAgICAgICAgICBhZG1pblVzZXIgPSBjbHVzdGVyLnNlY3JldDtcbiAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoXG4gICAgICAgICAgICAnQWRtaW5pc3RyYXRpdmUgYWNjZXNzIHRvIHRoZSBSZWRzaGlmdCBjbHVzdGVyIGlzIHJlcXVpcmVkIGJ1dCBhbiBhZG1pbiB1c2VyIHNlY3JldCB3YXMgbm90IHByb3ZpZGVkIGFuZCB0aGUgY2x1c3RlciBkaWQgbm90IGdlbmVyYXRlIGFkbWluIHVzZXIgY3JlZGVudGlhbHMgKHRoZXkgd2VyZSBwcm92aWRlZCBleHBsaWNpdGx5KScsXG4gICAgICAgICAgKTtcbiAgICAgICAgfVxuICAgICAgfSBlbHNlIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKFxuICAgICAgICAgICdBZG1pbmlzdHJhdGl2ZSBhY2Nlc3MgdG8gdGhlIFJlZHNoaWZ0IGNsdXN0ZXIgaXMgcmVxdWlyZWQgYnV0IGFuIGFkbWluIHVzZXIgc2VjcmV0IHdhcyBub3QgcHJvdmlkZWQgYW5kIHRoZSBjbHVzdGVyIHdhcyBpbXBvcnRlZCcsXG4gICAgICAgICk7XG4gICAgICB9XG4gICAgfVxuICAgIHJldHVybiBhZG1pblVzZXI7XG4gIH1cbn1cbiJdfQ==