# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['aws_lambda_powertools',
 'aws_lambda_powertools.helper',
 'aws_lambda_powertools.logging',
 'aws_lambda_powertools.metrics',
 'aws_lambda_powertools.middleware_factory',
 'aws_lambda_powertools.tracing']

package_data = \
{'': ['*']}

install_requires = \
['aws-xray-sdk>=2.5.0,<2.6.0', 'fastjsonschema>=2.14.4,<2.15.0']

setup_kwargs = {
    'name': 'aws-lambda-powertools',
    'version': '0.9.1',
    'description': 'Python utilities for AWS Lambda functions including but not limited to tracing, logging and custom metric',
    'long_description': '# Lambda Powertools\n\n![PackageStatus](https://img.shields.io/static/v1?label=status&message=beta&color=blueviolet?style=flat-square) ![PythonSupport](https://img.shields.io/static/v1?label=python&message=3.6%20|%203.7|%203.8&color=blue?style=flat-square&logo=python) ![PyPI version](https://badge.fury.io/py/aws-lambda-powertools.svg) ![PyPi monthly downloads](https://img.shields.io/pypi/dm/aws-lambda-powertools) ![Build](https://github.com/awslabs/aws-lambda-powertools/workflows/Powertools%20Python/badge.svg?branch=master)\n\nA suite of utilities for AWS Lambda Functions that makes tracing with AWS X-Ray, structured logging, and creating custom metrics asynchronously easier - Compatible with Python >=3.6.\n\n> During beta, this library may change its API/methods, or environment variables as it receives feedback from customers.\n\n* **Status**: Beta\n* **How long until GA?**: [Current progress](https://github.com/awslabs/aws-lambda-powertools/projects/1)\n\n## Features\n\n**[Tracing](###Tracing)**\n\n* Capture cold start as annotation, and response and exceptions as metadata\n* Run functions locally with SAM CLI without code change to disable tracing\n* Explicitly disable tracing via env var `POWERTOOLS_TRACE_DISABLED="true"`\n* Support tracing async methods\n\n**[Logging](###Logging)**\n\n* Capture key fields from Lambda context, cold start and structures logging output as JSON\n* Log Lambda event when instructed (disabled by default)\n    - Enable via `POWERTOOLS_LOGGER_LOG_EVENT="true"` or explicitly via decorator param\n* Log sampling enables DEBUG log level for a percentage of requests (disabled by default)\n    - Enable via `POWERTOOLS_LOGGER_SAMPLE_RATE=0.1`, ranges from 0 to 1, where 0.1 is 10% and 1 is 100%\n* Append additional keys to structured log at any point in time\n\n**[Metrics](###Metrics)**\n\n* Aggregate up to 100 metrics using a single CloudWatch Embedded Metric Format object (large JSON blob)\n* Context manager to create an one off metric with a different dimension than metrics already aggregated\n* Validate against common metric definitions mistakes (metric unit, values, max dimensions, max metrics, etc)\n\n**[Bring your own middleware](###Bring-your-own-middleware)**\n\n* Utility to easily create your own middleware\n* Run logic before, after, and handle exceptions\n* Receive lambda handler, event, context\n* Optionally create sub-segment for each custom middleware\n\n**Environment variables** used across suite of utilities\n\nEnvironment variable | Description | Default | Utility\n------------------------------------------------- | --------------------------------------------------------------------------------- | --------------------------------------------------------------------------------- | -------------------------------------------------\nPOWERTOOLS_SERVICE_NAME | Sets service name used for tracing namespace, metrics dimensions and structured logging | "service_undefined" | all\nPOWERTOOLS_TRACE_DISABLED | Disables tracing | "false" | [Tracing](###Tracing)\nPOWERTOOLS_TRACE_MIDDLEWARES | Creates sub-segment for each middleware created by lambda_handler_decorator | "false" | [middleware_factory](###Bring-your-own-middleware)\nPOWERTOOLS_LOGGER_LOG_EVENT | Logs incoming event | "false" | [Logging](###Logging)\nPOWERTOOLS_LOGGER_SAMPLE_RATE | Debug log sampling  | 0 | [Logging](###Logging)\nPOWERTOOLS_METRICS_NAMESPACE | Metrics namespace  | None | [Metrics](###Metrics)\nLOG_LEVEL | Sets logging level | "INFO" | [Logging](###Logging)\n\n## Usage\n\nSee **[example](./example/README.md)** of all features, testing, and a SAM template with all Powertools env vars. All features also provide full docs, and code completion for VSCode and PyCharm.\n\n### Installation\n\nWith [pip](https://pip.pypa.io/en/latest/index.html) installed, run: ``pip install aws-lambda-powertools``\n\n### Tracing\n\n#### Tracing Lambda handler and a function\n\n```python\nfrom aws_lambda_powertools.tracing import Tracer\ntracer = Tracer()\n# tracer = Tracer(service="payment") # can also be explicitly defined\n\n@tracer.capture_method\ndef collect_payment(charge_id):\n  ret = requests.post(PAYMENT_ENDPOINT) # logic\n  tracer.put_annotation("PAYMENT_STATUS", "SUCCESS") # custom annotation\n  return ret\n\n@tracer.capture_lambda_handler\ndef handler(event, context)\n  charge_id = event.get(\'charge_id\')\n  payment = collect_payment(charge_id)\n  ...\n```\n\n#### Tracing asynchronous functions\n\n```python\nimport asyncio\n\nfrom aws_lambda_powertools.tracing import Tracer\ntracer = Tracer()\n# tracer = Tracer(service="payment") # can also be explicitly defined\n\n@tracer.capture_method\nasync def collect_payment(charge_id):\n    ...\n\n@tracer.capture_lambda_handler\ndef handler(event, context)\n  charge_id = event.get(\'charge_id\')\n  payment = asyncio.run(collect_payment(charge_id)) # python 3.7+  \n  ...\n```\n\n#### Tracing concurrent asynchronous with gather\n\n:warning: This will no longer be necessary after [this X-Ray recorder issue is resolved](https://github.com/aws/aws-xray-sdk-python/issues/164) as it\'s an edge case. :warning:\n\nTo safely workaround this issue, use `@tracer.capture_method` on functions not being run with `async.gather`, and instead use `in_subsegment_async` context manager escape hatch to have the same tracing effect.\n\n\n```python\nimport asyncio\n\nfrom aws_lambda_powertools.tracing import Tracer\ntracer = Tracer()\n# tracer = Tracer(service="payment") # can also be explicitly defined\n\nasync def another_async_task():\n    async with tracer.provider.in_subsegment_async("## another_async_task"):\n        ...\n\nasync def another_async_task_2():\n    async with tracer.provider.in_subsegment_async("## another_async_task_2"):\n        ...\n\n@tracer.capture_method\nasync def collect_payment(charge_id):\n    asyncio.gather(another_async_task(), another_async_task_2())\n    ...\n\n@tracer.capture_lambda_handler\ndef handler(event, context)\n  charge_id = event.get(\'charge_id\')\n  payment = asyncio.run(collect_payment(charge_id)) # python 3.7+  \n  ...\n```\n\n#### Using escape hatch mechanisms\n\nYou can use `tracer.provider` attribute to access all methods provided by `xray_recorder`. This is useful when you need a feature available in X-Ray that is not available in the Tracer middleware, for example [thread-safe](https://github.com/aws/aws-xray-sdk-python/#user-content-trace-threadpoolexecutor), or [context managers](https://github.com/aws/aws-xray-sdk-python/#user-content-start-a-custom-segmentsubsegment).\n\n**Example using aiohttp with an async context manager**\n\n```python\nimport asyncio\nimport aiohttp\n\nfrom aws_lambda_powertools.tracing import Tracer, aiohttp_trace_config\ntracer = Tracer()\n\n# aiohttp_trace_config is x-ray extension for aiohttp trace config known as aws_xray_trace_config\n\nasync def aiohttp_task():\n    # Async context manager as opposed to `@tracer.capture_method`\n    async with tracer.provider.in_subsegment_async("## aiohttp escape hatch"):\n        async with aiohttp.ClientSession(trace_configs=[aiohttp_trace_config()]) as session:\n            async with session.get("https://httpbin.org/json") as resp:\n                resp = await resp.json()\n                return resp\n\n@tracer.capture_method\nasync def async_tasks():\n    ret = await aiohttp_task()\n    ...\n\n    return {\n        "task": "done",\n        **ret\n    }\n\n@tracer.capture_lambda_handler\ndef handler(event, context)\n  ret = asyncio.run(async_tasks()) # python 3.7+  \n  ...\n```\n\n#### Using a pre-configured tracer anywhere\n\n```python\n# handler.py\nfrom aws_lambda_powertools.tracing import Tracer\ntracer = Tracer(service="payment")\n\n@tracer.capture_lambda_handler\ndef handler(event, context)\n  charge_id = event.get(\'charge_id\')\n  payment = collect_payment(charge_id)\n  ...\n\n# another_file.py\nfrom aws_lambda_powertools.tracing import Tracer\ntracer = Tracer(auto_patch=False) # new instance using existing configuration with auto patching overriden\n```\n\n### Logging\n\n#### Structuring logs with Lambda context info\n\n```python\nfrom aws_lambda_powertools.logging import Logger\n\nlogger = Logger()\n# Logger(service="payment", level="INFO") # also accepts explicit service name, log level\n\n@logger.inject_lambda_context\ndef handler(event, context)\n  logger.info("Collecting payment")\n  ...\n  # You can log entire objects too\n  logger.info({\n    "operation": "collect_payment",\n    "charge_id": event[\'charge_id\']\n  })\n  ...\n```\n\n<details>\n<summary>Exerpt output in CloudWatch Logs</summary>\n\n```json\n{  \n   "timestamp":"2019-08-22 18:17:33,774",\n   "level":"INFO",\n   "location":"collect.handler:1",\n   "service":"payment",\n   "lambda_function_name":"test",\n   "lambda_function_memory_size":"128",\n   "lambda_function_arn":"arn:aws:lambda:eu-west-1:12345678910:function:test",\n   "lambda_request_id":"52fdfc07-2182-154f-163f-5f0f9a621d72",\n   "cold_start": "true",\n   "sampling_rate": 0.1,\n   "message": "Collecting payment"\n}\n\n{  \n   "timestamp":"2019-08-22 18:17:33,774",\n   "level":"INFO",\n   "location":"collect.handler:15",\n   "service":"payment",\n   "lambda_function_name":"test",\n   "lambda_function_memory_size":"128",\n   "lambda_function_arn":"arn:aws:lambda:eu-west-1:12345678910:function:test",\n   "lambda_request_id":"52fdfc07-2182-154f-163f-5f0f9a621d72",\n   "cold_start": "true",\n   "sampling_rate": 0.1,\n   "message":{  \n      "operation":"collect_payment",\n      "charge_id": "ch_AZFlk2345C0"\n   }\n}\n```\n</details>\n\n#### Appending additional keys to current logger\n\n```python\nfrom aws_lambda_powertools.logging import Logger\n\nlogger = Logger()\n\n@logger.inject_lambda_context\ndef handler(event, context)\n  if "order_id" in event:\n      logger.structure_logs(append=True, order_id=event["order_id"])\n  logger.info("Collecting payment")\n  ...\n```\n\n<details>\n<summary>Exerpt output in CloudWatch Logs</summary>\n\n```json\n{  \n   "timestamp":"2019-08-22 18:17:33,774",\n   "level":"INFO",\n   "location":"collect.handler:1",\n   "service":"payment",\n   "lambda_function_name":"test",\n   "lambda_function_memory_size":"128",\n   "lambda_function_arn":"arn:aws:lambda:eu-west-1:12345678910:function:test",\n   "lambda_request_id":"52fdfc07-2182-154f-163f-5f0f9a621d72",\n   "cold_start": "true",\n   "sampling_rate": 0.1,\n   "order_id": "order_id_value",\n   "message": "Collecting payment"\n}\n```\n</details>\n\n### Metrics\n\nThis feature makes use of CloudWatch Embedded Metric Format (EMF), and metrics are created asynchronously by CloudWatch service.\n\nMetrics middleware validates against the minimum necessary for a metric to be published:\n\n* At least of one Metric and Dimension \n* Maximum of 9 dimensions\n* Only one Namespace\n* [Any Metric unit supported by CloudWatch](https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_MetricDatum.html)\n\n#### Creating multiple metrics\n\nIf using multiple middlewares, use `log_metrics` as the last decorator, or else it will fail with `SchemaValidationError` if no metrics are recorded.\n\n```python\nfrom aws_lambda_powertools.metrics import Metrics, MetricUnit\n\nmetrics = Metrics()\nmetrics.add_namespace(name="ServerlessAirline")\nmetrics.add_metric(name="ColdStart", unit="Count", value=1)\nmetrics.add_dimension(name="service", value="booking")\n\n@metrics.log_metrics\n@tracer.capture_lambda_handler\ndef lambda_handler(evt, ctx):\n    metrics.add_metric(name="BookingConfirmation", unit="Count", value=1)\n    some_code()\n    return True\n\ndef some_code():\n    metrics.add_metric(name="some_other_metric", unit=MetricUnit.Seconds, value=1)\n    ...\n```\n\nCloudWatch EMF uses the same dimensions across all metrics. If you have metrics that should have different dimensions, use `single_metric` to create a single metric with any dimension you want. Generally, this would be an edge case since you [pay for unique metric](https://aws.amazon.com/cloudwatch/pricing/) \n\n> unique metric = (metric_name + dimension_name + dimension_value)\n\n```python\nfrom aws_lambda_powertools.metrics import MetricUnit, single_metric\n\nwith single_metric(name="ColdStart", unit=MetricUnit.Count, value=1) as metric:\n    metric.add_dimension(name="function_context", value="$LATEST")\n```\n\n> **NOTE**: When using Metrics() in multiple places in your code, make sure to use `POWERTOOLS_METRICS_NAMESPACE` env var, or setting namespace param.\n\n### Bring your own middleware\n\nThis feature allows you to create your own middleware as a decorator with ease by following a simple signature. \n\n* Accept 3 mandatory args - `handler, event, context` \n* Always return the handler with event/context or response if executed\n  - Supports nested middleware/decorators use case\n\n#### Middleware with no params\n\n```python\nfrom aws_lambda_powertools.middleware_factory import lambda_handler_decorator\n\n@lambda_handler_decorator\ndef middleware_name(handler, event, context):\n    return handler(event, context)\n\n@lambda_handler_decorator\ndef middleware_before_after(handler, event, context):\n    logic_before_handler_execution()\n    response = handler(event, context)\n    logic_after_handler_execution()\n    return response\n\n\n# middleware_name will wrap Lambda handler \n# and simply return the handler as we\'re not pre/post-processing anything\n# then middleware_before_after will wrap middleware_name\n# run some code before/after calling the handler returned by middleware_name\n# This way, lambda_handler is only actually called once (top-down)\n@middleware_before_after # This will run last\n@middleware_name # This will run first\ndef lambda_handler(event, context):\n    return True\n```\n\n#### Middleware with params\n\n```python\n@lambda_handler_decorator\ndef obfuscate_sensitive_data(handler, event, context, fields=None):\n    # Obfuscate email before calling Lambda handler\n    if fields:\n        for field in fields:\n            field = event.get(field, "")\n            event[field] = obfuscate_pii(field)\n\n    return handler(event, context)\n\n@obfuscate_sensitive_data(fields=["email"])\ndef lambda_handler(event, context):\n    return True\n```\n\n#### Tracing middleware execution\n\nThis makes use of an existing Tracer instance that you may have initialized anywhere in your code. If no Tracer instance is found,  it\'ll initialize one using default options.\n\n```python\nfrom aws_lambda_powertools.middleware_factory import lambda_handler_decorator\n\n@lambda_handler_decorator(trace_execution=True)\ndef middleware_name(handler, event, context):\n    return handler(event, context)\n\n@middleware_name\ndef lambda_handler(event, context):\n    return True\n```\n\nOptionally, you can enrich the final trace with additional annotations and metadata by retrieving a copy of the Tracer used.\n\n```python\nfrom aws_lambda_powertools.middleware_factory import lambda_handler_decorator\nfrom aws_lambda_powertools.tracing import Tracer\n\n@lambda_handler_decorator(trace_execution=True)\ndef middleware_name(handler, event, context):\n    tracer = Tracer() # Takes a copy of an existing tracer instance\n    tracer.add_anotation...\n    tracer.metadata...\n    return handler(event, context)\n\n@middleware_name\ndef lambda_handler(event, context):\n    return True\n```\n\n### Debug mode\n\nBy default, all log statements from AWS Lambda Powertools package are suppressed. If you\'d like to enable them, use `set_package_logger` utility:\n\n```python\nimport aws_lambda_powertools\naws_lambda_powertools.logging.logger.set_package_logger()\n...\n```\n',
    'author': 'Amazon Web Services',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/awslabs/aws-lambda-powertools/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
