"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const s3 = require("aws-cdk-lib/aws-s3");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test Case 2 - Test the getter methods
// --------------------------------------------------------------
test('Test properties', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    };
    const app = new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-kinesis-firehose-kinesis-analytics', props);
    // Assertions
    expect(app.kinesisAnalytics !== null);
    expect(app.kinesisFirehose !== null);
    expect(app.kinesisFirehoseRole !== null);
    expect(app.kinesisFirehoseLogGroup !== null);
    expect(app.s3Bucket !== null);
    expect(app.s3LoggingBucket !== null);
});
// --------------------------------------------------------------
// Test Case 3 - Override kinesisFirehose props
// --------------------------------------------------------------
test('test kinesisFirehose override ', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-firehose-s3', {
        kinesisFirehoseProps: {
            extendedS3DestinationConfiguration: {
                bufferingHints: {
                    intervalInSeconds: 600,
                    sizeInMBs: 55
                },
            }
        },
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        ExtendedS3DestinationConfiguration: {
            BufferingHints: {
                IntervalInSeconds: 600,
                SizeInMBs: 55
            }
        }
    });
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Test bad call with existingBucket and bucketProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testBucket = new s3.Bucket(stack, 'test-bucket', {});
    const app = () => {
        // Helper declaration
        new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, "bad-s3-args", {
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// s3 bucket with bucket, loggingBucket, and auto delete objects
// --------------------------------------------------------------
test('s3 bucket with bucket, loggingBucket, and auto delete objects', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'kinsisfirehose-s3-analytics', {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        },
        loggingBucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        AccessControl: "LogDeliveryWrite"
    });
    expect(stack).toHaveResource("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "kinsisfirehoses3analyticsKinesisFirehoseToS3S3LoggingBucket6EE455EF"
        }
    });
});
// --------------------------------------------------------------
// s3 bucket with one content bucket and no logging bucket
// --------------------------------------------------------------
test('s3 bucket with one content bucket and no logging bucket', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'kinsisfirehose-s3-analytics', {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        },
        kinesisFirehoseProps: {
            deliveryStreamType: 'KinesisStreamAsSource'
        },
        bucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        },
        logS3AccessLogs: false
    });
    expect(stack).toCountResources("AWS::S3::Bucket", 1);
});
//# sourceMappingURL=data:application/json;base64,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