"use strict";
const assert_1 = require("@aws-cdk/assert");
const kinesis = require("@aws-cdk/aws-kinesis");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const sources = require("../lib");
const test_function_1 = require("./test-function");
module.exports = {
    'sufficiently complex example'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        fn.addEventSource(new sources.KinesisEventSource(stream, {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            'PolicyDocument': {
                'Statement': [
                    {
                        'Action': [
                            'kinesis:DescribeStream',
                            'kinesis:GetRecords',
                            'kinesis:GetShardIterator'
                        ],
                        'Effect': 'Allow',
                        'Resource': {
                            'Fn::GetAtt': [
                                'S509448A1',
                                'Arn'
                            ]
                        }
                    }
                ],
                'Version': '2012-10-17'
            },
            'PolicyName': 'FnServiceRoleDefaultPolicyC6A839BF',
            'Roles': [{
                    'Ref': 'FnServiceRoleB9001A96'
                }]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            'EventSourceArn': {
                'Fn::GetAtt': [
                    'S509448A1',
                    'Arn'
                ]
            },
            'FunctionName': {
                'Ref': 'Fn9270CBC0'
            },
            'BatchSize': 100,
            'StartingPosition': 'TRIM_HORIZON'
        }));
        test.done();
    },
    'specific batch size'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        fn.addEventSource(new sources.KinesisEventSource(stream, {
            batchSize: 50,
            startingPosition: lambda.StartingPosition.LATEST
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            'EventSourceArn': {
                'Fn::GetAtt': [
                    'S509448A1',
                    'Arn'
                ]
            },
            'FunctionName': {
                'Ref': 'Fn9270CBC0'
            },
            'BatchSize': 50,
            'StartingPosition': 'LATEST'
        }));
        test.done();
    },
    'fails if batch size < 1'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        test.throws(() => fn.addEventSource(new sources.KinesisEventSource(stream, {
            batchSize: 0,
            startingPosition: lambda.StartingPosition.LATEST
        })), /Maximum batch size must be between 1 and 10000 inclusive \(given 0\)/);
        test.done();
    },
    'fails if batch size > 10000'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        test.throws(() => fn.addEventSource(new sources.KinesisEventSource(stream, {
            batchSize: 10001,
            startingPosition: lambda.StartingPosition.LATEST
        })), /Maximum batch size must be between 1 and 10000 inclusive \(given 10001\)/);
        test.done();
    },
    'specific maxBatchingWindow'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        // WHEN
        fn.addEventSource(new sources.KinesisEventSource(stream, {
            maxBatchingWindow: cdk.Duration.minutes(2),
            startingPosition: lambda.StartingPosition.LATEST
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            'EventSourceArn': {
                'Fn::GetAtt': [
                    'S509448A1',
                    'Arn'
                ]
            },
            'FunctionName': {
                'Ref': 'Fn9270CBC0'
            },
            'MaximumBatchingWindowInSeconds': 120,
            'StartingPosition': 'LATEST'
        }));
        test.done();
    },
    'contains eventSourceMappingId after lambda binding'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const stream = new kinesis.Stream(stack, 'S');
        const eventSource = new sources.KinesisEventSource(stream, {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON
        });
        // WHEN
        fn.addEventSource(eventSource);
        // THEN
        test.ok(eventSource.eventSourceMappingId);
        test.done();
    },
    'eventSourceMappingId throws error before binding to lambda'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const stream = new kinesis.Stream(stack, 'S');
        const eventSource = new sources.KinesisEventSource(stream, {
            startingPosition: lambda.StartingPosition.TRIM_HORIZON
        });
        // WHEN/THEN
        test.throws(() => eventSource.eventSourceMappingId, /KinesisEventSource is not yet bound to an event source mapping/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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