"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route53ToAlb = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const defaults = require("@aws-solutions-constructs/core");
const r53 = require("@aws-cdk/aws-route53");
const r53t = require("@aws-cdk/aws-route53-targets");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const core_1 = require("@aws-cdk/core");
/**
 * @summary Configures a Route53 Hosted Zone to route to an Application Load Balancer
 */
class Route53ToAlb extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the Route53ToAlb class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {Route53ToAlbProps} props - user provided props for the construct.
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckProps(props);
        // NOTE: We don't call CheckAlbProps() here, because this construct creates an ALB
        // with no listener or target, so some of those checks don't apply
        if (props?.loadBalancerProps?.vpc) {
            throw new Error('Specify any existing VPC at the construct level, not within loadBalancerProps.');
        }
        if (props.existingLoadBalancerObj && !props.existingVpc) {
            throw new Error('An existing ALB already exists in a VPC, so that VPC must be passed to the construct in props.existingVpc');
        }
        if (props.existingHostedZoneInterface && !props.publicApi && !props.existingVpc) {
            throw new Error('An existing Private Hosted Zone already exists in a VPC, so that VPC must be passed to the construct in props.existingVpc');
        }
        this.vpc = defaults.buildVpc(scope, {
            existingVpc: props.existingVpc,
            defaultVpcProps: props.publicApi ?
                defaults.DefaultPublicPrivateVpcProps() :
                // If this is an internal app, we're going to turn on DNS
                // by default to allow gateway and interface service endpoints
                defaults.overrideProps(defaults.DefaultIsolatedVpcProps(), { enableDnsHostnames: true, enableDnsSupport: true, }),
            userVpcProps: props.vpcProps,
        });
        if (props.existingHostedZoneInterface) {
            this.hostedZone = props.existingHostedZoneInterface;
        }
        else {
            if (props.publicApi) {
                throw new Error('Public APIs require an existingHostedZone be passed in the Props object.');
            }
            else {
                if (!props.privateHostedZoneProps) {
                    throw new Error('Must supply privateHostedZoneProps to create a private API');
                }
                if (props.privateHostedZoneProps.vpc) {
                    throw new Error('All VPC specs must be provided at the Construct level in Route53ToAlbProps');
                }
                const manufacturedProps = defaults.overrideProps(props.privateHostedZoneProps, { vpc: this.vpc });
                this.hostedZone = new r53.PrivateHostedZone(this, `${id}-zone`, manufacturedProps);
            }
        }
        this.loadBalancer = defaults.ObtainAlb(this, id, this.vpc, props.publicApi, props.existingLoadBalancerObj, props.loadBalancerProps, props.logAlbAccessLogs, props.albLoggingBucketProps);
        // Add the ALB to the HostedZone as a target
        const hostedZoneTarget = new r53t.LoadBalancerTarget(this.loadBalancer);
        new r53.ARecord(this, `${id}-alias`, {
            target: { aliasTarget: hostedZoneTarget },
            zone: this.hostedZone
        });
    }
}
exports.Route53ToAlb = Route53ToAlb;
_a = JSII_RTTI_SYMBOL_1;
Route53ToAlb[_a] = { fqn: "@aws-solutions-constructs/aws-route53-alb.Route53ToAlb", version: "1.175.0" };
//# sourceMappingURL=data:application/json;base64,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