"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const r53 = require("aws-cdk-lib/aws-route53");
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
test('Test Public API, new VPC', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const newZone = new r53.PublicHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com'
    });
    const props = {
        publicApi: true,
        existingHostedZoneInterface: newZone,
    };
    new lib_1.Route53ToAlb(stack, 'test-route53-alb', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    expect(stack).toHaveResourceLike('AWS::Route53::RecordSet', {
        Name: 'www.example-test.com.',
        Type: 'A'
    });
});
test('Test Private API, existing VPC', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const newZone = new r53.PrivateHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com',
        vpc: testExistingVpc
    });
    const props = {
        publicApi: false,
        existingHostedZoneInterface: newZone,
        existingVpc: testExistingVpc
    };
    new lib_1.Route53ToAlb(stack, 'test-route53-alb', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internal'
    });
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    expect(stack).toHaveResourceLike('AWS::Route53::RecordSet', {
        Name: 'www.example-test.com.',
        Type: 'A'
    });
});
test('Test Private API, new VPC', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        publicApi: false,
        privateHostedZoneProps: {
            zoneName: 'www.example-test.com',
        }
    };
    new lib_1.Route53ToAlb(stack, 'test-route53-alb', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internal'
    });
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    expect(stack).toHaveResourceLike('AWS::Route53::RecordSet', {
        Name: 'www.example-test.com.',
        Type: 'A'
    });
});
test('Check publicApi and zone props is an error', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const testExistingVpc = defaults.getTestVpc(stack);
    const props = {
        publicApi: true,
        existingVpc: testExistingVpc,
        privateHostedZoneProps: {
            zoneName: 'www.example-test.com',
        }
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Check no Zone props and no existing zone interface is an error', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const testExistingVpc = defaults.getTestVpc(stack);
    const props = {
        publicApi: false,
        existingVpc: testExistingVpc,
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Check Zone props with VPC is an error', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const testExistingVpc = defaults.getTestVpc(stack);
    const props = {
        publicApi: false,
        existingVpc: testExistingVpc,
        privateHostedZoneProps: {
            zoneName: 'www.example-test.com',
            vpc: testExistingVpc
        }
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Test with privateHostedZoneProps', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const props = {
        publicApi: false,
        existingVpc: testExistingVpc,
        privateHostedZoneProps: {
            zoneName: 'www.example-test.com',
        }
    };
    new lib_1.Route53ToAlb(stack, 'test-error', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internal'
    });
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    expect(stack).toHaveResourceLike('AWS::Route53::RecordSet', {
        Name: 'www.example-test.com.',
        Type: 'A'
    });
});
test('Check that passing an existing hosted Zone without passing an existingVPC is an error', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testExistingVpc = defaults.getTestVpc(stack);
    const newZone = new r53.PrivateHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com',
        vpc: testExistingVpc
    });
    const props = {
        publicApi: false,
        existingHostedZoneInterface: newZone,
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Check that passing an existing Load Balancer without passing an existingVPC is an error', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testExistingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc
    });
    const props = {
        publicApi: false,
        existingLoadBalancerObj: existingAlb,
        privateHostedZoneProps: {
            zoneName: 'www.example-test.com',
        }
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Check that passing an existing ALB without passing an existingVPC is an error', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testExistingVpc = defaults.getTestVpc(stack);
    const newZone = new r53.PrivateHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com',
        vpc: testExistingVpc
    });
    const props = {
        publicApi: false,
        existingHostedZoneInterface: newZone,
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Check that passing loadBalancerProps with a vpc is an error', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testExistingVpc = defaults.getTestVpc(stack);
    const newZone = new r53.PrivateHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com',
        vpc: testExistingVpc
    });
    const props = {
        publicApi: false,
        existingHostedZoneInterface: newZone,
        loadBalancerProps: {
            loadBalancerName: 'my-alb',
            vpc: testExistingVpc,
        }
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
test('Test providing loadBalancerProps', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const newZone = new r53.PrivateHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com',
        vpc: testExistingVpc
    });
    const props = {
        publicApi: false,
        existingHostedZoneInterface: newZone,
        existingVpc: testExistingVpc,
        loadBalancerProps: {
            loadBalancerName: 'find-this-name'
        },
    };
    new lib_1.Route53ToAlb(stack, 'test-route53-alb', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internal',
        Name: 'find-this-name'
    });
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    expect(stack).toHaveResourceLike('AWS::Route53::RecordSet', {
        Name: 'www.example-test.com.',
        Type: 'A'
    });
});
test('Test providing an existingLoadBalancer', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const testExistingVpc = defaults.getTestVpc(stack);
    const newZone = new r53.PrivateHostedZone(stack, 'test-zone', {
        zoneName: 'www.example-test.com',
        vpc: testExistingVpc
    });
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        loadBalancerName: 'find-this-name'
    });
    const props = {
        publicApi: false,
        existingHostedZoneInterface: newZone,
        existingVpc: testExistingVpc,
        existingLoadBalancerObj: existingAlb,
    };
    new lib_1.Route53ToAlb(stack, 'test-route53-alb', props);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internal',
        Name: 'find-this-name'
    });
    expect(stack).toHaveResourceLike('AWS::EC2::VPC', {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    expect(stack).toHaveResourceLike('AWS::Route53::RecordSet', {
        Name: 'www.example-test.com.',
        Type: 'A'
    });
});
test('Check publicApi and without an existing hosted zone is an error', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const testExistingVpc = defaults.getTestVpc(stack);
    const props = {
        publicApi: true,
        existingVpc: testExistingVpc,
    };
    const app = () => {
        new lib_1.Route53ToAlb(stack, 'test-error', props);
    };
    // Assertion
    expect(app).toThrowError();
});
//# sourceMappingURL=data:application/json;base64,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