"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route53ToAlb = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
// Imports
const defaults = require("@aws-solutions-constructs/core");
const r53 = require("aws-cdk-lib/aws-route53");
const r53t = require("aws-cdk-lib/aws-route53-targets");
// Note: To ensure CDKv2 compatibility, keep the import statement for Construct separate
const constructs_1 = require("constructs");
/**
 * @summary Configures a Route53 Hosted Zone to route to an Application Load Balancer
 */
class Route53ToAlb extends constructs_1.Construct {
    /**
     * @summary Constructs a new instance of the Route53ToAlb class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {Route53ToAlbProps} props - user provided props for the construct.
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        defaults.CheckProps(props);
        // NOTE: We don't call CheckAlbProps() here, because this construct creates an ALB
        // with no listener or target, so some of those checks don't apply
        if (props?.loadBalancerProps?.vpc) {
            throw new Error('Specify any existing VPC at the construct level, not within loadBalancerProps.');
        }
        if (props.existingLoadBalancerObj && !props.existingVpc) {
            throw new Error('An existing ALB already exists in a VPC, so that VPC must be passed to the construct in props.existingVpc');
        }
        if (props.existingHostedZoneInterface && !props.publicApi && !props.existingVpc) {
            throw new Error('An existing Private Hosted Zone already exists in a VPC, so that VPC must be passed to the construct in props.existingVpc');
        }
        this.vpc = defaults.buildVpc(scope, {
            existingVpc: props.existingVpc,
            defaultVpcProps: props.publicApi ?
                defaults.DefaultPublicPrivateVpcProps() :
                // If this is an internal app, we're going to turn on DNS
                // by default to allow gateway and interface service endpoints
                defaults.overrideProps(defaults.DefaultIsolatedVpcProps(), { enableDnsHostnames: true, enableDnsSupport: true, }),
            userVpcProps: props.vpcProps,
        });
        if (props.existingHostedZoneInterface) {
            this.hostedZone = props.existingHostedZoneInterface;
        }
        else {
            if (props.publicApi) {
                throw new Error('Public APIs require an existingHostedZone be passed in the Props object.');
            }
            else {
                if (!props.privateHostedZoneProps) {
                    throw new Error('Must supply privateHostedZoneProps to create a private API');
                }
                if (props.privateHostedZoneProps.vpc) {
                    throw new Error('All VPC specs must be provided at the Construct level in Route53ToAlbProps');
                }
                const manufacturedProps = defaults.overrideProps(props.privateHostedZoneProps, { vpc: this.vpc });
                this.hostedZone = new r53.PrivateHostedZone(this, `${id}-zone`, manufacturedProps);
            }
        }
        this.loadBalancer = defaults.ObtainAlb(this, id, this.vpc, props.publicApi, props.existingLoadBalancerObj, props.loadBalancerProps, props.logAlbAccessLogs, props.albLoggingBucketProps);
        // Add the ALB to the HostedZone as a target
        const hostedZoneTarget = new r53t.LoadBalancerTarget(this.loadBalancer);
        new r53.ARecord(this, `${id}-alias`, {
            target: { aliasTarget: hostedZoneTarget },
            zone: this.hostedZone
        });
    }
}
exports.Route53ToAlb = Route53ToAlb;
_a = JSII_RTTI_SYMBOL_1;
Route53ToAlb[_a] = { fqn: "@aws-solutions-constructs/aws-route53-alb.Route53ToAlb", version: "2.31.0" };
//# sourceMappingURL=data:application/json;base64,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