"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ *
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const lambda = require("aws-cdk-lib/aws-lambda");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-solutions-constructs/core");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-cloudfront-apigateway custom Cloudfront Logging Bucket';
const inProps = {
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    runtime: lambda.Runtime.NODEJS_14_X,
    handler: 'index.handler'
};
const func = defaults.deployLambdaFunction(stack, inProps);
const [_api] = defaults.RegionalLambdaRestApi(stack, func);
_api.methods.forEach((apiMethod) => {
    // Override the API Gateway Authorization Type from AWS_IAM to NONE
    const child = apiMethod.node.findChild('Resource');
    if (child.authorizationType === 'AWS_IAM') {
        child.addPropertyOverride('AuthorizationType', 'NONE');
        defaults.addCfnSuppressRules(apiMethod, [
            {
                id: "W59",
                reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`,
            },
        ]);
    }
});
new lib_1.CloudFrontToApiGateway(stack, 'cf-apigw', {
    existingApiGatewayObj: _api,
    cloudFrontLoggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true,
        encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
        versioned: true
    }
});
defaults.suppressAutoDeleteHandlerWarnings(stack);
// Synth
app.synth();
//# sourceMappingURL=data:application/json;base64,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