import { AwsAsset } from "./generated/assets";
import { CfnSpec } from "./generated/cfnspec";
import { DrawioSpec } from "./generated/drawio-spec";
import { ParsedAssetKey } from "./internal/assets/helpers";
import { AwsCategoryDefinition, AwsCategoryId } from "./internal/categories/definitions";
import { DrawioAws4ParentShapes, DrawioAwsResourceIconStyleBase } from "./internal/drawio/types";
import { CfnMappedService, CfnMappedResource } from "./internal/mapping/types";
import { PricingManifest } from "./internal/pricing-manifest";
import { Themes } from "./themes";
/**
 * Available icon formats for assets.
 */
export declare type IconFormats = "png" | "svg";
/**
 * Drawio resource icon style definition for AWS Resources
 * @struct
 */
export interface AwsDrawioResourceIconStyle extends DrawioAwsResourceIconStyleBase {
    readonly shape: DrawioAws4ParentShapes.RESOURCE_ICON;
    readonly resIcon: DrawioSpec.Aws4.ShapeNames;
}
/**
 * Drawio shape based style definition.
 * @struct
 */
export interface AwsDrawioShapeStyle extends DrawioAwsResourceIconStyleBase {
    readonly shape: DrawioSpec.Aws4.ShapeNames;
}
/**
 * AwsCategoryDrawioStyles is a utility class for constructing drawio shape
 * styles for services and resources.
 */
export declare class AwsCategoryDrawioStyles {
    readonly categoryShape: DrawioSpec.Aws4.ShapeNames;
    readonly base: DrawioAwsResourceIconStyleBase;
    constructor(categoryShape: DrawioSpec.Aws4.ShapeNames, base: DrawioAwsResourceIconStyleBase);
    /**
     * Get the drawio style for this category.
     */
    get categoryStyle(): AwsDrawioResourceIconStyle;
    /**
     * Gets the drawio style for a service based on the category style.
     * @param serviceShape The service shape to style based on category
     * @returns {AwsDrawioResourceIconStyle} The style drawio style definition for the resource based on category style.
     */
    getServiceStyle(serviceShape: DrawioSpec.Aws4.ShapeNames): AwsDrawioResourceIconStyle;
    /**
     * Gets the drawio style for a resource based on the category style.
     * @param resourceShape The resource shape to style based on category
     * @returns {AwsDrawioResourceIconStyle} The style drawio style definition for the resource based on category style.
     */
    getResourceStyle(resourceShape: DrawioSpec.Aws4.ShapeNames): AwsDrawioShapeStyle;
}
/**
 * Type for category dictionary.
 */
export declare type TAwsCategoryDict = {
    [key: string]: AwsCategory;
};
/**
 * AwsCategory class provides an interface for normalizing category metadata
 * between mapped systems.
 */
export declare class AwsCategory {
    /** @internal */
    static _register(definition: AwsCategoryDefinition): AwsCategory;
    /**
     * Get {@link AwsCategory} based on {@link AwsCategoryId}
     * @param id {AwsCategoryId} The id of the category to retrieve.
     * @returns Returns the category with the id
     */
    static getCategory(id: AwsCategoryId): AwsCategory;
    /** @internal */
    private static _instanceMap;
    /**
     * Get record of all categories keyed by category id.
     */
    static get categories(): TAwsCategoryDict;
    /**
     * The unique id of the category.
     *
     * @example "security_identity_compliance"
     */
    readonly id: AwsCategoryId;
    /**
     * The proper name of the category.
     *
     * @example "Security, Identity, & Compliance"
     */
    readonly name: string;
    /**
     * Fill color for the category.
     */
    readonly fillColor: string;
    /**
     * Gradien color for the category.
     */
    readonly gradientColor: string;
    /**
     * Font color for the category.
     */
    readonly fontColor: string;
    /**
     * Alternative names used to identity this category.
     */
    readonly variants: string[];
    /**
     * Drawio style definition for this category.
     */
    readonly drawioStyles?: AwsCategoryDrawioStyles;
    /** @internal */
    private readonly _assetIcon?;
    /** @internal */
    private _services?;
    /** @internal */
    private constructor();
    /**
     * Retrieves a well-formatted relative path to the icon for this given
     * category in the specified format.
     */
    icon(format: IconFormats, theme?: Themes): string | undefined;
    /**
     * Gets a list of all services within this category.
     */
    categoryServices(): AwsService[];
}
/**
 * Type for service dictionary.
 */
export declare type TAwsServiceDict = {
    [key: string]: AwsService;
};
/**
 * AwsService class provides an interface for normalizing service metadata
 * between mapped systems.
 */
export declare class AwsService {
    /** @internal */
    static _register(cfnKey: CfnSpec.ServiceName, cfnMapped: CfnMappedService): AwsService;
    /**
     * Get {@link AwsService} by CloudFormation "service" name, where service name is expressed
     * as `<provider>::<service>::<resource>`.
     * @param cfnService The service name to retrieve {@link AwsService} for.
     * @returns Returns the {@link AwsService} associated with the `cfnService` provided
     * @throws Error is service not found
     */
    static getService(cfnService: CfnSpec.ServiceName): AwsService;
    /**
     * Finds the {@link AwsService} associated with a given value.
     * @param value Value to match {@link AwsService}, which can be `id`, `assetKey`, `fullName`, etc.
     * @returns Returns matching {@link AwsService} or `undefined` if not found
     * @throws Error if service not found
     */
    static findService(value: string): AwsService | undefined;
    /** @internal */
    private static _instanceMap;
    /**
     * Get record of all {@link AwsService}s keyed by `id`
     */
    static get services(): TAwsServiceDict;
    /**
     * The category the service belongs to, or undefined if does not belong to a category.
     */
    readonly category?: AwsCategory;
    /**
     * The CloudFormation "provider" for the service, as expressed by `<provicer>::<service>::<resource>`.
     */
    readonly cfnProvider: string;
    /**
     * The CloudFormation "service" for the service, as expressed by `<provicer>::<service>::<resource>`.
     */
    readonly cfnService: string;
    /**
     * Drawio shape associated with this service, or undefined if service not mapped to draiwio shape.
     */
    readonly drawioShape?: DrawioSpec.Aws4.ShapeNames;
    /**
     * The pricing `serviceCode` associated with this service, or undefined if service not mapped to pricing.
     */
    readonly pricingServiceCode?: PricingManifest.ServiceCode;
    /** @internal */
    private readonly _assetKey?;
    /** @internal */
    private readonly _assetIcon?;
    /** @internal */
    private _drawioStyle?;
    /** @internal */
    private _resources?;
    /** @internal */
    private constructor();
    /**
     * The proper full name of the service.
     *
     * @example "AWS Glue", "Amazon S3"
     */
    get fullName(): string;
    /**
     * Get relative asset icon for the service for a given format and optional theme.
     * @param {IconFormats} format - The format of icon.
     * @param {Themes} [theme] - Optional theme
     * @returns Returns relative asset icon path
     */
    icon(format: IconFormats, theme?: Themes): string | undefined;
    /**
     * Get drawio style for this service
     */
    drawioStyle(): AwsDrawioResourceIconStyle | undefined;
    /**
     * List all resources of this service
     */
    serviceResources(): AwsResource[];
    /**
     * Get service pricing metadata.
     */
    get pricingMetadata(): PricingManifest.Service | undefined;
}
/**
 * Type for resource dictionary.
 */
export declare type TAwsResourceDict = {
    [key: string]: AwsResource;
};
/**
 * AwsResource class provides an interface for normalizing resource metadata
 * between mapped systems.
 */
export declare class AwsResource {
    /** @internal */
    static _register(cfnResourceType: CfnSpec.ResourceType, cfnMapped: CfnMappedResource): AwsResource;
    /**
     * Get {@link AwsResource} by CloudFormation resource type.
     * @param cfnResourceType - Fully qualifief CloudFormation resource type (eg: AWS:S3:Bucket)
     * @throws Error is no resource found
     */
    static getResource(cfnResourceType: CfnSpec.ResourceType): AwsResource;
    /**
     * Find {@link AwsResource} associated with given value.
     * @param value - The value to match {@link AwsResource}; can be id, asset key, full name, etc.
     * @throws Error is no resource found
     */
    static findResource(value: string): AwsResource | undefined;
    /** @internal */
    private static _instanceMap;
    /**
     * Get record of all resources keyed by resource id.
     */
    static get resources(): TAwsResourceDict;
    /**
     * Fully-qualified CloudFormation resource type (eg: "AWS:S3:Bucket")
     */
    readonly cfnResourceType: CfnSpec.ResourceType;
    /**
     * The {@link AwsService} the resource belongs to.
     */
    readonly service: AwsService;
    /**
     * The proper full name of the resource.
     *
     * @example "Bucket", "Amazon S3 on Outposts"
     */
    readonly fullName?: string;
    /**
     * The drawio shape mapped to this resource, or undefined if no mapping.
     */
    readonly drawioShape?: DrawioSpec.Aws4.ShapeNames;
    /** @internal */
    private readonly _category?;
    /** @internal */
    private readonly _serviceIcon?;
    /** @internal */
    private readonly _assetIcon?;
    /** @internal */
    private readonly _assetKey?;
    /** @internal */
    private readonly _generalIcon?;
    /** @internal */
    private _drawioStyle?;
    /** @internal */
    private constructor();
    /**
     * Gets the category icon for the resource.
     *
     * This maybe different than {@link AwsResource.service.category.icon} based on mappings overrides, which
     * if do not exist will fallback to {@link AwsResource.service.category.icon}.
     *
     * @param {IconFormats} format - The format of icon.
     * @param {Themes} [theme] - Optional theme
     * @returns Returns relative asset icon path
     *
     * @see {@link AwsService.icon}
     */
    getCategoryIcon(format: IconFormats, theme?: Themes): string | undefined;
    /**
     * Gets the service icon for the resource.
     *
     * This maybe different than {@link AwsResource.service.icon} based on mappings overrides, which
     * if do not exist will fallback to {@link AwsResource.service.icon}.
     *
     * @param {IconFormats} format - The format of icon.
     * @param {Themes} [theme] - Optional theme
     * @returns Returns relative asset icon path
     *
     * @see {@link AwsService.icon}
     */
    getServiceIcon(format: IconFormats, theme?: Themes): string | undefined;
    /**
     * Gets the resource specific icon for the resource.
     * @param {IconFormats} format - The format of icon.
     * @param {Themes} [theme] - Optional theme
     * @returns Returns relative asset icon path or undefined if does not have resource icon
     */
    getResourceIcon(format: IconFormats, theme?: Themes): string | undefined;
    /**
     * Gets the general icon for the resource if available.
     * @param {IconFormats} format - The format of icon.
     * @param {Themes} [theme] - Optional theme
     * @returns Returns relative asset icon path or undefined if does not have general icon
     */
    getGeneralIcon(format: IconFormats, theme?: Themes): string | undefined;
    /**
     * Gets the best icon match for the resource following the order of:
     * 1. explicit resource icon
     * 2. general icon
     * 3. service icon
     * @param {IconFormats} format - The format of icon.
     * @param {Themes} [theme] - Optional theme
     * @returns Returns relative asset icon path
     */
    icon(format: IconFormats, theme?: Themes): string | undefined;
    /**
     * Gets the draiwio style for the resource.
     */
    drawioStyle(): AwsDrawioShapeStyle | undefined;
}
/**
 * AwsArchitecture provides an interface for retrieving the inferred normalization between [@aws-cdk/cfnspec](https://github.com/aws/aws-cdk/blob/main/packages/%40aws-cdk/cfnspec)
 * and [AWS Architecture Icons](https://aws.amazon.com/architecture/icons/) systems
 * for all CloudFormation "services" and "resources".
 */
export declare class AwsArchitecture {
    /**
     * The absolute directory where [AWS Architecture Icons](https://aws.amazon.com/architecture/icons/)
     * are stored and retrieved.
     */
    static get assetDirectory(): string;
    /**
     * Get all categories
     * @see {@link AwsCategory.categories}
     */
    static get categories(): TAwsCategoryDict;
    /**
     * Get all services
     * @see {@link AwsService.services}
     */
    static get services(): TAwsServiceDict;
    /**
     * Get all resources
     * @see {@link AwsResource.resources}
     */
    static get resources(): TAwsResourceDict;
    /**
     * Get specific category based on id
     * @see {@link AwsCategory.getCategory}
     */
    static getCategory(category: AwsCategoryId): AwsCategory;
    /**
     * Get specific service based on identifier (eg: S3, AWS::S3, AWS::S3::Bucket)
     * @see {@link AwsSerfice.getService}
     */
    static getService(identifier: string): AwsService;
    /**
     * Get resource based on Cfn Resource Type (eg: AWS::S3::Bucket)
     * @see {@link AwsResource.getResource}
     */
    static getResource(cfnType: CfnSpec.ResourceType): AwsResource;
    /**
     * Get icon for EC2 instance type.
     * @param instanceType - The {@link AwsAsset.InstanceType} to get icon for
     * @param {IconFormats} format - The format of icon.
     * @param {Themes} [theme] - Optional theme
     * @returns Returns relative asset icon path
     */
    static getInstanceTypeIcon(instanceType: AwsAsset.InstanceType, format?: "png" | "svg", theme?: Themes): string;
    /**
     * Get icon for RDS instance type.
     * @param instanceType - The {@link AwsAsset.RdsInstanceType} to get icon for
     * @param {IconFormats} format - The format of icon.
     * @param {Themes} [theme] - Optional theme
     * @returns Returns relative asset icon path
     */
    static getRdsInstanceTypeIcon(instanceType: AwsAsset.RdsInstanceType, format?: "png" | "svg", theme?: Themes): string;
    /**
     * Resolve relative asset path to absolute asset path.
     * @param assetPath - The relative asset path to resolve.
     * @returns {string} Absolute asset path
     * @throws Error if asset path is undefined
     * @throws Error if asset path is not relative
     */
    static resolveAssetPath(assetPath: string): string;
    /**
     * Resolve relative asset path as SVG [Data URL](https://developer.mozilla.org/en-US/docs/Web/HTTP/Basics_of_HTTP/Data_URLs).
     *
     * `data:image/svg+xml;base64,...`
     * @param svgAssetPath - The relative path of svg asset to resolve
     * @returns SVG [Data URL](https://developer.mozilla.org/en-US/docs/Web/HTTP/Basics_of_HTTP/Data_URLs)
     * @throws Error if path is not svg
     */
    static resolveAssetSvgDataUrl(svgAssetPath: string): string;
    /**
     * Gets formatted asset path including extension and theme.
     * @param qualifiedAssetKey The qualified asset key (eg: compute/ec2/service_icon, storage/s3/bucket)
     * @param format {IconFormats} The format to return (eg: png, svg)
     * @param theme {Themes} - (Optional) The theme to use, if not specific or now matching asset for the them, the default theme is used
     * @returns Relative asset file path
     */
    static formatAssetPath(qualifiedAssetKey: string, format: IconFormats, theme?: Themes): string;
    /**
     * Parse assets path into part descriptor.
     * @param assetPath - Absolute or relative asset file path to parse
     */
    static parseAssetPath(assetPath: string): ParsedAssetKey;
    /** @internal */
    private static _assetDirectory?;
    private constructor();
}
