"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pipeline = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const notifications = require("@aws-cdk/aws-codestarnotifications");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const action_1 = require("./action");
const codepipeline_generated_1 = require("./codepipeline.generated");
const cross_region_support_stack_1 = require("./private/cross-region-support-stack");
const full_action_descriptor_1 = require("./private/full-action-descriptor");
const rich_action_1 = require("./private/rich-action");
const stage_1 = require("./private/stage");
const validation_1 = require("./private/validation");
class PipelineBase extends core_1.Resource {
    /**
     * Defines an event rule triggered by this CodePipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codepipeline'],
            resources: [this.pipelineArn],
        });
        return rule;
    }
    /**
     * Defines an event rule triggered by the "CodePipeline Pipeline Execution
     * State Change" event emitted from this pipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodePipeline Pipeline Execution State Change'],
        });
        return rule;
    }
    bindAsNotificationRuleSource(_scope) {
        return {
            sourceArn: this.pipelineArn,
        };
    }
    notifyOn(id, target, options) {
        return new notifications.NotificationRule(this, id, {
            ...options,
            source: this,
            targets: [target],
        });
    }
    notifyOnExecutionStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_FAILED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_CANCELED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_STARTED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_RESUMED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_SUCCEEDED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_SUPERSEDED,
            ],
        });
    }
    notifyOnAnyStageStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_CANCELED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_FAILED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_RESUMED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_STARTED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_SUCCEEDED,
            ],
        });
    }
    notifyOnAnyActionStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_CANCELED,
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_FAILED,
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_STARTED,
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_SUCCEEDED,
            ],
        });
    }
    notifyOnAnyManualApprovalStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.MANUAL_APPROVAL_FAILED,
                action_1.PipelineNotificationEvents.MANUAL_APPROVAL_NEEDED,
                action_1.PipelineNotificationEvents.MANUAL_APPROVAL_SUCCEEDED,
            ],
        });
    }
}
/**
 * An AWS CodePipeline pipeline with its associated IAM role and S3 bucket.
 *
 * @example
 * // create a pipeline
 * import * as codecommit from '@aws-cdk/aws-codecommit';
 *
 * const pipeline = new codepipeline.Pipeline(this, 'Pipeline');
 *
 * // add a stage
 * const sourceStage = pipeline.addStage({ stageName: 'Source' });
 *
 * // add a source action to the stage
 * declare const repo: codecommit.Repository;
 * declare const sourceArtifact: codepipeline.Artifact;
 * sourceStage.addAction(new codepipeline_actions.CodeCommitSourceAction({
 *   actionName: 'Source',
 *   output: sourceArtifact,
 *   repository: repo,
 * }));
 *
 * // ... add more stages
 */
class Pipeline extends PipelineBase {
    constructor(scope, id, props = {}) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.pipelineName,
        });
        this._stages = new Array();
        this._crossRegionSupport = {};
        this._crossAccountSupport = {};
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codepipeline_PipelineProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        validation_1.validateName('Pipeline', this.physicalName);
        // only one of artifactBucket and crossRegionReplicationBuckets can be supplied
        if (props.artifactBucket && props.crossRegionReplicationBuckets) {
            throw new Error('Only one of artifactBucket and crossRegionReplicationBuckets can be specified!');
        }
        // @deprecated(v2): switch to default false
        this.crossAccountKeys = (_b = props.crossAccountKeys) !== null && _b !== void 0 ? _b : true;
        this.enableKeyRotation = props.enableKeyRotation;
        // Cross account keys must be set for key rotation to be enabled
        if (this.enableKeyRotation && !this.crossAccountKeys) {
            throw new Error("Setting 'enableKeyRotation' to true also requires 'crossAccountKeys' to be enabled");
        }
        this.reuseCrossRegionSupportStacks = (_c = props.reuseCrossRegionSupportStacks) !== null && _c !== void 0 ? _c : true;
        // If a bucket has been provided, use it - otherwise, create a bucket.
        let propsBucket = this.getArtifactBucketFromProps(props);
        if (!propsBucket) {
            let encryptionKey;
            if (this.crossAccountKeys) {
                encryptionKey = new kms.Key(this, 'ArtifactsBucketEncryptionKey', {
                    // remove the key - there is a grace period of a few days before it's gone for good,
                    // that should be enough for any emergency access to the bucket artifacts
                    removalPolicy: core_1.RemovalPolicy.DESTROY,
                    enableKeyRotation: this.enableKeyRotation,
                });
                // add an alias to make finding the key in the console easier
                new kms.Alias(this, 'ArtifactsBucketEncryptionKeyAlias', {
                    aliasName: this.generateNameForDefaultBucketKeyAlias(),
                    targetKey: encryptionKey,
                    removalPolicy: core_1.RemovalPolicy.DESTROY,
                });
            }
            propsBucket = new s3.Bucket(this, 'ArtifactsBucket', {
                bucketName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                encryptionKey,
                encryption: encryptionKey ? s3.BucketEncryption.KMS : s3.BucketEncryption.KMS_MANAGED,
                enforceSSL: true,
                blockPublicAccess: new s3.BlockPublicAccess(s3.BlockPublicAccess.BLOCK_ALL),
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        this.artifactBucket = propsBucket;
        // If a role has been provided, use it - otherwise, create a role.
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com'),
        });
        const codePipeline = new codepipeline_generated_1.CfnPipeline(this, 'Resource', {
            artifactStore: core_1.Lazy.any({ produce: () => this.renderArtifactStoreProperty() }),
            artifactStores: core_1.Lazy.any({ produce: () => this.renderArtifactStoresProperty() }),
            stages: core_1.Lazy.any({ produce: () => this.renderStages() }),
            disableInboundStageTransitions: core_1.Lazy.any({ produce: () => this.renderDisabledTransitions() }, { omitEmptyArray: true }),
            roleArn: this.role.roleArn,
            restartExecutionOnUpdate: props && props.restartExecutionOnUpdate,
            name: this.physicalName,
        });
        // this will produce a DependsOn for both the role and the policy resources.
        codePipeline.node.addDependency(this.role);
        this.artifactBucket.grantReadWrite(this.role);
        this.pipelineName = this.getResourceNameAttribute(codePipeline.ref);
        this.pipelineVersion = codePipeline.attrVersion;
        this.crossRegionBucketsPassed = !!props.crossRegionReplicationBuckets;
        for (const [region, replicationBucket] of Object.entries(props.crossRegionReplicationBuckets || {})) {
            this._crossRegionSupport[region] = {
                replicationBucket,
                stack: core_1.Stack.of(replicationBucket),
            };
        }
        // Does not expose a Fn::GetAtt for the ARN so we'll have to make it ourselves
        this.pipelineArn = core_1.Stack.of(this).formatArn({
            service: 'codepipeline',
            resource: this.pipelineName,
        });
        for (const stage of props.stages || []) {
            this.addStage(stage);
        }
    }
    /**
     * Import a pipeline into this app.
     *
     * @param scope the scope into which to import this pipeline
     * @param id the logical ID of the returned pipeline construct
     * @param pipelineArn The ARN of the pipeline (e.g. `arn:aws:codepipeline:us-east-1:123456789012:MyDemoPipeline`)
     */
    static fromPipelineArn(scope, id, pipelineArn) {
        class Import extends PipelineBase {
            constructor() {
                super(...arguments);
                this.pipelineName = core_1.Stack.of(scope).splitArn(pipelineArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resource;
                this.pipelineArn = pipelineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Creates a new Stage, and adds it to this Pipeline.
     *
     * @param props the creation properties of the new Stage
     * @returns the newly created Stage
     */
    addStage(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_codepipeline_StageOptions(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addStage);
            }
            throw error;
        }
        // check for duplicate Stages and names
        if (this._stages.find(s => s.stageName === props.stageName)) {
            throw new Error(`Stage with duplicate name '${props.stageName}' added to the Pipeline`);
        }
        const stage = new stage_1.Stage(props, this);
        const index = props.placement
            ? this.calculateInsertIndexFromPlacement(props.placement)
            : this.stageCount;
        this._stages.splice(index, 0, stage);
        return stage;
    }
    /**
     * Adds a statement to the pipeline role.
     */
    addToRolePolicy(statement) {
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Get the number of Stages in this Pipeline.
     */
    get stageCount() {
        return this._stages.length;
    }
    /**
     * Returns the stages that comprise the pipeline.
     *
     * **Note**: the returned array is a defensive copy,
     * so adding elements to it has no effect.
     * Instead, use the {@link addStage} method if you want to add more stages
     * to the pipeline.
     */
    get stages() {
        return this._stages.slice();
    }
    /**
     * Access one of the pipeline's stages by stage name
     */
    stage(stageName) {
        for (const stage of this._stages) {
            if (stage.stageName === stageName) {
                return stage;
            }
        }
        throw new Error(`Pipeline does not contain a stage named '${stageName}'. Available stages: ${this._stages.map(s => s.stageName).join(', ')}`);
    }
    /**
     * Returns all of the {@link CrossRegionSupportStack}s that were generated automatically
     * when dealing with Actions that reside in a different region than the Pipeline itself.
     *
     */
    get crossRegionSupport() {
        const ret = {};
        Object.keys(this._crossRegionSupport).forEach((key) => {
            ret[key] = this._crossRegionSupport[key];
        });
        return ret;
    }
    /** @internal */
    _attachActionToPipeline(stage, action, actionScope) {
        const richAction = new rich_action_1.RichAction(action, this);
        // handle cross-region actions here
        const crossRegionInfo = this.ensureReplicationResourcesExistFor(richAction);
        // get the role for the given action, handling if it's cross-account
        const actionRole = this.getRoleForAction(stage, richAction, actionScope);
        // // CodePipeline Variables
        validation_1.validateNamespaceName(richAction.actionProperties.variablesNamespace);
        // bind the Action (type h4x)
        const actionConfig = richAction.bind(actionScope, stage, {
            role: actionRole ? actionRole : this.role,
            bucket: crossRegionInfo.artifactBucket,
        });
        return new full_action_descriptor_1.FullActionDescriptor({
            // must be 'action', not 'richAction',
            // as those are returned by the IStage.actions property,
            // and it's important customers of Pipeline get the same instance
            // back as they added to the pipeline
            action,
            actionConfig,
            actionRole,
            actionRegion: crossRegionInfo.region,
        });
    }
    /**
     * Validate the pipeline structure
     *
     * Validation happens according to the rules documented at
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements
     * @override
     */
    validate() {
        return [
            ...this.validateSourceActionLocations(),
            ...this.validateHasStages(),
            ...this.validateStages(),
            ...this.validateArtifacts(),
        ];
    }
    ensureReplicationResourcesExistFor(action) {
        if (!action.isCrossRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // The action has a specific region,
        // require the pipeline to have a known region as well.
        this.requireRegion();
        // source actions have to be in the same region as the pipeline
        if (action.actionProperties.category === action_1.ActionCategory.SOURCE) {
            throw new Error(`Source action '${action.actionProperties.actionName}' must be in the same region as the pipeline`);
        }
        // check whether we already have a bucket in that region,
        // either passed from the outside or previously created
        const crossRegionSupport = this.obtainCrossRegionSupportFor(action);
        // the stack containing the replication bucket must be deployed before the pipeline
        core_1.Stack.of(this).addDependency(crossRegionSupport.stack);
        // The Pipeline role must be able to replicate to that bucket
        crossRegionSupport.replicationBucket.grantReadWrite(this.role);
        return {
            artifactBucket: crossRegionSupport.replicationBucket,
            region: action.effectiveRegion,
        };
    }
    /**
     * Get or create the cross-region support construct for the given action
     */
    obtainCrossRegionSupportFor(action) {
        // this method is never called for non cross-region actions
        const actionRegion = action.effectiveRegion;
        let crossRegionSupport = this._crossRegionSupport[actionRegion];
        if (!crossRegionSupport) {
            // we need to create scaffolding resources for this region
            const otherStack = action.resourceStack;
            crossRegionSupport = this.createSupportResourcesForRegion(otherStack, actionRegion);
            this._crossRegionSupport[actionRegion] = crossRegionSupport;
        }
        return crossRegionSupport;
    }
    createSupportResourcesForRegion(otherStack, actionRegion) {
        // if we have a stack from the resource passed - use that!
        if (otherStack) {
            // check if the stack doesn't have this magic construct already
            const id = `CrossRegionReplicationSupport-d823f1d8-a990-4e5c-be18-4ac698532e65-${actionRegion}`;
            let crossRegionSupportConstruct = otherStack.node.tryFindChild(id);
            if (!crossRegionSupportConstruct) {
                crossRegionSupportConstruct = new cross_region_support_stack_1.CrossRegionSupportConstruct(otherStack, id, {
                    createKmsKey: this.crossAccountKeys,
                    enableKeyRotation: this.enableKeyRotation,
                });
            }
            return {
                replicationBucket: crossRegionSupportConstruct.replicationBucket,
                stack: otherStack,
            };
        }
        // otherwise - create a stack with the resources needed for replication across regions
        const pipelineStack = core_1.Stack.of(this);
        const pipelineAccount = pipelineStack.account;
        if (core_1.Token.isUnresolved(pipelineAccount)) {
            throw new Error("You need to specify an explicit account when using CodePipeline's cross-region support");
        }
        const app = this.supportScope();
        const supportStackId = `cross-region-stack-${this.reuseCrossRegionSupportStacks ? pipelineAccount : pipelineStack.stackName}:${actionRegion}`;
        let supportStack = app.node.tryFindChild(supportStackId);
        if (!supportStack) {
            supportStack = new cross_region_support_stack_1.CrossRegionSupportStack(app, supportStackId, {
                pipelineStackName: pipelineStack.stackName,
                region: actionRegion,
                account: pipelineAccount,
                synthesizer: this.getCrossRegionSupportSynthesizer(),
                createKmsKey: this.crossAccountKeys,
                enableKeyRotation: this.enableKeyRotation,
            });
        }
        return {
            stack: supportStack,
            replicationBucket: supportStack.replicationBucket,
        };
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the pipeline account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer.cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
    generateNameForDefaultBucketKeyAlias() {
        const prefix = 'alias/codepipeline-';
        const maxAliasLength = 256;
        const uniqueId = core_1.Names.uniqueId(this);
        // take the last 256 - (prefix length) characters of uniqueId
        const startIndex = Math.max(0, uniqueId.length - (maxAliasLength - prefix.length));
        return prefix + uniqueId.substring(startIndex).toLowerCase();
    }
    /**
     * Gets the role used for this action,
     * including handling the case when the action is supposed to be cross-account.
     *
     * @param stage the stage the action belongs to
     * @param action the action to return/create a role for
     * @param actionScope the scope, unique to the action, to create new resources in
     */
    getRoleForAction(stage, action, actionScope) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRole = this.getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action);
        if (!actionRole && this.isAwsOwned(action)) {
            // generate a Role for this specific Action
            actionRole = new iam.Role(actionScope, 'CodePipelineActionRole', {
                assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            });
        }
        // the pipeline role needs assumeRole permissions to the action role
        if (actionRole) {
            this.role.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                resources: [actionRole.roleArn],
            }));
        }
        return actionRole;
    }
    getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action) {
        const pipelineStack = core_1.Stack.of(this);
        // if we have a cross-account action, the pipeline's bucket must have a KMS key
        // (otherwise we can't configure cross-account trust policies)
        if (action.isCrossAccount) {
            const artifactBucket = this.ensureReplicationResourcesExistFor(action).artifactBucket;
            if (!artifactBucket.encryptionKey) {
                throw new Error(`Artifact Bucket must have a KMS Key to add cross-account action '${action.actionProperties.actionName}' ` +
                    `(pipeline account: '${renderEnvDimension(this.env.account)}', action account: '${renderEnvDimension(action.effectiveAccount)}'). ` +
                    'Create Pipeline with \'crossAccountKeys: true\' (or pass an existing Bucket with a key)');
            }
        }
        // if a Role has been passed explicitly, always use it
        // (even if the backing resource is from a different account -
        // this is how the user can override our default support logic)
        if (action.actionProperties.role) {
            if (this.isAwsOwned(action)) {
                // the role has to be deployed before the pipeline
                // (our magical cross-stack dependencies will not work,
                // because the role might be from a different environment),
                // but _only_ if it's a new Role -
                // an imported Role should not add the dependency
                if (action.actionProperties.role instanceof iam.Role) {
                    const roleStack = core_1.Stack.of(action.actionProperties.role);
                    pipelineStack.addDependency(roleStack);
                }
                return action.actionProperties.role;
            }
            else {
                // ...except if the Action is not owned by 'AWS',
                // as that would be rejected by CodePipeline at deploy time
                throw new Error("Specifying a Role is not supported for actions with an owner different than 'AWS' - " +
                    `got '${action.actionProperties.owner}' (Action: '${action.actionProperties.actionName}' in Stage: '${stage.stageName}')`);
            }
        }
        // if we don't have a Role passed,
        // and the action is cross-account,
        // generate a Role in that other account stack
        const otherAccountStack = this.getOtherStackIfActionIsCrossAccount(action);
        if (!otherAccountStack) {
            return undefined;
        }
        // generate a role in the other stack, that the Pipeline will assume for executing this action
        const ret = new iam.Role(otherAccountStack, `${core_1.Names.uniqueId(this)}-${stage.stageName}-${action.actionProperties.actionName}-ActionRole`, {
            assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
        });
        // the other stack with the role has to be deployed before the pipeline stack
        // (CodePipeline verifies you can assume the action Role on creation)
        pipelineStack.addDependency(otherAccountStack);
        return ret;
    }
    /**
     * Returns the Stack this Action belongs to if this is a cross-account Action.
     * If this Action is not cross-account (i.e., it lives in the same account as the Pipeline),
     * it returns undefined.
     *
     * @param action the Action to return the Stack for
     */
    getOtherStackIfActionIsCrossAccount(action) {
        const targetAccount = action.actionProperties.resource
            ? action.actionProperties.resource.env.account
            : action.actionProperties.account;
        if (targetAccount === undefined) {
            // if the account of the Action is not specified,
            // then it defaults to the same account the pipeline itself is in
            return undefined;
        }
        // check whether the action's account is a static string
        if (core_1.Token.isUnresolved(targetAccount)) {
            if (core_1.Token.isUnresolved(this.env.account)) {
                // the pipeline is also env-agnostic, so that's fine
                return undefined;
            }
            else {
                throw new Error(`The 'account' property must be a concrete value (action: '${action.actionProperties.actionName}')`);
            }
        }
        // At this point, we know that the action's account is a static string.
        // In this case, the pipeline's account must also be a static string.
        if (core_1.Token.isUnresolved(this.env.account)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set account');
        }
        // at this point, we know that both the Pipeline's account,
        // and the action-backing resource's account are static strings
        // if they are identical - nothing to do (the action is not cross-account)
        if (this.env.account === targetAccount) {
            return undefined;
        }
        // at this point, we know that the action is certainly cross-account,
        // so we need to return a Stack in its account to create the helper Role in
        const candidateActionResourceStack = action.actionProperties.resource
            ? core_1.Stack.of(action.actionProperties.resource)
            : undefined;
        if ((candidateActionResourceStack === null || candidateActionResourceStack === void 0 ? void 0 : candidateActionResourceStack.account) === targetAccount) {
            // we always use the "latest" action-backing resource's Stack for this account,
            // even if a different one was used earlier
            this._crossAccountSupport[targetAccount] = candidateActionResourceStack;
            return candidateActionResourceStack;
        }
        let targetAccountStack = this._crossAccountSupport[targetAccount];
        if (!targetAccountStack) {
            const stackId = `cross-account-support-stack-${targetAccount}`;
            const app = this.supportScope();
            targetAccountStack = app.node.tryFindChild(stackId);
            if (!targetAccountStack) {
                const actionRegion = action.actionProperties.resource
                    ? action.actionProperties.resource.env.region
                    : action.actionProperties.region;
                const pipelineStack = core_1.Stack.of(this);
                targetAccountStack = new core_1.Stack(app, stackId, {
                    stackName: `${pipelineStack.stackName}-support-${targetAccount}`,
                    env: {
                        account: targetAccount,
                        region: actionRegion !== null && actionRegion !== void 0 ? actionRegion : pipelineStack.region,
                    },
                });
            }
            this._crossAccountSupport[targetAccount] = targetAccountStack;
        }
        return targetAccountStack;
    }
    isAwsOwned(action) {
        const owner = action.actionProperties.owner;
        return !owner || owner === 'AWS';
    }
    getArtifactBucketFromProps(props) {
        if (props.artifactBucket) {
            return props.artifactBucket;
        }
        if (props.crossRegionReplicationBuckets) {
            const pipelineRegion = this.requireRegion();
            return props.crossRegionReplicationBuckets[pipelineRegion];
        }
        return undefined;
    }
    calculateInsertIndexFromPlacement(placement) {
        // check if at most one placement property was provided
        const providedPlacementProps = ['rightBefore', 'justAfter', 'atIndex']
            .filter((prop) => placement[prop] !== undefined);
        if (providedPlacementProps.length > 1) {
            throw new Error('Error adding Stage to the Pipeline: ' +
                'you can only provide at most one placement property, but ' +
                `'${providedPlacementProps.join(', ')}' were given`);
        }
        if (placement.rightBefore !== undefined) {
            const targetIndex = this.findStageIndex(placement.rightBefore);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it before, '${placement.rightBefore.stageName}', was not found`);
            }
            return targetIndex;
        }
        if (placement.justAfter !== undefined) {
            const targetIndex = this.findStageIndex(placement.justAfter);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it after, '${placement.justAfter.stageName}', was not found`);
            }
            return targetIndex + 1;
        }
        return this.stageCount;
    }
    findStageIndex(targetStage) {
        return this._stages.findIndex(stage => stage === targetStage);
    }
    validateSourceActionLocations() {
        const errors = new Array();
        let firstStage = true;
        for (const stage of this._stages) {
            const onlySourceActionsPermitted = firstStage;
            for (const action of stage.actionDescriptors) {
                errors.push(...validation_1.validateSourceAction(onlySourceActionsPermitted, action.category, action.actionName, stage.stageName));
            }
            firstStage = false;
        }
        return errors;
    }
    validateHasStages() {
        if (this.stageCount < 2) {
            return ['Pipeline must have at least two stages'];
        }
        return [];
    }
    validateStages() {
        const ret = new Array();
        for (const stage of this._stages) {
            ret.push(...stage.validate());
        }
        return ret;
    }
    validateArtifacts() {
        const ret = new Array();
        const producers = {};
        const firstConsumers = {};
        for (const [stageIndex, stage] of enumerate(this._stages)) {
            // For every output artifact, get the producer
            for (const action of stage.actionDescriptors) {
                const actionLoc = new PipelineLocation(stageIndex, stage, action);
                for (const outputArtifact of action.outputs) {
                    // output Artifacts always have a name set
                    const name = outputArtifact.artifactName;
                    if (producers[name]) {
                        ret.push(`Both Actions '${producers[name].actionName}' and '${action.actionName}' are producting Artifact '${name}'. Every artifact can only be produced once.`);
                        continue;
                    }
                    producers[name] = actionLoc;
                }
                // For every input artifact, get the first consumer
                for (const inputArtifact of action.inputs) {
                    const name = inputArtifact.artifactName;
                    if (!name) {
                        ret.push(`Action '${action.actionName}' is using an unnamed input Artifact, which is not being produced in this pipeline`);
                        continue;
                    }
                    firstConsumers[name] = firstConsumers[name] ? firstConsumers[name].first(actionLoc) : actionLoc;
                }
            }
        }
        // Now validate that every input artifact is produced before it's
        // being consumed.
        for (const [artifactName, consumerLoc] of Object.entries(firstConsumers)) {
            const producerLoc = producers[artifactName];
            if (!producerLoc) {
                ret.push(`Action '${consumerLoc.actionName}' is using input Artifact '${artifactName}', which is not being produced in this pipeline`);
                continue;
            }
            if (consumerLoc.beforeOrEqual(producerLoc)) {
                ret.push(`${consumerLoc} is consuming input Artifact '${artifactName}' before it is being produced at ${producerLoc}`);
            }
        }
        return ret;
    }
    renderArtifactStoresProperty() {
        if (!this.crossRegion) {
            return undefined;
        }
        // add the Pipeline's artifact store
        const primaryRegion = this.requireRegion();
        this._crossRegionSupport[primaryRegion] = {
            replicationBucket: this.artifactBucket,
            stack: core_1.Stack.of(this),
        };
        return Object.entries(this._crossRegionSupport).map(([region, support]) => ({
            region,
            artifactStore: this.renderArtifactStore(support.replicationBucket),
        }));
    }
    renderArtifactStoreProperty() {
        if (this.crossRegion) {
            return undefined;
        }
        return this.renderPrimaryArtifactStore();
    }
    renderPrimaryArtifactStore() {
        return this.renderArtifactStore(this.artifactBucket);
    }
    renderArtifactStore(bucket) {
        let encryptionKey;
        const bucketKey = bucket.encryptionKey;
        if (bucketKey) {
            encryptionKey = {
                type: 'KMS',
                id: bucketKey.keyArn,
            };
        }
        return {
            type: 'S3',
            location: bucket.bucketName,
            encryptionKey,
        };
    }
    get crossRegion() {
        if (this.crossRegionBucketsPassed) {
            return true;
        }
        return this._stages.some(stage => stage.actionDescriptors.some(action => action.region !== undefined));
    }
    renderStages() {
        return this._stages.map(stage => stage.render());
    }
    renderDisabledTransitions() {
        return this._stages
            .filter(stage => !stage.transitionToEnabled)
            .map(stage => ({
            reason: stage.transitionDisabledReason,
            stageName: stage.stageName,
        }));
    }
    requireRegion() {
        const region = this.env.region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set region');
        }
        return region;
    }
    supportScope() {
        const scope = core_1.Stage.of(this);
        if (!scope) {
            throw new Error('Pipeline stack which uses cross-environment actions must be part of a CDK App or Stage');
        }
        return scope;
    }
}
exports.Pipeline = Pipeline;
_a = JSII_RTTI_SYMBOL_1;
Pipeline[_a] = { fqn: "@aws-cdk/aws-codepipeline.Pipeline", version: "1.156.1" };
function enumerate(xs) {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        ret.push([i, xs[i]]);
    }
    return ret;
}
class PipelineLocation {
    constructor(stageIndex, stage, action) {
        this.stageIndex = stageIndex;
        this.stage = stage;
        this.action = action;
    }
    get stageName() {
        return this.stage.stageName;
    }
    get actionName() {
        return this.action.actionName;
    }
    /**
     * Returns whether a is before or the same order as b
     */
    beforeOrEqual(rhs) {
        if (this.stageIndex !== rhs.stageIndex) {
            return rhs.stageIndex < rhs.stageIndex;
        }
        return this.action.runOrder <= rhs.action.runOrder;
    }
    /**
     * Returns the first location between this and the other one
     */
    first(rhs) {
        return this.beforeOrEqual(rhs) ? this : rhs;
    }
    toString() {
        // runOrders are 1-based, so make the stageIndex also 1-based otherwise it's going to be confusing.
        return `Stage ${this.stageIndex + 1} Action ${this.action.runOrder} ('${this.stageName}'/'${this.actionName}')`;
    }
}
/**
 * Render an env dimension without showing the ugly stringified tokens
 */
function renderEnvDimension(s) {
    return core_1.Token.isUnresolved(s) ? '(current)' : s;
}
//# sourceMappingURL=data:application/json;base64,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