##
#     Project: Awaitable
# Description: A decorator to asynchronously execute synchronous functions
#      Author: Fabio Castelli (Muflone) <muflone@muflone.com>
#   Copyright: 2022 Fabio Castelli
#     License: GPL-3+
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <https://www.gnu.org/licenses/>.
##

import asyncio
import functools
import typing


def awaitable(func: typing.Callable):
    """
    Decorates a synchronous function to run it in asynchronous
    :param func: function to decorate
    :return: the awaited data from the decorated function
    """
    @functools.wraps(func)
    async def run(*args, loop=None, executor=None, **kwargs):
        if loop is None:
            loop = asyncio.get_event_loop()
        pfunc = functools.partial(func, *args, **kwargs)
        return await loop.run_in_executor(executor, pfunc)
    return run


def run_awaitable(func: typing.Callable, *args, **kwargs) -> None:
    """
    Execute an awaitable function with arguments
    """
    return asyncio.run(func(*args, **kwargs))
