"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const codepipeline = require("../lib");
const fake_build_action_1 = require("./fake-build-action");
const fake_source_action_1 = require("./fake-source-action");
function validate(construct) {
    cdk.ConstructNode.prepare(construct.node);
    return cdk.ConstructNode.validate(construct.node);
}
module.exports = {
    'Artifacts in CodePipeline': {
        'cannot be created with an empty name'(test) {
            test.throws(() => new codepipeline.Artifact(''), /Artifact name must match regular expression/);
            test.done();
        },
        'without a name, when used as an input without being used as an output first - should fail validation'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: new codepipeline.Artifact(),
                            }),
                        ],
                    },
                ],
            });
            const errors = validate(stack);
            test.equal(errors.length, 1);
            const error = errors[0];
            test.same(error.source, pipeline);
            test.equal(error.message, "Action 'Build' is using an unnamed input Artifact, which is not being produced in this pipeline");
            test.done();
        },
        'with a name, when used as an input without being used as an output first - should fail validation'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: new codepipeline.Artifact('named'),
                            }),
                        ],
                    },
                ],
            });
            const errors = validate(stack);
            test.equal(errors.length, 1);
            const error = errors[0];
            test.same(error.source, pipeline);
            test.equal(error.message, "Action 'Build' is using input Artifact 'named', which is not being produced in this pipeline");
            test.done();
        },
        'without a name, when used as an output multiple times - should fail validation'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'Source',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'Build',
                                input: sourceOutput,
                                output: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            const errors = validate(stack);
            test.equal(errors.length, 1);
            const error = errors[0];
            test.same(error.source, pipeline);
            test.equal(error.message, "Both Actions 'Source' and 'Build' are producting Artifact 'Artifact_Source_Source'. Every artifact can only be produced once.");
            test.done();
        },
        "an Action's output can be used as input for an Action in the same Stage with a higher runOrder"(test) {
            const stack = new cdk.Stack();
            const sourceOutput1 = new codepipeline.Artifact('sourceOutput1');
            const buildOutput1 = new codepipeline.Artifact('buildOutput1');
            const sourceOutput2 = new codepipeline.Artifact('sourceOutput2');
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source1',
                                output: sourceOutput1,
                            }),
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source2',
                                output: sourceOutput2,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build1',
                                input: sourceOutput1,
                                output: buildOutput1,
                            }),
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build2',
                                input: sourceOutput2,
                                extraInputs: [buildOutput1],
                                output: new codepipeline.Artifact('buildOutput2'),
                                runOrder: 2,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
            //
            }));
            test.done();
        },
        'violation of runOrder constraints is detected and reported'(test) {
            const stack = new cdk.Stack();
            const sourceOutput1 = new codepipeline.Artifact('sourceOutput1');
            const buildOutput1 = new codepipeline.Artifact('buildOutput1');
            const sourceOutput2 = new codepipeline.Artifact('sourceOutput2');
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source1',
                                output: sourceOutput1,
                            }),
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source2',
                                output: sourceOutput2,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build1',
                                input: sourceOutput1,
                                output: buildOutput1,
                                runOrder: 3,
                            }),
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build2',
                                input: sourceOutput2,
                                extraInputs: [buildOutput1],
                                output: new codepipeline.Artifact('buildOutput2'),
                                runOrder: 2,
                            }),
                        ],
                    },
                ],
            });
            const errors = validate(stack);
            test.equal(errors.length, 1);
            const error = errors[0];
            test.same(error.source, pipeline);
            test.equal(error.message, "Stage 2 Action 2 ('Build'/'build2') is consuming input Artifact 'buildOutput1' before it is being produced at Stage 2 Action 3 ('Build'/'build1')");
            test.done();
        },
        'without a name, sanitize the auto stage-action derived name'(test) {
            const stack = new cdk.Stack();
            const sourceOutput = new codepipeline.Artifact();
            new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'Source.@',
                        actions: [
                            new fake_source_action_1.FakeSourceAction({
                                actionName: 'source1',
                                output: sourceOutput,
                            }),
                        ],
                    },
                    {
                        stageName: 'Build',
                        actions: [
                            new fake_build_action_1.FakeBuildAction({
                                actionName: 'build1',
                                input: sourceOutput,
                            }),
                        ],
                    },
                ],
            });
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    {
                        'Name': 'Source.@',
                        'Actions': [
                            {
                                'Name': 'source1',
                                'OutputArtifacts': [
                                    { 'Name': 'Artifact_Source_source1' },
                                ],
                            },
                        ],
                    },
                    {
                        'Name': 'Build',
                        'Actions': [
                            {
                                'Name': 'build1',
                                'InputArtifacts': [
                                    { 'Name': 'Artifact_Source_source1' },
                                ],
                            },
                        ],
                    },
                ],
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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