"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const index_1 = require("../lib/index");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
require("@aws-cdk/assert/jest");
const cdk = require("aws-cdk-lib");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
function deployNewStateMachine(stack) {
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        }
    };
    return new index_1.S3ToStepfunctions(stack, 'test-s3-stepfunctions', props);
}
test('check deployCloudTrail = false', () => {
    const stack = new cdk.Stack();
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        deployCloudTrail: false
    };
    const construct = new index_1.S3ToStepfunctions(stack, 'test-s3-stepfunctions', props);
    expect(construct.cloudtrail === undefined);
});
test('override eventRuleProps', () => {
    const stack = new cdk.Stack();
    const mybucket = new aws_s3_1.Bucket(stack, 'mybucket');
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        existingBucketObj: mybucket,
        eventRuleProps: {
            eventPattern: {
                source: ['aws.s3'],
                detailType: ['AWS API Call via CloudTrail'],
                detail: {
                    eventSource: [
                        "s3.amazonaws.com"
                    ],
                    eventName: [
                        "GetObject"
                    ],
                    requestParameters: {
                        bucketName: [
                            mybucket.bucketName
                        ]
                    }
                }
            }
        }
    };
    new index_1.S3ToStepfunctions(stack, 'test-s3-stepfunctions', props);
    expect(stack).toHaveResource('AWS::Events::Rule', {
        EventPattern: {
            "source": [
                "aws.s3"
            ],
            "detail-type": [
                "AWS API Call via CloudTrail"
            ],
            "detail": {
                eventSource: [
                    "s3.amazonaws.com"
                ],
                eventName: [
                    "GetObject"
                ],
                requestParameters: {
                    bucketName: [
                        {
                            Ref: "mybucket160F8132"
                        }
                    ]
                }
            }
        },
        State: "ENABLED",
        Targets: [
            {
                Arn: {
                    Ref: "tests3stepfunctionstests3stepfunctionseventrulestepfunctionconstructStateMachine67197269"
                },
                Id: "Target0",
                RoleArn: {
                    "Fn::GetAtt": [
                        "tests3stepfunctionstests3stepfunctionseventrulestepfunctionconstructEventsRuleRoleE7CAD359",
                        "Arn"
                    ]
                }
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStateMachine(stack);
    expect(construct.cloudtrail !== null);
    expect(construct.stateMachine !== null);
    expect(construct.s3Bucket !== null);
    expect(construct.cloudwatchAlarms !== null);
    expect(construct.stateMachineLogGroup !== null);
    expect(construct.s3LoggingBucket !== null);
    expect(construct.cloudtrail !== null);
    expect(construct.cloudtrailBucket !== null);
    expect(construct.cloudtrailLoggingBucket !== null);
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Test bad call with existingBucket and bucketProps", () => {
    // Stack
    const stack = new cdk.Stack();
    const testBucket = new aws_s3_1.Bucket(stack, 'test-bucket', {});
    const startState = new sfn.Pass(stack, 'StartState');
    const app = () => {
        // Helper declaration
        new index_1.S3ToStepfunctions(stack, "bad-s3-args", {
            stateMachineProps: {
                definition: startState
            },
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: cdk.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError();
});
// --------------------------------------------------------------
// s3 bucket with bucket, loggingBucket, and auto delete objects
// --------------------------------------------------------------
test('s3 bucket with bucket, loggingBucket, and auto delete objects', () => {
    const stack = new cdk.Stack();
    const startState = new sfn.Pass(stack, 'StartState');
    const testProps = {
        stateMachineProps: {
            definition: startState
        },
        bucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY
        },
        loggingBucketProps: {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    };
    new index_1.S3ToStepfunctions(stack, 'test-s3-stepfunctions', testProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        AccessControl: "LogDeliveryWrite"
    });
    expect(stack).toHaveResource("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "tests3stepfunctionsS3LoggingBucketF7586A92"
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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