import json
from datetime import datetime, timezone
from unittest import TestCase

from ......indy.models.pres_preview import PRESENTATION_PREVIEW
from ......messaging.decorators.attach_decorator import AttachDecorator
from ......messaging.util import str_to_datetime, str_to_epoch

from .....didcomm_prefix import DIDCommPrefix

from ...message_types import ATTACH_DECO_IDS, PRESENTATION

from ..presentation import Presentation, PresentationSchema

INDY_PROOF = json.loads(
    """{
        "proof": {
            "proofs": [
                {
                    "primary_proof": {
                        "eq_proof": {
                            "revealed_attrs": {
                                "player": "51643998292319337989293919354395093705917445045137690661130646028663839100479",
                                "screencapture": "44349549498354774830429200435932754610833874667251545521048906777181334567815"
                            },
                            "a_prime": "99225796363129499107604366233301127916801972855861917994091548556785477066502130364812678473656139160841991495705941663142249404264191136660967090000331013804872112053828446231809361437830836319053659748356431221803865514426935793809384997872056337751830616632363564742453553492463002290910985263848243042219992778220569301291770422529015543837803240796109915443060533925706334481433988647943330126921627882396088865869804752372160284361135101444121353171684921889674279086653358367368851378746682682017641177839566946615542088910003479771742848013401738739436319413416417782857315505655723403098381845469564775640588",
                            "e": "28484172708495089688591061270972815967199639781170588390863001534745829714460906432474939589201651469315129053056279820725958192110265136",
                            "v": "310372334186966867767394517648718214703060542831249429833856393387373123821840122943078768258679350688701210557571314095023780969910990133962277141702540794078947706642651608840502392669887782435197020314627659897216201821476693212776945691916847975724720629133302982522740709659244048850715861867163370042548490158397753134853555833520807793752332629616695158900467048806794540963892785661237500652426649694476839734543710070772542960644069106447786837388953205349573770161012926822761642443603379863014577283739370081232865330663720633954578384601051328768426422459925307605555673543912329275856396630694738618400128412881389666175048976575778590587039239852074579445246476657508798666439893978860122625600280248580653651989659501745788120866210838204610848826992305622269021702688221635993649651671518759866100294588385482901147670474709698187899410921387549396343476268788611619095756118794378610337782521199137176224",
                            "m": {
                                "master_secret": "8862971523696585539875886113995946020345090415446970983664333029999473510798222518918685777004680817221644138616821331322893963179678008732473561080506239631376575074759262558623",
                                "date": "3701779401056653400708681878780576462168055130242160156441633682389568986593680911678649493653787250169881692457012639607423195648009201693522087171287177627155679953691027082306",
                                "highscore": "15076065627409123507707791591890677721352422176962229116158012124884023896353283613850809563416017162039356935197216493911366484372240599638993754251972383037120760793174059437326"
                            },
                            "m2": "936898843995229611075174877423066852536402039331414286329629096155063110397949209326899164087270236968111471019540493930568502892781443118611948331343540849982215419978654911341"
                        },
                        "ge_proofs": [
                            {
                                "u": {
                                    "0": "9910469266382558421810537687107445527637953525140204243652090909154732881567346670639902692019649848585497930780041894066589111086262231121289584890680947709857922351898933228959",
                                    "3": "13248890365144372967021124637790988823123419165600968402954657790395188046865908780216014168108873473963822724485182321396055154711186623889234974568160016086782335901983921278203",
                                    "2": "12729039529929764954731327277162243472469670773258016331674525566138793186295239771259296208473089652983817249211287815365374343774154094615763169572305994728783319085378462750119",
                                    "1": "7521808223922555179229230989469494924108464850902024304215849946306721494292541804707880060117792628557556919883251581183099791703469635100219991762282695219119375485542725378777"
                                },
                                "r": {
                                    "3": "2474586272027077975486776866285873096434331606893837372003598899080539007560599606386516216782289932402894639261205313757395792446618729476125758006073556965534129613180311177356435264610207048503766208536525437467431668179066975773784256747360733829457728689042564760785075167221871190381905962993342585054474809475874638461649882224522900216073005325070483781773167488104736004488166472769233964211119250710987817940295641154170092028642853085492684423831557787832223441677166939601580910358630679766277360055054583280321123214001754046780045473183143311899432032961118915743585200408079711683568075159956637355186537460890150634531127711285758617739650166494804700747238587328374163718880396805711619195410428497141",
                                    "2": "2630464840206472451352435662547628615461823248193121896366292326754757111056015434024860402045528167154717793472610145065612236007297496233371229987771148480914236050468139151516543123252130359806069866913832582652430060368351667452390252792745981559902564451307173881741056494603759634524628446420629554945618261322035719400890137143004894938649567505283955045583301734843724484105958980144825603458470170246633173176192352102370358808823038609216670566297573459331481693366189829604088382174720921421068848195803685053584587847959340545747151323994860573252761484349482452752365951814578536977631851802458952874933908594142054875532155473403049377997857193944575096457437636198069049894085647451273888200116980726092",
                                    "1": "2852147521010832076474693080123829749829373205563299890275783906381127244731842078098806253403603591226341534169437752738669698923225573040124923814326088208465858997309773274462266090025447286378141544917213418789777276232863321772419735930833747576309139155217894968446024099207333956546610641531588126922714769703447074214896402884035961312686767119156707888839495093047502240940442068243444839642678428392561564279122033304060367830470731800699885137708112213347900071682836322404659581146632750296228233441035302852186755012460856485782729749727931571925009194110383907166489891246153746477910501305713189452876479941940283249570571466801547498554092112942172290619708436101630604721002777991653223187127539407188",
                                    "DELTA": "400740231805932179607546658608482360416676808879808936308239007450069335712770990135423875914299915061808733825416765503161922435087393607455279098108543704733230814698288332881292132566532892722244536550609474863487095816391676106247864333163126795882262678039103218492008333619274792818770308974444039810096709828122153085809072205039719201560334210985909087337968918296450456759914221258287823859138473869581326860149282690703526479416994879663317413415525469689392534867388970645182739614666457086788145724809368914878257774143699515974528212285813531498884015621850779340589427600835454594927635608618313963836648119837777098673007860656489994343544396208432731266075365830717274498351940211946906749568641992530",
                                    "0": "1206673881851533752176657850353675358524597024445357836801291763123272463247544653871603547107824681844497100741157733091042299879547696954660696997520368168483474593036101472335505287047339386308031509611499543209773577503155192535635651933608157610580443175876534879594575874473220014224237470499919793664212944782077926675612730961243351448995290239215801924035454011696132815884654568365382507261689165029962957712345451405751438882798844088168256631131921905245510548989991506332080163507201398283921938862585730222296508424960186566696340473016767188656883762864118588802209468135703456208025238541839477324582906436589408122727413989766360283059475263178640070468528674228956264205722590748705114610224502937924"
                                },
                                "mj": "15076065627409123507707791591890677721352422176962229116158012124884023896353283613850809563416017162039356935197216493911366484372240599638993754251972383037120760793174059437326",
                                "alpha": "20251550018805200717806858447687954659786798446794929315180340450717009476769405863150379133594211911561358797900043949141708879953965949034296837455168571993614131838308136400934080334484858045221098438795729643169952299593947544536931262318894249728605957882845005603393308631544441292352568325855333088710935761954865297018529190379824999153478968957974616452369432737206876894394433169704416574734350248494633324787866283669843201013454433040804437117653085130836624250598443032846839559239219803003053865279821640383428381442135797658167010830469471880710203270574788431679668220274396595963367127851323648855427656787139315767612273110920619973147968805075620184678295428649237076408491062282135768652643652528789723106929481799840238867321416457406628403722479816549749574895345371486180196838697381621782729034821539642473730374275",
                                "t": {
                                    "1": "12625197327271658918836077104165526795551240746110291943686378253405709448817799744491714171274845559259436160572533367386665079411321345166308961745117961730852997405633497191717007336383275404469818669156913360917193297281721819431590624901667128875332561566036439988983652391578753211775620012967251966145029999420901103522072647380944582775843791262702644151927742979273959780439648875773579687106661837930590989533046533664173215551687012232903455587682542013722715620746003808218596250895466798585940038977660031964518043170383195848432855521396949050006496669882466103602834555814104353098012178481563132990657",
                                    "3": "82102416726449754901570630901431660447826687124743534767954749884467633032358726226619062293813250820543583618667653110864397826099702636976514863698490371598871942970264169528954417033219855319798151022602756245903134559243361308006137131575819330897670063637536795053765101329851925607560890238602738686737347630399680932950512292412006361269539738453753560364596561872651528860308101942007770489206306048924418921104517753483478955863296623417733412161191192531054326372049247205543273207371278781809399097610512792780914259992762072456575639120070897889219135350947197581287043954055372025101673838669553746551523",
                                    "0": "100578099981822727242488292109669009229478055276500695778799086886344998432604032811665840061704724353178176792298171825200217745276011656576161627798148614876492383276153655146449848780838571509873143828996025628954667317519574656744701630828190045526936155193536814016169445565475181479441229047491855276823646742587245970832496856994388840793376871874193330364608394771574962996229647270622689890201908589893313568444474914909794303851820492781326574727803226373005399197371492982012783800353741451399606551384719595965296619783050926116585174881782168129321205830465290478768408675156580724359333089093105010344487",
                                    "2": "47291536708088381287407033267847414228876334422991232636387475485756328314399598367105968385520172836890544717976118198568671113811836108861048793780118048683411340116566023370245246884524520199561342298868861751758445312599348599287067000725278934840752177807977101054892905295530294108292736307777321970970868898458355273485795649568677223443447768573057466329236959267653001983213430774265365847091875699626385937604178216275273379502023024485339694410370916685404579472512288185963724377525685276628144678139522579811749896221643038522340842472046618109166452353106698715375908659582424315255951960930185079622552",
                                    "DELTA": "55673614276503115042406892194681370272903807098038274960776275804979087176140123726613332530447421097732347173352956738522605590407126570163366084735258393133886870700490345929950624260625461471012084011187108973815830590105522983606251371051538463584013547099942110852482167674597067842508689609606420417081221833855564428834462819662758502495039815615824926366319292041564418283778981490957086445486745581161189382261760754225714728934548296947403634289640526526314947616964122321833465956469712078741533550908164428460947933509296796422321858634999992086190358241952920458802129165732538146634862846975496258789679"
                                },
                                "predicate": {
                                    "attr_name": "highscore",
                                    "p_type": "GE",
                                    "value": 1000000
                                }
                            }
                        ]
                    },
                    "non_revoc_proof": null
                }
            ],
            "aggregated_proof": {
                "c_hash": "81147637626301581116624461636159287563986704950981430016774756525127013830996",
                "c_list": [
                    [
                        3,
                        18,
                        5,
                        11,
                        249,
                        192,
                        147,
                        232,
                        208,
                        2,
                        120,
                        15,
                        246,
                        67,
                        152,
                        178,
                        13,
                        223,
                        45,
                        197,
                        49,
                        251,
                        124,
                        129,
                        88,
                        30,
                        22,
                        215,
                        93,
                        198,
                        188,
                        111,
                        134,
                        78,
                        237,
                        244,
                        150,
                        57,
                        134,
                        207,
                        48,
                        252,
                        238,
                        215,
                        44,
                        69,
                        28,
                        38,
                        231,
                        95,
                        66,
                        222,
                        118,
                        30,
                        137,
                        6,
                        78,
                        103,
                        185,
                        218,
                        139,
                        176,
                        149,
                        97,
                        40,
                        224,
                        246,
                        241,
                        87,
                        80,
                        58,
                        169,
                        185,
                        39,
                        121,
                        175,
                        175,
                        181,
                        73,
                        172,
                        152,
                        149,
                        252,
                        2,
                        237,
                        255,
                        147,
                        215,
                        212,
                        0,
                        134,
                        24,
                        198,
                        1,
                        241,
                        191,
                        206,
                        227,
                        200,
                        228,
                        32,
                        22,
                        90,
                        101,
                        237,
                        161,
                        32,
                        157,
                        211,
                        231,
                        28,
                        106,
                        42,
                        227,
                        234,
                        207,
                        116,
                        119,
                        121,
                        173,
                        188,
                        167,
                        195,
                        218,
                        223,
                        194,
                        123,
                        102,
                        140,
                        36,
                        121,
                        231,
                        254,
                        240,
                        155,
                        55,
                        244,
                        236,
                        106,
                        84,
                        62,
                        169,
                        69,
                        56,
                        191,
                        61,
                        29,
                        29,
                        117,
                        196,
                        40,
                        26,
                        210,
                        204,
                        194,
                        164,
                        5,
                        25,
                        138,
                        235,
                        164,
                        176,
                        182,
                        32,
                        100,
                        24,
                        52,
                        71,
                        227,
                        199,
                        45,
                        162,
                        88,
                        66,
                        245,
                        222,
                        51,
                        250,
                        174,
                        222,
                        34,
                        93,
                        63,
                        181,
                        49,
                        45,
                        226,
                        120,
                        183,
                        81,
                        127,
                        222,
                        168,
                        100,
                        99,
                        8,
                        8,
                        248,
                        24,
                        142,
                        118,
                        99,
                        42,
                        157,
                        170,
                        117,
                        103,
                        183,
                        22,
                        253,
                        189,
                        186,
                        234,
                        88,
                        129,
                        202,
                        193,
                        32,
                        237,
                        49,
                        251,
                        49,
                        131,
                        183,
                        2,
                        22,
                        44,
                        207,
                        13,
                        83,
                        98,
                        38,
                        14,
                        160,
                        14,
                        13,
                        146,
                        108,
                        239,
                        43,
                        47,
                        238,
                        251,
                        17,
                        206,
                        164,
                        179,
                        185,
                        103,
                        219,
                        80,
                        159,
                        145,
                        184,
                        239,
                        46,
                        12
                    ],
                    [
                        3,
                        28,
                        187,
                        101,
                        204,
                        218,
                        140,
                        64,
                        119,
                        109,
                        189,
                        77,
                        133,
                        186,
                        157,
                        230,
                        147,
                        59,
                        219,
                        42,
                        64,
                        16,
                        163,
                        132,
                        197,
                        115,
                        236,
                        3,
                        117,
                        211,
                        98,
                        142,
                        33,
                        166,
                        85,
                        1,
                        88,
                        93,
                        245,
                        55,
                        253,
                        248,
                        59,
                        240,
                        70,
                        169,
                        206,
                        15,
                        157,
                        202,
                        59,
                        254,
                        204,
                        251,
                        3,
                        126,
                        139,
                        138,
                        251,
                        103,
                        229,
                        185,
                        66,
                        105,
                        188,
                        36,
                        47,
                        233,
                        32,
                        148,
                        14,
                        116,
                        14,
                        40,
                        62,
                        209,
                        131,
                        62,
                        108,
                        124,
                        251,
                        157,
                        114,
                        208,
                        94,
                        195,
                        239,
                        168,
                        196,
                        162,
                        19,
                        23,
                        21,
                        215,
                        235,
                        26,
                        12,
                        211,
                        250,
                        184,
                        14,
                        57,
                        116,
                        53,
                        94,
                        179,
                        92,
                        6,
                        45,
                        72,
                        140,
                        173,
                        133,
                        162,
                        150,
                        17,
                        235,
                        31,
                        82,
                        88,
                        14,
                        89,
                        143,
                        166,
                        97,
                        157,
                        250,
                        191,
                        236,
                        95,
                        115,
                        137,
                        102,
                        29,
                        61,
                        179,
                        40,
                        219,
                        182,
                        124,
                        162,
                        134,
                        146,
                        113,
                        137,
                        234,
                        30,
                        130,
                        201,
                        215,
                        22,
                        28,
                        40,
                        108,
                        174,
                        166,
                        191,
                        239,
                        251,
                        166,
                        163,
                        248,
                        245,
                        140,
                        249,
                        199,
                        168,
                        137,
                        50,
                        230,
                        83,
                        204,
                        238,
                        235,
                        156,
                        202,
                        77,
                        1,
                        12,
                        112,
                        242,
                        56,
                        189,
                        100,
                        37,
                        43,
                        139,
                        230,
                        60,
                        235,
                        94,
                        110,
                        13,
                        51,
                        230,
                        136,
                        33,
                        208,
                        191,
                        83,
                        149,
                        167,
                        17,
                        255,
                        252,
                        115,
                        11,
                        177,
                        12,
                        98,
                        208,
                        13,
                        82,
                        83,
                        78,
                        81,
                        44,
                        77,
                        166,
                        235,
                        230,
                        94,
                        52,
                        76,
                        191,
                        176,
                        18,
                        64,
                        223,
                        96,
                        145,
                        51,
                        38,
                        236,
                        143,
                        134,
                        22,
                        244,
                        116,
                        214,
                        26,
                        66,
                        199,
                        249,
                        64,
                        11,
                        164,
                        153,
                        174,
                        107,
                        201,
                        247,
                        134,
                        223,
                        136,
                        2,
                        39
                    ],
                    [
                        100,
                        2,
                        197,
                        149,
                        94,
                        78,
                        16,
                        15,
                        216,
                        212,
                        33,
                        205,
                        178,
                        90,
                        159,
                        110,
                        12,
                        9,
                        195,
                        172,
                        98,
                        84,
                        106,
                        166,
                        143,
                        8,
                        199,
                        177,
                        41,
                        127,
                        219,
                        144,
                        203,
                        178,
                        101,
                        82,
                        112,
                        39,
                        1,
                        201,
                        198,
                        130,
                        88,
                        22,
                        198,
                        20,
                        169,
                        14,
                        201,
                        230,
                        67,
                        228,
                        169,
                        137,
                        134,
                        157,
                        105,
                        111,
                        4,
                        85,
                        56,
                        183,
                        107,
                        8,
                        1,
                        230,
                        16,
                        54,
                        137,
                        81,
                        99,
                        165,
                        2,
                        191,
                        84,
                        188,
                        68,
                        200,
                        91,
                        223,
                        145,
                        201,
                        36,
                        217,
                        23,
                        124,
                        88,
                        78,
                        186,
                        186,
                        63,
                        25,
                        188,
                        95,
                        138,
                        240,
                        187,
                        154,
                        27,
                        12,
                        228,
                        173,
                        156,
                        225,
                        43,
                        200,
                        163,
                        221,
                        241,
                        105,
                        61,
                        99,
                        182,
                        150,
                        56,
                        141,
                        248,
                        113,
                        54,
                        231,
                        19,
                        51,
                        4,
                        232,
                        15,
                        70,
                        213,
                        186,
                        10,
                        247,
                        219,
                        255,
                        159,
                        30,
                        42,
                        205,
                        228,
                        91,
                        1,
                        158,
                        90,
                        6,
                        112,
                        252,
                        153,
                        234,
                        57,
                        90,
                        107,
                        172,
                        180,
                        150,
                        189,
                        188,
                        201,
                        143,
                        121,
                        38,
                        51,
                        235,
                        122,
                        163,
                        129,
                        205,
                        24,
                        30,
                        59,
                        91,
                        233,
                        1,
                        80,
                        186,
                        199,
                        153,
                        222,
                        201,
                        78,
                        156,
                        74,
                        111,
                        31,
                        105,
                        83,
                        23,
                        167,
                        55,
                        2,
                        38,
                        102,
                        254,
                        51,
                        157,
                        37,
                        83,
                        232,
                        48,
                        29,
                        108,
                        30,
                        13,
                        152,
                        151,
                        27,
                        218,
                        2,
                        59,
                        4,
                        74,
                        22,
                        127,
                        186,
                        54,
                        120,
                        127,
                        203,
                        250,
                        161,
                        6,
                        9,
                        166,
                        122,
                        112,
                        141,
                        64,
                        60,
                        192,
                        95,
                        47,
                        191,
                        8,
                        94,
                        231,
                        5,
                        11,
                        61,
                        239,
                        136,
                        85,
                        56,
                        42,
                        11,
                        224,
                        60,
                        229,
                        139,
                        244,
                        25,
                        26,
                        159,
                        166,
                        79,
                        67,
                        12,
                        111,
                        148,
                        193
                    ],
                    [
                        1,
                        118,
                        159,
                        2,
                        129,
                        184,
                        137,
                        5,
                        51,
                        164,
                        24,
                        85,
                        155,
                        119,
                        100,
                        109,
                        91,
                        14,
                        209,
                        217,
                        55,
                        243,
                        140,
                        157,
                        24,
                        70,
                        85,
                        43,
                        5,
                        8,
                        112,
                        215,
                        228,
                        90,
                        166,
                        205,
                        46,
                        79,
                        107,
                        162,
                        136,
                        139,
                        7,
                        34,
                        80,
                        253,
                        216,
                        178,
                        107,
                        67,
                        44,
                        184,
                        135,
                        90,
                        140,
                        117,
                        10,
                        237,
                        33,
                        146,
                        73,
                        88,
                        123,
                        61,
                        203,
                        227,
                        138,
                        96,
                        130,
                        148,
                        4,
                        70,
                        34,
                        234,
                        229,
                        13,
                        25,
                        202,
                        122,
                        58,
                        244,
                        228,
                        234,
                        223,
                        237,
                        124,
                        22,
                        222,
                        229,
                        79,
                        223,
                        138,
                        52,
                        50,
                        28,
                        168,
                        4,
                        214,
                        26,
                        111,
                        217,
                        22,
                        205,
                        149,
                        100,
                        36,
                        40,
                        42,
                        248,
                        58,
                        10,
                        35,
                        103,
                        175,
                        77,
                        175,
                        198,
                        195,
                        122,
                        176,
                        250,
                        57,
                        64,
                        233,
                        128,
                        200,
                        162,
                        124,
                        129,
                        200,
                        54,
                        99,
                        99,
                        237,
                        246,
                        107,
                        97,
                        196,
                        62,
                        167,
                        109,
                        187,
                        143,
                        106,
                        43,
                        133,
                        219,
                        70,
                        181,
                        42,
                        107,
                        13,
                        12,
                        146,
                        149,
                        22,
                        234,
                        39,
                        69,
                        126,
                        128,
                        174,
                        121,
                        208,
                        84,
                        98,
                        130,
                        153,
                        17,
                        20,
                        239,
                        13,
                        190,
                        143,
                        247,
                        160,
                        214,
                        157,
                        53,
                        196,
                        181,
                        181,
                        187,
                        175,
                        76,
                        97,
                        142,
                        193,
                        183,
                        80,
                        88,
                        109,
                        73,
                        178,
                        79,
                        222,
                        47,
                        193,
                        232,
                        233,
                        110,
                        215,
                        229,
                        80,
                        49,
                        145,
                        59,
                        202,
                        136,
                        50,
                        49,
                        12,
                        253,
                        21,
                        122,
                        80,
                        183,
                        142,
                        34,
                        141,
                        237,
                        142,
                        23,
                        99,
                        69,
                        231,
                        105,
                        76,
                        248,
                        237,
                        130,
                        200,
                        215,
                        160,
                        59,
                        25,
                        198,
                        105,
                        130,
                        20,
                        96,
                        200,
                        183,
                        159,
                        232,
                        177,
                        244,
                        84,
                        169,
                        245,
                        209,
                        111,
                        53,
                        240,
                        123,
                        11,
                        152
                    ],
                    [
                        2,
                        138,
                        96,
                        92,
                        255,
                        34,
                        116,
                        173,
                        20,
                        69,
                        199,
                        3,
                        5,
                        92,
                        201,
                        32,
                        201,
                        31,
                        179,
                        150,
                        90,
                        107,
                        31,
                        3,
                        191,
                        223,
                        78,
                        115,
                        65,
                        64,
                        16,
                        87,
                        247,
                        247,
                        21,
                        69,
                        196,
                        57,
                        136,
                        39,
                        234,
                        158,
                        1,
                        163,
                        252,
                        36,
                        57,
                        107,
                        168,
                        117,
                        225,
                        98,
                        29,
                        146,
                        235,
                        106,
                        133,
                        38,
                        101,
                        9,
                        184,
                        149,
                        75,
                        179,
                        75,
                        156,
                        5,
                        109,
                        37,
                        180,
                        150,
                        97,
                        61,
                        70,
                        97,
                        32,
                        135,
                        82,
                        71,
                        4,
                        200,
                        150,
                        253,
                        125,
                        232,
                        119,
                        231,
                        74,
                        221,
                        185,
                        139,
                        56,
                        214,
                        209,
                        46,
                        138,
                        92,
                        102,
                        93,
                        249,
                        240,
                        97,
                        245,
                        177,
                        115,
                        108,
                        189,
                        68,
                        93,
                        85,
                        108,
                        216,
                        40,
                        161,
                        55,
                        32,
                        13,
                        34,
                        12,
                        198,
                        184,
                        69,
                        10,
                        191,
                        38,
                        79,
                        194,
                        167,
                        19,
                        135,
                        195,
                        62,
                        245,
                        248,
                        122,
                        144,
                        132,
                        233,
                        238,
                        78,
                        242,
                        137,
                        129,
                        117,
                        210,
                        244,
                        53,
                        87,
                        73,
                        246,
                        30,
                        223,
                        83,
                        0,
                        84,
                        83,
                        36,
                        211,
                        231,
                        24,
                        60,
                        58,
                        114,
                        223,
                        218,
                        47,
                        32,
                        47,
                        34,
                        227,
                        224,
                        122,
                        50,
                        215,
                        242,
                        198,
                        104,
                        205,
                        192,
                        11,
                        142,
                        139,
                        17,
                        101,
                        236,
                        88,
                        9,
                        119,
                        137,
                        218,
                        215,
                        73,
                        235,
                        183,
                        59,
                        223,
                        42,
                        203,
                        218,
                        76,
                        184,
                        27,
                        70,
                        225,
                        6,
                        151,
                        2,
                        183,
                        106,
                        124,
                        14,
                        219,
                        58,
                        71,
                        100,
                        2,
                        135,
                        124,
                        43,
                        178,
                        12,
                        140,
                        45,
                        136,
                        135,
                        69,
                        195,
                        219,
                        63,
                        249,
                        58,
                        140,
                        198,
                        123,
                        143,
                        203,
                        132,
                        105,
                        55,
                        36,
                        14,
                        107,
                        211,
                        251,
                        173,
                        102,
                        241,
                        193,
                        165,
                        3,
                        168,
                        108,
                        93,
                        127,
                        3,
                        162,
                        227
                    ],
                    [
                        1,
                        185,
                        5,
                        29,
                        44,
                        82,
                        241,
                        206,
                        149,
                        5,
                        122,
                        252,
                        235,
                        120,
                        16,
                        15,
                        71,
                        16,
                        151,
                        103,
                        254,
                        245,
                        217,
                        73,
                        207,
                        230,
                        48,
                        243,
                        78,
                        241,
                        168,
                        104,
                        15,
                        36,
                        251,
                        86,
                        253,
                        17,
                        224,
                        55,
                        55,
                        167,
                        239,
                        241,
                        16,
                        62,
                        0,
                        100,
                        53,
                        9,
                        36,
                        151,
                        215,
                        143,
                        218,
                        214,
                        72,
                        24,
                        152,
                        42,
                        144,
                        168,
                        100,
                        122,
                        101,
                        248,
                        55,
                        109,
                        225,
                        78,
                        58,
                        108,
                        185,
                        206,
                        44,
                        23,
                        114,
                        116,
                        222,
                        91,
                        168,
                        112,
                        48,
                        141,
                        64,
                        71,
                        142,
                        191,
                        255,
                        83,
                        126,
                        61,
                        160,
                        123,
                        215,
                        116,
                        45,
                        198,
                        122,
                        62,
                        63,
                        107,
                        40,
                        58,
                        56,
                        166,
                        148,
                        204,
                        220,
                        10,
                        67,
                        200,
                        94,
                        140,
                        173,
                        98,
                        26,
                        61,
                        146,
                        74,
                        106,
                        73,
                        162,
                        150,
                        210,
                        96,
                        244,
                        191,
                        80,
                        109,
                        153,
                        157,
                        59,
                        31,
                        151,
                        218,
                        156,
                        244,
                        212,
                        208,
                        160,
                        112,
                        220,
                        134,
                        64,
                        28,
                        164,
                        111,
                        219,
                        198,
                        234,
                        130,
                        54,
                        20,
                        217,
                        56,
                        115,
                        0,
                        28,
                        44,
                        18,
                        3,
                        8,
                        70,
                        248,
                        157,
                        67,
                        198,
                        216,
                        69,
                        232,
                        236,
                        111,
                        145,
                        191,
                        214,
                        186,
                        208,
                        126,
                        133,
                        151,
                        166,
                        251,
                        30,
                        26,
                        163,
                        255,
                        234,
                        241,
                        251,
                        253,
                        132,
                        247,
                        204,
                        95,
                        124,
                        142,
                        76,
                        250,
                        115,
                        91,
                        240,
                        169,
                        203,
                        162,
                        57,
                        41,
                        42,
                        150,
                        242,
                        72,
                        227,
                        223,
                        76,
                        149,
                        87,
                        153,
                        77,
                        193,
                        63,
                        159,
                        32,
                        190,
                        32,
                        126,
                        53,
                        26,
                        99,
                        95,
                        59,
                        205,
                        22,
                        161,
                        9,
                        195,
                        16,
                        48,
                        79,
                        53,
                        235,
                        46,
                        71,
                        0,
                        8,
                        57,
                        55,
                        6,
                        87,
                        1,
                        198,
                        107,
                        255,
                        135,
                        80,
                        239,
                        33,
                        47
                    ]
                ]
            }
        },
        "requested_proof": {
            "revealed_attrs": {
                "0_player_uuid": {
                    "sub_proof_index": 0,
                    "raw": "Richie Knucklez",
                    "encoded": "51643998292319337989293919354395093705917445045137690661130646028663839100479"
                },
                "0_screencapture_uuid": {
                    "sub_proof_index": 0,
                    "raw": "aW1hZ2luZSBhIHNjcmVlbiBjYXB0dXJl",
                    "encoded": "44349549498354774830429200435932754610833874667251545521048906777181334567815"
                }
            },
            "self_attested_attrs": {},
            "unrevealed_attrs": {},
            "predicates": {
                "0_highscore_GE_uuid": {
                    "sub_proof_index": 0
                }
            }
        },
        "identifiers": [
            {
                "schema_id": "WjFgAM9qFept242HWzUSTZ:2:high_score:1.0",
                "cred_def_id": "WjFgAM9qFept242HWzUSTZ:3:CL:13:tag",
                "rev_reg_id": null,
                "timestamp": null
            }
        ]
    }"""
)

PRES = Presentation(
    comment="Test",
    presentations_attach=[
        AttachDecorator.data_base64(
            mapping=INDY_PROOF,
            ident=ATTACH_DECO_IDS[PRESENTATION],
        )
    ],
)


class TestPresentation(TestCase):
    """Presentation tests."""

    def test_init(self):
        """Test initializer."""
        assert PRES.presentations_attach[0].content == INDY_PROOF
        assert PRES.indy_proof(0) == INDY_PROOF

    def test_type(self):
        """Test type."""
        assert PRES._type == DIDCommPrefix.qualify_current(PRESENTATION)

    def test_deserialize(self):
        """Test deserialization."""
        dump = json.dumps(
            {
                "@type": DIDCommPrefix.qualify_current(PRESENTATION),
                "comment": "Hello World",
                "presentations~attach": [
                    AttachDecorator.data_base64(
                        mapping=INDY_PROOF,
                        ident=ATTACH_DECO_IDS[PRESENTATION],
                    ).serialize()
                ],
            }
        )

        presentation = Presentation.deserialize(dump)
        assert type(presentation) == Presentation

    def test_serialize(self):
        """Test serialization."""
        pres_dict = PRES.serialize()
        pres_dict.pop("@id")

        assert pres_dict == {
            "@type": DIDCommPrefix.qualify_current(PRESENTATION),
            "presentations~attach": [
                AttachDecorator.data_base64(
                    mapping=INDY_PROOF,
                    ident=ATTACH_DECO_IDS[PRESENTATION],
                ).serialize()
            ],
            "comment": "Test",
        }


class TestPresentationSchema(TestCase):
    """Test presentation schema"""

    def test_make_model(self):
        """Test making model."""
        pres_dict = PRES.serialize()
        """
        Looks like: {
            "@type": ".../present-proof/1.0/presentation",
            "@id": "f49773e3-bd56-4868-a5f1-456d1e6d1a16",
            "comment": "Test",
            "presentations~attach": [
                {
                    "mime-type": "application/json",
                    "data": {
                        "base64": "eyJuYW..."
                    }
                }
            ]
        }
        """

        model_instance = PRES.deserialize(pres_dict)
        assert isinstance(model_instance, Presentation)
