"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InputType = exports.TaskInput = void 0;
const fields_1 = require("./fields");
/**
 * Type union for task classes that accept multiple types of payload.
 */
class TaskInput {
    /**
     *
     * @param type type of task input
     * @param value payload for the corresponding input type.
     * It can be a JSON-encoded object, context, data, etc.
     */
    constructor(type, value) {
        this.type = type;
        this.value = value;
    }
    /**
     * Use a literal string as task input.
     *
     * This might be a JSON-encoded object, or just a text.
     */
    static fromText(text) {
        return new TaskInput(InputType.TEXT, text);
    }
    /**
     * Use an object as task input.
     *
     * This object may contain Data and Context fields
     * as object values, if desired.
     */
    static fromObject(obj) {
        return new TaskInput(InputType.OBJECT, obj);
    }
    /**
     * Use a part of the execution data or task context as task input.
     *
     * Use this when you want to use a subobject or string from
     * the current state machine execution or the current task context
     * as complete payload to a task.
     */
    static fromJsonPathAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.JsonPath.stringAt(path));
    }
    /**
     * Use a part of the execution data as task input.
     *
     * Use this when you want to use a subobject or string from
     * the current state machine execution as complete payload
     * to a task.
     */
    static fromDataAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.JsonPath.stringAt(path));
    }
    /**
     * Use a part of the task context as task input.
     *
     * Use this when you want to use a subobject or string from
     * the current task context as complete payload
     * to a task.
     */
    static fromContextAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.JsonPath.stringAt(path));
    }
}
exports.TaskInput = TaskInput;
/**
 * The type of task input.
 */
var InputType;
(function (InputType) {
    InputType[InputType["TEXT"] = 0] = "TEXT";
    InputType[InputType["OBJECT"] = 1] = "OBJECT";
})(InputType = exports.InputType || (exports.InputType = {}));
//# sourceMappingURL=data:application/json;base64,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