"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineDeployStackAction = void 0;
const cfn = require("@aws-cdk/aws-cloudformation");
const cpactions = require("@aws-cdk/aws-codepipeline-actions");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cdk = require("@aws-cdk/core");
/**
 * A class to deploy a stack that is part of a CDK App, using CodePipeline.
 * This composite Action takes care of preparing and executing a CloudFormation ChangeSet.
 *
 * It currently does *not* support stacks that make use of ``Asset``s, and
 * requires the deployed stack is in the same account and region where the
 * CodePipeline is hosted.
 */
class PipelineDeployStackAction {
    constructor(props) {
        var _a, _b;
        this.stack = props.stack;
        const assets = this.stack.node.metadata.filter(md => md.type === cxschema.ArtifactMetadataEntryType.ASSET);
        if (assets.length > 0) {
            // FIXME: Implement the necessary actions to publish assets
            throw new Error(`Cannot deploy the stack ${this.stack.stackName} because it references ${assets.length} asset(s)`);
        }
        const createChangeSetRunOrder = props.createChangeSetRunOrder || 1;
        const executeChangeSetRunOrder = props.executeChangeSetRunOrder || (createChangeSetRunOrder + 1);
        if (createChangeSetRunOrder >= executeChangeSetRunOrder) {
            throw new Error(`createChangeSetRunOrder (${createChangeSetRunOrder}) must be < executeChangeSetRunOrder (${executeChangeSetRunOrder})`);
        }
        const changeSetName = props.changeSetName || 'CDK-CodePipeline-ChangeSet';
        const capabilities = cfnCapabilities(props.adminPermissions, props.capabilities);
        this.prepareChangeSetAction = new cpactions.CloudFormationCreateReplaceChangeSetAction({
            actionName: (_a = props.createChangeSetActionName) !== null && _a !== void 0 ? _a : 'ChangeSet',
            changeSetName,
            runOrder: createChangeSetRunOrder,
            stackName: props.stack.stackName,
            templatePath: props.input.atPath(props.stack.templateFile),
            adminPermissions: props.adminPermissions,
            deploymentRole: props.role,
            capabilities,
        });
        this.executeChangeSetAction = new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: (_b = props.executeChangeSetActionName) !== null && _b !== void 0 ? _b : 'Execute',
            changeSetName,
            runOrder: executeChangeSetRunOrder,
            stackName: this.stack.stackName,
        });
    }
    bind(scope, stage, options) {
        if (this.stack.environment !== cdk.Stack.of(scope).environment) {
            // FIXME: Add the necessary to extend to stacks in a different account
            throw new Error('Cross-environment deployment is not supported');
        }
        stage.addAction(this.prepareChangeSetAction);
        this._deploymentRole = this.prepareChangeSetAction.deploymentRole;
        return this.executeChangeSetAction.bind(scope, stage, options);
    }
    get deploymentRole() {
        if (!this._deploymentRole) {
            throw new Error('Use this action in a pipeline first before accessing \'deploymentRole\'');
        }
        return this._deploymentRole;
    }
    /**
     * Add policy statements to the role deploying the stack.
     *
     * This role is passed to CloudFormation and must have the IAM permissions
     * necessary to deploy the stack or you can grant this role `adminPermissions`
     * by using that option during creation. If you do not grant
     * `adminPermissions` you need to identify the proper statements to add to
     * this role based on the CloudFormation Resources in your stack.
     */
    addToDeploymentRolePolicy(statement) {
        this.deploymentRole.addToPolicy(statement);
    }
    onStateChange(name, target, options) {
        return this.executeChangeSetAction.onStateChange(name, target, options);
    }
    get actionProperties() {
        return this.executeChangeSetAction.actionProperties;
    }
}
exports.PipelineDeployStackAction = PipelineDeployStackAction;
function cfnCapabilities(adminPermissions, capabilities) {
    if (adminPermissions && capabilities === undefined) {
        // admin true default capability to NamedIAM and AutoExpand
        return [cfn.CloudFormationCapabilities.NAMED_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND];
    }
    else if (capabilities === undefined) {
        // else capabilities are undefined set AnonymousIAM and AutoExpand
        return [cfn.CloudFormationCapabilities.ANONYMOUS_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND];
    }
    else {
        // else capabilities are defined use them
        return capabilities;
    }
}
//# sourceMappingURL=data:application/json;base64,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