"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const sfn = require("../lib");
describe('Task base', () => {
    let stack;
    let task;
    beforeEach(() => {
        // GIVEN
        stack = new cdk.Stack();
        task = new FakeTask(stack, 'my-task', {
            metrics: {
                metricPrefixPlural: '',
                metricPrefixSingular: '',
            },
        });
    });
    test('instantiate a concrete implementation with properties', () => {
        // WHEN
        task = new FakeTask(stack, 'my-exciting-task', {
            comment: 'my exciting task',
            heartbeat: cdk.Duration.seconds(10),
            timeout: cdk.Duration.minutes(10),
        });
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-exciting-task',
            States: {
                'my-exciting-task': {
                    End: true,
                    Type: 'Task',
                    Comment: 'my exciting task',
                    TimeoutSeconds: 600,
                    HeartbeatSeconds: 10,
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
            },
        });
    });
    test('add catch configuration', () => {
        // GIVEN
        const failure = new sfn.Fail(stack, 'failed', {
            error: 'DidNotWork',
            cause: 'We got stuck',
        });
        // WHEN
        task.addCatch(failure);
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-task',
            States: {
                'my-task': {
                    End: true,
                    Catch: [{
                            ErrorEquals: ['States.ALL'],
                            Next: 'failed',
                        }],
                    Type: 'Task',
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
                'failed': {
                    Type: 'Fail',
                    Error: 'DidNotWork',
                    Cause: 'We got stuck',
                },
            },
        });
    });
    test('States.ALL catch appears at end of list', () => {
        // GIVEN
        const httpFailure = new sfn.Fail(stack, 'http', { error: 'HTTP' });
        const otherFailure = new sfn.Fail(stack, 'other', { error: 'Other' });
        const allFailure = new sfn.Fail(stack, 'all');
        // WHEN
        task
            .addCatch(httpFailure, { errors: ['HTTPError'] })
            .addCatch(allFailure)
            .addCatch(otherFailure, { errors: ['OtherError'] });
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-task',
            States: {
                'all': {
                    Type: 'Fail',
                },
                'http': {
                    Error: 'HTTP',
                    Type: 'Fail',
                },
                'my-task': {
                    End: true,
                    Catch: [
                        {
                            ErrorEquals: ['HTTPError'],
                            Next: 'http',
                        },
                        {
                            ErrorEquals: ['OtherError'],
                            Next: 'other',
                        },
                        {
                            ErrorEquals: ['States.ALL'],
                            Next: 'all',
                        },
                    ],
                    Type: 'Task',
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
                'other': {
                    Error: 'Other',
                    Type: 'Fail',
                },
            },
        });
    });
    test('addCatch throws when errors are combined with States.ALL', () => {
        // GIVEN
        const failure = new sfn.Fail(stack, 'failed', {
            error: 'DidNotWork',
            cause: 'We got stuck',
        });
        expect(() => task.addCatch(failure, {
            errors: ['States.ALL', 'HTTPError'],
        })).toThrow(/must appear alone/);
    });
    test('add retry configuration', () => {
        // WHEN
        task.addRetry({ errors: ['HTTPError'], maxAttempts: 2 })
            .addRetry(); // adds default retry
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-task',
            States: {
                'my-task': {
                    End: true,
                    Retry: [
                        {
                            ErrorEquals: ['HTTPError'],
                            MaxAttempts: 2,
                        },
                        {
                            ErrorEquals: ['States.ALL'],
                        },
                    ],
                    Type: 'Task',
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
            },
        });
    });
    test('States.ALL retry appears at end of list', () => {
        // WHEN
        task
            .addRetry({ errors: ['HTTPError'] })
            .addRetry()
            .addRetry({ errors: ['OtherError'] });
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-task',
            States: {
                'my-task': {
                    End: true,
                    Retry: [
                        {
                            ErrorEquals: ['HTTPError'],
                        },
                        {
                            ErrorEquals: ['OtherError'],
                        },
                        {
                            ErrorEquals: ['States.ALL'],
                        },
                    ],
                    Type: 'Task',
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
            },
        });
    });
    test('addRetry throws when errors are combined with States.ALL', () => {
        expect(() => task.addRetry({
            errors: ['States.ALL', 'HTTPError'],
        })).toThrow(/must appear alone/);
    });
    test('add a next state to the task in the chain', () => {
        // WHEN
        task.next(new sfn.Pass(stack, 'passState'));
        // THEN
        expect(render(task)).toEqual({
            StartAt: 'my-task',
            States: {
                'my-task': {
                    Next: 'passState',
                    Type: 'Task',
                    Resource: 'my-resource',
                    Parameters: { MyParameter: 'myParameter' },
                },
                'passState': { Type: 'Pass', End: true },
            },
        });
    });
    test('get named metric for this task', () => {
        // WHEN
        const metric = task.metric('my-metric');
        // THEN
        verifyMetric(metric, 'my-metric', 'Sum');
    });
    test('add metric for task state run time', () => {
        // WHEN
        const metric = task.metricRunTime();
        // THEN
        verifyMetric(metric, 'RunTime', 'Average');
    });
    test('add metric for task schedule time', () => {
        // WHEN
        const metric = task.metricScheduleTime();
        // THEN
        verifyMetric(metric, 'ScheduleTime', 'Average');
    });
    test('add metric for time between task being scheduled to closing', () => {
        // WHEN
        const metric = task.metricTime();
        // THEN
        verifyMetric(metric, 'Time', 'Average');
    });
    test('add metric for number of times the task is scheduled', () => {
        // WHEN
        const metric = task.metricScheduled();
        // THEN
        verifyMetric(metric, 'Scheduled', 'Sum');
    });
    test('add metric for number of times the task times out', () => {
        // WHEN
        const metric = task.metricTimedOut();
        // THEN
        verifyMetric(metric, 'TimedOut', 'Sum');
    });
    test('add metric for number of times the task was started', () => {
        // WHEN
        const metric = task.metricStarted();
        // THEN
        verifyMetric(metric, 'Started', 'Sum');
    });
    test('add metric for number of times the task succeeded', () => {
        // WHEN
        const metric = task.metricSucceeded();
        // THEN
        verifyMetric(metric, 'Succeeded', 'Sum');
    });
    test('add metric for number of times the task failed', () => {
        // WHEN
        const metric = task.metricFailed();
        // THEN
        verifyMetric(metric, 'Failed', 'Sum');
    });
    test('add metric for number of times the metrics heartbeat timed out', () => {
        // WHEN
        const metric = task.metricHeartbeatTimedOut();
        // THEN
        verifyMetric(metric, 'HeartbeatTimedOut', 'Sum');
    });
    test('metrics must be configured to use metric* APIs', () => {
        // GIVEN
        task = new FakeTask(stack, 'mytask', {});
        // THEN
        expect(() => {
            task.metricFailed();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricHeartbeatTimedOut();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricRunTime();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricScheduleTime();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricScheduled();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricStarted();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricSucceeded();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricTime();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
        expect(() => {
            task.metricTimedOut();
        }).toThrow('Task does not expose metrics. Use the \'metric()\' function to add metrics.');
    });
});
function verifyMetric(metric, metricName, statistic) {
    expect(metric).toEqual(expect.objectContaining({
        namespace: 'AWS/States',
        metricName,
        statistic,
    }));
}
function render(sm) {
    return new cdk.Stack().resolve(new sfn.StateGraph(sm.startState, 'Test Graph').toGraphJson());
}
class FakeTask extends sfn.TaskStateBase {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.taskMetrics = props.metrics;
    }
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: 'my-resource',
            Parameters: sfn.FieldUtils.renderObject({
                MyParameter: 'myParameter',
            }),
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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