"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const stepfunctions = require("../lib");
describe('State Machine', () => {
    test('Instantiate Default State Machine', () => {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new stepfunctions.StateMachine(stack, 'MyStateMachine', {
            stateMachineName: 'MyStateMachine',
            definition: stepfunctions.Chain.start(new stepfunctions.Pass(stack, 'Pass')),
        });
        // THEN
        expect(stack).toHaveResource('AWS::StepFunctions::StateMachine', {
            StateMachineName: 'MyStateMachine',
            DefinitionString: '{"StartAt":"Pass","States":{"Pass":{"Type":"Pass","End":true}}}',
        });
    }),
        test('Instantiate Standard State Machine', () => {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new stepfunctions.StateMachine(stack, 'MyStateMachine', {
                stateMachineName: 'MyStateMachine',
                definition: stepfunctions.Chain.start(new stepfunctions.Pass(stack, 'Pass')),
                stateMachineType: stepfunctions.StateMachineType.STANDARD,
            });
            // THEN
            expect(stack).toHaveResource('AWS::StepFunctions::StateMachine', {
                StateMachineName: 'MyStateMachine',
                StateMachineType: 'STANDARD',
                DefinitionString: '{"StartAt":"Pass","States":{"Pass":{"Type":"Pass","End":true}}}',
            });
        }),
        test('Instantiate Express State Machine', () => {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            new stepfunctions.StateMachine(stack, 'MyStateMachine', {
                stateMachineName: 'MyStateMachine',
                definition: stepfunctions.Chain.start(new stepfunctions.Pass(stack, 'Pass')),
                stateMachineType: stepfunctions.StateMachineType.EXPRESS,
            });
            // THEN
            expect(stack).toHaveResource('AWS::StepFunctions::StateMachine', {
                StateMachineName: 'MyStateMachine',
                StateMachineType: 'EXPRESS',
                DefinitionString: '{"StartAt":"Pass","States":{"Pass":{"Type":"Pass","End":true}}}',
            });
        }),
        test('log configuration', () => {
            // GIVEN
            const stack = new cdk.Stack();
            // WHEN
            const logGroup = new logs.LogGroup(stack, 'MyLogGroup');
            new stepfunctions.StateMachine(stack, 'MyStateMachine', {
                definition: stepfunctions.Chain.start(new stepfunctions.Pass(stack, 'Pass')),
                logs: {
                    destination: logGroup,
                    level: stepfunctions.LogLevel.FATAL,
                    includeExecutionData: false,
                },
            });
            // THEN
            expect(stack).toHaveResource('AWS::StepFunctions::StateMachine', {
                DefinitionString: '{"StartAt":"Pass","States":{"Pass":{"Type":"Pass","End":true}}}',
                LoggingConfiguration: {
                    Destinations: [{
                            CloudWatchLogsLogGroup: {
                                LogGroupArn: {
                                    'Fn::GetAtt': ['MyLogGroup5C0DAD85', 'Arn'],
                                },
                            },
                        }],
                    IncludeExecutionData: false,
                    Level: 'FATAL',
                },
            });
            expect(stack).toHaveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [{
                            Action: [
                                'logs:CreateLogDelivery',
                                'logs:GetLogDelivery',
                                'logs:UpdateLogDelivery',
                                'logs:DeleteLogDelivery',
                                'logs:ListLogDeliveries',
                                'logs:PutResourcePolicy',
                                'logs:DescribeResourcePolicies',
                                'logs:DescribeLogGroups',
                            ],
                            Effect: 'Allow',
                            Resource: '*',
                        }],
                    Version: '2012-10-17',
                },
                PolicyName: 'MyStateMachineRoleDefaultPolicyE468EB18',
                Roles: [
                    {
                        Ref: 'MyStateMachineRoleD59FFEBC',
                    },
                ],
            });
        });
    test('tracing configuration', () => {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        new stepfunctions.StateMachine(stack, 'MyStateMachine', {
            definition: stepfunctions.Chain.start(new stepfunctions.Pass(stack, 'Pass')),
            tracingEnabled: true,
        });
        // THEN
        expect(stack).toHaveResource('AWS::StepFunctions::StateMachine', {
            DefinitionString: '{"StartAt":"Pass","States":{"Pass":{"Type":"Pass","End":true}}}',
            TracingConfiguration: {
                Enabled: true,
            },
        });
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [{
                        Action: [
                            'xray:PutTraceSegments',
                            'xray:PutTelemetryRecords',
                        ],
                        Effect: 'Allow',
                        Resource: '*',
                    }],
                Version: '2012-10-17',
            },
            PolicyName: 'MyStateMachineRoleDefaultPolicyE468EB18',
            Roles: [
                {
                    Ref: 'MyStateMachineRoleD59FFEBC',
                },
            ],
        });
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhdGUtbWFjaGluZS50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic3RhdGUtbWFjaGluZS50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsZ0NBQThCO0FBQzlCLDBDQUEwQztBQUMxQyxxQ0FBcUM7QUFDckMsd0NBQXdDO0FBRXhDLFFBQVEsQ0FBQyxlQUFlLEVBQUUsR0FBRyxFQUFFO0lBQzdCLElBQUksQ0FBQyxtQ0FBbUMsRUFBRSxHQUFHLEVBQUU7UUFDN0MsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1FBRTlCLE9BQU87UUFDUCxJQUFJLGFBQWEsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFO1lBQ3RELGdCQUFnQixFQUFFLGdCQUFnQjtZQUNsQyxVQUFVLEVBQUUsYUFBYSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsSUFBSSxhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztTQUM3RSxDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxrQ0FBa0MsRUFBRTtZQUMvRCxnQkFBZ0IsRUFBRSxnQkFBZ0I7WUFDbEMsZ0JBQWdCLEVBQUUsaUVBQWlFO1NBQ3BGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQztRQUVGLElBQUksQ0FBQyxvQ0FBb0MsRUFBRSxHQUFHLEVBQUU7WUFDOUMsUUFBUTtZQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBRTlCLE9BQU87WUFDUCxJQUFJLGFBQWEsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFO2dCQUN0RCxnQkFBZ0IsRUFBRSxnQkFBZ0I7Z0JBQ2xDLFVBQVUsRUFBRSxhQUFhLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxJQUFJLGFBQWEsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO2dCQUM1RSxnQkFBZ0IsRUFBRSxhQUFhLENBQUMsZ0JBQWdCLENBQUMsUUFBUTthQUMxRCxDQUFDLENBQUM7WUFFSCxPQUFPO1lBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxrQ0FBa0MsRUFBRTtnQkFDL0QsZ0JBQWdCLEVBQUUsZ0JBQWdCO2dCQUNsQyxnQkFBZ0IsRUFBRSxVQUFVO2dCQUM1QixnQkFBZ0IsRUFBRSxpRUFBaUU7YUFDcEYsQ0FBQyxDQUFDO1FBRUwsQ0FBQyxDQUFDO1FBRUYsSUFBSSxDQUFDLG1DQUFtQyxFQUFFLEdBQUcsRUFBRTtZQUM3QyxRQUFRO1lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFFOUIsT0FBTztZQUNQLElBQUksYUFBYSxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7Z0JBQ3RELGdCQUFnQixFQUFFLGdCQUFnQjtnQkFDbEMsVUFBVSxFQUFFLGFBQWEsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLElBQUksYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7Z0JBQzVFLGdCQUFnQixFQUFFLGFBQWEsQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPO2FBQ3pELENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGtDQUFrQyxFQUFFO2dCQUMvRCxnQkFBZ0IsRUFBRSxnQkFBZ0I7Z0JBQ2xDLGdCQUFnQixFQUFFLFNBQVM7Z0JBQzNCLGdCQUFnQixFQUFFLGlFQUFpRTthQUNwRixDQUFDLENBQUM7UUFFTCxDQUFDLENBQUM7UUFFRixJQUFJLENBQUMsbUJBQW1CLEVBQUUsR0FBRyxFQUFFO1lBQzdCLFFBQVE7WUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUU5QixPQUFPO1lBQ1AsTUFBTSxRQUFRLEdBQUcsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztZQUV4RCxJQUFJLGFBQWEsQ0FBQyxZQUFZLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFO2dCQUN0RCxVQUFVLEVBQUUsYUFBYSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsSUFBSSxhQUFhLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQztnQkFDNUUsSUFBSSxFQUFFO29CQUNKLFdBQVcsRUFBRSxRQUFRO29CQUNyQixLQUFLLEVBQUUsYUFBYSxDQUFDLFFBQVEsQ0FBQyxLQUFLO29CQUNuQyxvQkFBb0IsRUFBRSxLQUFLO2lCQUM1QjthQUNGLENBQUMsQ0FBQztZQUVILE9BQU87WUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGtDQUFrQyxFQUFFO2dCQUMvRCxnQkFBZ0IsRUFBRSxpRUFBaUU7Z0JBQ25GLG9CQUFvQixFQUFFO29CQUNwQixZQUFZLEVBQUUsQ0FBQzs0QkFDYixzQkFBc0IsRUFBRTtnQ0FDdEIsV0FBVyxFQUFFO29DQUNYLFlBQVksRUFBRSxDQUFDLG9CQUFvQixFQUFFLEtBQUssQ0FBQztpQ0FDNUM7NkJBQ0Y7eUJBQ0YsQ0FBQztvQkFDRixvQkFBb0IsRUFBRSxLQUFLO29CQUMzQixLQUFLLEVBQUUsT0FBTztpQkFDZjthQUNGLENBQUMsQ0FBQztZQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsa0JBQWtCLEVBQUU7Z0JBQy9DLGNBQWMsRUFBRTtvQkFDZCxTQUFTLEVBQUUsQ0FBQzs0QkFDVixNQUFNLEVBQUU7Z0NBQ04sd0JBQXdCO2dDQUN4QixxQkFBcUI7Z0NBQ3JCLHdCQUF3QjtnQ0FDeEIsd0JBQXdCO2dDQUN4Qix3QkFBd0I7Z0NBQ3hCLHdCQUF3QjtnQ0FDeEIsK0JBQStCO2dDQUMvQix3QkFBd0I7NkJBQ3pCOzRCQUNELE1BQU0sRUFBRSxPQUFPOzRCQUNmLFFBQVEsRUFBRSxHQUFHO3lCQUNkLENBQUM7b0JBQ0YsT0FBTyxFQUFFLFlBQVk7aUJBQ3RCO2dCQUNELFVBQVUsRUFBRSx5Q0FBeUM7Z0JBQ3JELEtBQUssRUFBRTtvQkFDTDt3QkFDRSxHQUFHLEVBQUUsNEJBQTRCO3FCQUNsQztpQkFDRjthQUNGLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHVCQUF1QixFQUFFLEdBQUcsRUFBRTtRQUNqQyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7UUFFOUIsT0FBTztRQUNQLElBQUksYUFBYSxDQUFDLFlBQVksQ0FBQyxLQUFLLEVBQUUsZ0JBQWdCLEVBQUU7WUFDdEQsVUFBVSxFQUFFLGFBQWEsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLElBQUksYUFBYSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7WUFDNUUsY0FBYyxFQUFFLElBQUk7U0FDckIsQ0FBQyxDQUFDO1FBRUgsT0FBTztRQUNQLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsa0NBQWtDLEVBQUU7WUFDL0QsZ0JBQWdCLEVBQUUsaUVBQWlFO1lBQ25GLG9CQUFvQixFQUFFO2dCQUNwQixPQUFPLEVBQUUsSUFBSTthQUNkO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsRUFBRTtZQUMvQyxjQUFjLEVBQUU7Z0JBQ2QsU0FBUyxFQUFFLENBQUM7d0JBQ1YsTUFBTSxFQUFFOzRCQUNOLHVCQUF1Qjs0QkFDdkIsMEJBQTBCO3lCQUMzQjt3QkFDRCxNQUFNLEVBQUUsT0FBTzt3QkFDZixRQUFRLEVBQUUsR0FBRztxQkFDZCxDQUFDO2dCQUNGLE9BQU8sRUFBRSxZQUFZO2FBQ3RCO1lBQ0QsVUFBVSxFQUFFLHlDQUF5QztZQUNyRCxLQUFLLEVBQUU7Z0JBQ0w7b0JBQ0UsR0FBRyxFQUFFLDRCQUE0QjtpQkFDbEM7YUFDRjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgJ0Bhd3MtY2RrL2Fzc2VydC9qZXN0JztcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0ICogYXMgc3RlcGZ1bmN0aW9ucyBmcm9tICcuLi9saWInO1xuXG5kZXNjcmliZSgnU3RhdGUgTWFjaGluZScsICgpID0+IHtcbiAgdGVzdCgnSW5zdGFudGlhdGUgRGVmYXVsdCBTdGF0ZSBNYWNoaW5lJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lKHN0YWNrLCAnTXlTdGF0ZU1hY2hpbmUnLCB7XG4gICAgICBzdGF0ZU1hY2hpbmVOYW1lOiAnTXlTdGF0ZU1hY2hpbmUnLFxuICAgICAgZGVmaW5pdGlvbjogc3RlcGZ1bmN0aW9ucy5DaGFpbi5zdGFydChuZXcgc3RlcGZ1bmN0aW9ucy5QYXNzKHN0YWNrLCAnUGFzcycpKSxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OlN0ZXBGdW5jdGlvbnM6OlN0YXRlTWFjaGluZScsIHtcbiAgICAgIFN0YXRlTWFjaGluZU5hbWU6ICdNeVN0YXRlTWFjaGluZScsXG4gICAgICBEZWZpbml0aW9uU3RyaW5nOiAne1wiU3RhcnRBdFwiOlwiUGFzc1wiLFwiU3RhdGVzXCI6e1wiUGFzc1wiOntcIlR5cGVcIjpcIlBhc3NcIixcIkVuZFwiOnRydWV9fX0nLFxuICAgIH0pO1xuICB9KSxcblxuICB0ZXN0KCdJbnN0YW50aWF0ZSBTdGFuZGFyZCBTdGF0ZSBNYWNoaW5lJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lKHN0YWNrLCAnTXlTdGF0ZU1hY2hpbmUnLCB7XG4gICAgICBzdGF0ZU1hY2hpbmVOYW1lOiAnTXlTdGF0ZU1hY2hpbmUnLFxuICAgICAgZGVmaW5pdGlvbjogc3RlcGZ1bmN0aW9ucy5DaGFpbi5zdGFydChuZXcgc3RlcGZ1bmN0aW9ucy5QYXNzKHN0YWNrLCAnUGFzcycpKSxcbiAgICAgIHN0YXRlTWFjaGluZVR5cGU6IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lVHlwZS5TVEFOREFSRCxcbiAgICB9KTtcblxuICAgIC8vIFRIRU5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OlN0ZXBGdW5jdGlvbnM6OlN0YXRlTWFjaGluZScsIHtcbiAgICAgIFN0YXRlTWFjaGluZU5hbWU6ICdNeVN0YXRlTWFjaGluZScsXG4gICAgICBTdGF0ZU1hY2hpbmVUeXBlOiAnU1RBTkRBUkQnLFxuICAgICAgRGVmaW5pdGlvblN0cmluZzogJ3tcIlN0YXJ0QXRcIjpcIlBhc3NcIixcIlN0YXRlc1wiOntcIlBhc3NcIjp7XCJUeXBlXCI6XCJQYXNzXCIsXCJFbmRcIjp0cnVlfX19JyxcbiAgICB9KTtcblxuICB9KSxcblxuICB0ZXN0KCdJbnN0YW50aWF0ZSBFeHByZXNzIFN0YXRlIE1hY2hpbmUnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgc3RlcGZ1bmN0aW9ucy5TdGF0ZU1hY2hpbmUoc3RhY2ssICdNeVN0YXRlTWFjaGluZScsIHtcbiAgICAgIHN0YXRlTWFjaGluZU5hbWU6ICdNeVN0YXRlTWFjaGluZScsXG4gICAgICBkZWZpbml0aW9uOiBzdGVwZnVuY3Rpb25zLkNoYWluLnN0YXJ0KG5ldyBzdGVwZnVuY3Rpb25zLlBhc3Moc3RhY2ssICdQYXNzJykpLFxuICAgICAgc3RhdGVNYWNoaW5lVHlwZTogc3RlcGZ1bmN0aW9ucy5TdGF0ZU1hY2hpbmVUeXBlLkVYUFJFU1MsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpTdGVwRnVuY3Rpb25zOjpTdGF0ZU1hY2hpbmUnLCB7XG4gICAgICBTdGF0ZU1hY2hpbmVOYW1lOiAnTXlTdGF0ZU1hY2hpbmUnLFxuICAgICAgU3RhdGVNYWNoaW5lVHlwZTogJ0VYUFJFU1MnLFxuICAgICAgRGVmaW5pdGlvblN0cmluZzogJ3tcIlN0YXJ0QXRcIjpcIlBhc3NcIixcIlN0YXRlc1wiOntcIlBhc3NcIjp7XCJUeXBlXCI6XCJQYXNzXCIsXCJFbmRcIjp0cnVlfX19JyxcbiAgICB9KTtcblxuICB9KSxcblxuICB0ZXN0KCdsb2cgY29uZmlndXJhdGlvbicsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuXG4gICAgLy8gV0hFTlxuICAgIGNvbnN0IGxvZ0dyb3VwID0gbmV3IGxvZ3MuTG9nR3JvdXAoc3RhY2ssICdNeUxvZ0dyb3VwJyk7XG5cbiAgICBuZXcgc3RlcGZ1bmN0aW9ucy5TdGF0ZU1hY2hpbmUoc3RhY2ssICdNeVN0YXRlTWFjaGluZScsIHtcbiAgICAgIGRlZmluaXRpb246IHN0ZXBmdW5jdGlvbnMuQ2hhaW4uc3RhcnQobmV3IHN0ZXBmdW5jdGlvbnMuUGFzcyhzdGFjaywgJ1Bhc3MnKSksXG4gICAgICBsb2dzOiB7XG4gICAgICAgIGRlc3RpbmF0aW9uOiBsb2dHcm91cCxcbiAgICAgICAgbGV2ZWw6IHN0ZXBmdW5jdGlvbnMuTG9nTGV2ZWwuRkFUQUwsXG4gICAgICAgIGluY2x1ZGVFeGVjdXRpb25EYXRhOiBmYWxzZSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpTdGVwRnVuY3Rpb25zOjpTdGF0ZU1hY2hpbmUnLCB7XG4gICAgICBEZWZpbml0aW9uU3RyaW5nOiAne1wiU3RhcnRBdFwiOlwiUGFzc1wiLFwiU3RhdGVzXCI6e1wiUGFzc1wiOntcIlR5cGVcIjpcIlBhc3NcIixcIkVuZFwiOnRydWV9fX0nLFxuICAgICAgTG9nZ2luZ0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgRGVzdGluYXRpb25zOiBbe1xuICAgICAgICAgIENsb3VkV2F0Y2hMb2dzTG9nR3JvdXA6IHtcbiAgICAgICAgICAgIExvZ0dyb3VwQXJuOiB7XG4gICAgICAgICAgICAgICdGbjo6R2V0QXR0JzogWydNeUxvZ0dyb3VwNUMwREFEODUnLCAnQXJuJ10sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH1dLFxuICAgICAgICBJbmNsdWRlRXhlY3V0aW9uRGF0YTogZmFsc2UsXG4gICAgICAgIExldmVsOiAnRkFUQUwnLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBTdGF0ZW1lbnQ6IFt7XG4gICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAnbG9nczpDcmVhdGVMb2dEZWxpdmVyeScsXG4gICAgICAgICAgICAnbG9nczpHZXRMb2dEZWxpdmVyeScsXG4gICAgICAgICAgICAnbG9nczpVcGRhdGVMb2dEZWxpdmVyeScsXG4gICAgICAgICAgICAnbG9nczpEZWxldGVMb2dEZWxpdmVyeScsXG4gICAgICAgICAgICAnbG9nczpMaXN0TG9nRGVsaXZlcmllcycsXG4gICAgICAgICAgICAnbG9nczpQdXRSZXNvdXJjZVBvbGljeScsXG4gICAgICAgICAgICAnbG9nczpEZXNjcmliZVJlc291cmNlUG9saWNpZXMnLFxuICAgICAgICAgICAgJ2xvZ3M6RGVzY3JpYmVMb2dHcm91cHMnLFxuICAgICAgICAgIF0sXG4gICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgIFJlc291cmNlOiAnKicsXG4gICAgICAgIH1dLFxuICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICB9LFxuICAgICAgUG9saWN5TmFtZTogJ015U3RhdGVNYWNoaW5lUm9sZURlZmF1bHRQb2xpY3lFNDY4RUIxOCcsXG4gICAgICBSb2xlczogW1xuICAgICAgICB7XG4gICAgICAgICAgUmVmOiAnTXlTdGF0ZU1hY2hpbmVSb2xlRDU5RkZFQkMnLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgndHJhY2luZyBjb25maWd1cmF0aW9uJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IHN0ZXBmdW5jdGlvbnMuU3RhdGVNYWNoaW5lKHN0YWNrLCAnTXlTdGF0ZU1hY2hpbmUnLCB7XG4gICAgICBkZWZpbml0aW9uOiBzdGVwZnVuY3Rpb25zLkNoYWluLnN0YXJ0KG5ldyBzdGVwZnVuY3Rpb25zLlBhc3Moc3RhY2ssICdQYXNzJykpLFxuICAgICAgdHJhY2luZ0VuYWJsZWQ6IHRydWUsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpTdGVwRnVuY3Rpb25zOjpTdGF0ZU1hY2hpbmUnLCB7XG4gICAgICBEZWZpbml0aW9uU3RyaW5nOiAne1wiU3RhcnRBdFwiOlwiUGFzc1wiLFwiU3RhdGVzXCI6e1wiUGFzc1wiOntcIlR5cGVcIjpcIlBhc3NcIixcIkVuZFwiOnRydWV9fX0nLFxuICAgICAgVHJhY2luZ0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgRW5hYmxlZDogdHJ1ZSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jywge1xuICAgICAgUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbe1xuICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgJ3hyYXk6UHV0VHJhY2VTZWdtZW50cycsXG4gICAgICAgICAgICAneHJheTpQdXRUZWxlbWV0cnlSZWNvcmRzJyxcbiAgICAgICAgICBdLFxuICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICBSZXNvdXJjZTogJyonLFxuICAgICAgICB9XSxcbiAgICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnLFxuICAgICAgfSxcbiAgICAgIFBvbGljeU5hbWU6ICdNeVN0YXRlTWFjaGluZVJvbGVEZWZhdWx0UG9saWN5RTQ2OEVCMTgnLFxuICAgICAgUm9sZXM6IFtcbiAgICAgICAge1xuICAgICAgICAgIFJlZjogJ015U3RhdGVNYWNoaW5lUm9sZUQ1OUZGRUJDJyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSk7XG4gIH0pO1xufSk7XG4iXX0=