"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const stepfunctions = require("../lib");
describe('Condition Variables', () => {
    test('Condition variables must start with $. or $[', () => {
        expect(() => stepfunctions.Condition.stringEquals('a', 'b')).toThrow();
    }),
        test('Condition variables can start with $.', () => {
            expect(() => stepfunctions.Condition.stringEquals('$.a', 'b')).not.toThrow();
        }),
        test('Condition variables can start with $[', () => {
            expect(() => stepfunctions.Condition.stringEquals('$[0]', 'a')).not.toThrow();
        }),
        test('Condition variables can reference the state input $', () => {
            expect(() => stepfunctions.Condition.stringEquals('$', 'a')).not.toThrow();
        }),
        test('NotConditon must render properly', () => {
            assertRendersTo(stepfunctions.Condition.not(stepfunctions.Condition.stringEquals('$.a', 'b')), { Not: { Variable: '$.a', StringEquals: 'b' } });
        }),
        test('CompoundCondition must render properly', () => {
            assertRendersTo(stepfunctions.Condition.and(stepfunctions.Condition.booleanEquals('$.a', true), stepfunctions.Condition.numberGreaterThan('$.b', 3)), {
                And: [
                    { Variable: '$.a', BooleanEquals: true },
                    { Variable: '$.b', NumericGreaterThan: 3 },
                ],
            });
        }),
        test('Exercise a number of other conditions', () => {
            const cases = [
                [stepfunctions.Condition.stringLessThan('$.a', 'foo'), { Variable: '$.a', StringLessThan: 'foo' }],
                [stepfunctions.Condition.stringLessThanEquals('$.a', 'foo'), { Variable: '$.a', StringLessThanEquals: 'foo' }],
                [stepfunctions.Condition.stringGreaterThan('$.a', 'foo'), { Variable: '$.a', StringGreaterThan: 'foo' }],
                [stepfunctions.Condition.stringGreaterThanEquals('$.a', 'foo'), { Variable: '$.a', StringGreaterThanEquals: 'foo' }],
                [stepfunctions.Condition.numberEquals('$.a', 5), { Variable: '$.a', NumericEquals: 5 }],
            ];
            for (const [cond, expected] of cases) {
                assertRendersTo(cond, expected);
            }
        }),
        test('Exercise string conditions', () => {
            const cases = [
                [stepfunctions.Condition.stringEquals('$.a', 'foo'), { Variable: '$.a', StringEquals: 'foo' }],
                [stepfunctions.Condition.stringEqualsJsonPath('$.a', '$.b'), { Variable: '$.a', StringEqualsPath: '$.b' }],
                [stepfunctions.Condition.stringLessThan('$.a', 'foo'), { Variable: '$.a', StringLessThan: 'foo' }],
                [stepfunctions.Condition.stringLessThanJsonPath('$.a', '$.b'), { Variable: '$.a', StringLessThanPath: '$.b' }],
                [stepfunctions.Condition.stringLessThanEquals('$.a', 'foo'), { Variable: '$.a', StringLessThanEquals: 'foo' }],
                [stepfunctions.Condition.stringLessThanEqualsJsonPath('$.a', '$.b'), { Variable: '$.a', StringLessThanEqualsPath: '$.b' }],
                [stepfunctions.Condition.stringGreaterThan('$.a', 'foo'), { Variable: '$.a', StringGreaterThan: 'foo' }],
                [stepfunctions.Condition.stringGreaterThanJsonPath('$.a', '$.b'), { Variable: '$.a', StringGreaterThanPath: '$.b' }],
                [stepfunctions.Condition.stringGreaterThanEquals('$.a', 'foo'), { Variable: '$.a', StringGreaterThanEquals: 'foo' }],
                [stepfunctions.Condition.stringGreaterThanEqualsJsonPath('$.a', '$.b'), { Variable: '$.a', StringGreaterThanEqualsPath: '$.b' }],
            ];
            for (const [cond, expected] of cases) {
                assertRendersTo(cond, expected);
            }
        }),
        test('Exercise number conditions', () => {
            const cases = [
                [stepfunctions.Condition.numberEquals('$.a', 5), { Variable: '$.a', NumericEquals: 5 }],
                [stepfunctions.Condition.numberEqualsJsonPath('$.a', '$.b'), { Variable: '$.a', NumericEqualsPath: '$.b' }],
                [stepfunctions.Condition.numberLessThan('$.a', 5), { Variable: '$.a', NumericLessThan: 5 }],
                [stepfunctions.Condition.numberLessThanJsonPath('$.a', '$.b'), { Variable: '$.a', NumericLessThanPath: '$.b' }],
                [stepfunctions.Condition.numberGreaterThan('$.a', 5), { Variable: '$.a', NumericGreaterThan: 5 }],
                [stepfunctions.Condition.numberGreaterThanJsonPath('$.a', '$.b'), { Variable: '$.a', NumericGreaterThanPath: '$.b' }],
                [stepfunctions.Condition.numberLessThanEquals('$.a', 5), { Variable: '$.a', NumericLessThanEquals: 5 }],
                [stepfunctions.Condition.numberLessThanEqualsJsonPath('$.a', '$.b'), { Variable: '$.a', NumericLessThanEqualsPath: '$.b' }],
                [stepfunctions.Condition.numberGreaterThanEquals('$.a', 5), { Variable: '$.a', NumericGreaterThanEquals: 5 }],
                [stepfunctions.Condition.numberGreaterThanEqualsJsonPath('$.a', '$.b'), { Variable: '$.a', NumericGreaterThanEqualsPath: '$.b' }],
            ];
            for (const [cond, expected] of cases) {
                assertRendersTo(cond, expected);
            }
        }),
        test('Exercise type conditions', () => {
            const cases = [
                [stepfunctions.Condition.isString('$.a'), { Variable: '$.a', IsString: true }],
                [stepfunctions.Condition.isNotString('$.a'), { Variable: '$.a', IsString: false }],
                [stepfunctions.Condition.isNumeric('$.a'), { Variable: '$.a', IsNumeric: true }],
                [stepfunctions.Condition.isNotNumeric('$.a'), { Variable: '$.a', IsNumeric: false }],
                [stepfunctions.Condition.isBoolean('$.a'), { Variable: '$.a', IsBoolean: true }],
                [stepfunctions.Condition.isNotBoolean('$.a'), { Variable: '$.a', IsBoolean: false }],
                [stepfunctions.Condition.isTimestamp('$.a'), { Variable: '$.a', IsTimestamp: true }],
                [stepfunctions.Condition.isNotTimestamp('$.a'), { Variable: '$.a', IsTimestamp: false }],
            ];
            for (const [cond, expected] of cases) {
                assertRendersTo(cond, expected);
            }
        }),
        test('Exercise timestamp conditions', () => {
            const cases = [
                [stepfunctions.Condition.timestampEquals('$.a', 'timestamp'), { Variable: '$.a', TimestampEquals: 'timestamp' }],
                [stepfunctions.Condition.timestampEqualsJsonPath('$.a', '$.b'), { Variable: '$.a', TimestampEqualsPath: '$.b' }],
                [stepfunctions.Condition.timestampLessThan('$.a', 'timestamp'), { Variable: '$.a', TimestampLessThan: 'timestamp' }],
                [stepfunctions.Condition.timestampLessThanJsonPath('$.a', '$.b'), { Variable: '$.a', TimestampLessThanPath: '$.b' }],
                [stepfunctions.Condition.timestampGreaterThan('$.a', 'timestamp'), { Variable: '$.a', TimestampGreaterThan: 'timestamp' }],
                [stepfunctions.Condition.timestampGreaterThanJsonPath('$.a', '$.b'), { Variable: '$.a', TimestampGreaterThanPath: '$.b' }],
                [stepfunctions.Condition.timestampLessThanEquals('$.a', 'timestamp'), { Variable: '$.a', TimestampLessThanEquals: 'timestamp' }],
                [stepfunctions.Condition.timestampLessThanEqualsJsonPath('$.a', '$.b'), { Variable: '$.a', TimestampLessThanEqualsPath: '$.b' }],
                [stepfunctions.Condition.timestampGreaterThanEquals('$.a', 'timestamp'), { Variable: '$.a', TimestampGreaterThanEquals: 'timestamp' }],
                [stepfunctions.Condition.timestampGreaterThanEqualsJsonPath('$.a', '$.b'), { Variable: '$.a', TimestampGreaterThanEqualsPath: '$.b' }],
            ];
            for (const [cond, expected] of cases) {
                assertRendersTo(cond, expected);
            }
        }),
        test('Exercise other conditions', () => {
            const cases = [
                [stepfunctions.Condition.booleanEqualsJsonPath('$.a', '$.b'), { Variable: '$.a', BooleanEqualsPath: '$.b' }],
                [stepfunctions.Condition.booleanEquals('$.a', true), { Variable: '$.a', BooleanEquals: true }],
                [stepfunctions.Condition.isPresent('$.a'), { Variable: '$.a', IsPresent: true }],
                [stepfunctions.Condition.isNotPresent('$.a'), { Variable: '$.a', IsPresent: false }],
                [stepfunctions.Condition.stringMatches('$.a', 'foo'), { Variable: '$.a', StringMatches: 'foo' }],
            ];
            for (const [cond, expected] of cases) {
                assertRendersTo(cond, expected);
            }
        });
});
function assertRendersTo(cond, expected) {
    expect(cond.renderCondition()).toStrictEqual(expected);
}
//# sourceMappingURL=data:application/json;base64,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