"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Wait = exports.WaitTime = void 0;
const chain_1 = require("../chain");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Represents the Wait state which delays a state machine from continuing for a specified time
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-wait-state.html
 */
class WaitTime {
    constructor(json) {
        this.json = json;
    }
    /**
     * Wait a fixed amount of time.
     */
    static duration(duration) { return new WaitTime({ Seconds: duration.toSeconds() }); }
    /**
     * Wait until the given ISO8601 timestamp
     *
     * @example 2016-03-14T01:59:00Z
     */
    static timestamp(timestamp) { return new WaitTime({ Timestamp: timestamp }); }
    /**
     * Wait for a number of seconds stored in the state object.
     *
     * @example $.waitSeconds
     */
    static secondsPath(path) { return new WaitTime({ SecondsPath: path }); }
    /**
     * Wait until a timestamp found in the state object.
     *
     * @example $.waitTimestamp
     */
    static timestampPath(path) { return new WaitTime({ TimestampPath: path }); }
    /**
     * @internal
     */
    get _json() {
        return this.json;
    }
}
exports.WaitTime = WaitTime;
/**
 * Define a Wait state in the state machine
 *
 * A Wait state can be used to delay execution of the state machine for a while.
 */
class Wait extends state_1.State {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.time = props.time;
        this.endStates = [this];
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.WAIT,
            Comment: this.comment,
            ...this.time._json,
            ...this.renderNextEnd(),
        };
    }
}
exports.Wait = Wait;
//# sourceMappingURL=data:application/json;base64,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