"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Parallel = void 0;
const chain_1 = require("../chain");
const state_graph_1 = require("../state-graph");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Define a Parallel state in the state machine
 *
 * A Parallel state can be used to run one or more state machines at the same
 * time.
 *
 * The Result of a Parallel state is an array of the results of its substatemachines.
 */
class Parallel extends state_1.State {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.endStates = [this];
    }
    /**
     * Add retry configuration for this state
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * Add a recovery handler for this state
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Define one or more branches to run in parallel
     */
    branch(...branches) {
        for (const branch of branches) {
            const name = `Parallel '${this.stateId}' branch ${this.branches.length + 1}`;
            super.addBranch(new state_graph_1.StateGraph(branch.startState, name));
        }
        return this;
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.PARALLEL,
            Comment: this.comment,
            ResultPath: state_1.renderJsonPath(this.resultPath),
            ...this.renderNextEnd(),
            ...this.renderInputOutput(),
            ...this.renderRetryCatch(),
            ...this.renderBranches(),
        };
    }
    /**
     * Validate this state
     */
    validate() {
        if (this.branches.length === 0) {
            return ['Parallel must have at least one branch'];
        }
        return [];
    }
}
exports.Parallel = Parallel;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGFyYWxsZWwuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJwYXJhbGxlbC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFDQSxvQ0FBaUM7QUFDakMsZ0RBQTRDO0FBRTVDLHFEQUFpRDtBQUNqRCxtQ0FBZ0Q7QUE0Q2hEOzs7Ozs7O0dBT0c7QUFDSCxNQUFhLFFBQVMsU0FBUSxhQUFLO0lBR2pDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBdUIsRUFBRTtRQUNqRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUV4QixJQUFJLENBQUMsU0FBUyxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDMUIsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksUUFBUSxDQUFDLFFBQW9CLEVBQUU7UUFDcEMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUN2QixPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLFFBQVEsQ0FBQyxPQUFtQixFQUFFLFFBQW9CLEVBQUU7UUFDekQsS0FBSyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQzNDLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOztPQUVHO0lBQ0ksSUFBSSxDQUFDLElBQWdCO1FBQzFCLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2hDLE9BQU8sYUFBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUVEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEdBQUcsUUFBc0I7UUFDckMsS0FBSyxNQUFNLE1BQU0sSUFBSSxRQUFRLEVBQUU7WUFDN0IsTUFBTSxJQUFJLEdBQUcsYUFBYSxJQUFJLENBQUMsT0FBTyxZQUFZLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRSxDQUFDO1lBQzdFLEtBQUssQ0FBQyxTQUFTLENBQUMsSUFBSSx3QkFBVSxDQUFDLE1BQU0sQ0FBQyxVQUFVLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQztTQUMxRDtRQUNELE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOztPQUVHO0lBQ0ksV0FBVztRQUNoQixPQUFPO1lBQ0wsSUFBSSxFQUFFLHNCQUFTLENBQUMsUUFBUTtZQUN4QixPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87WUFDckIsVUFBVSxFQUFFLHNCQUFjLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQztZQUMzQyxHQUFHLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFDdkIsR0FBRyxJQUFJLENBQUMsaUJBQWlCLEVBQUU7WUFDM0IsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLEVBQUU7WUFDMUIsR0FBRyxJQUFJLENBQUMsY0FBYyxFQUFFO1NBQ3pCLENBQUM7SUFDSixDQUFDO0lBRUQ7O09BRUc7SUFDTyxRQUFRO1FBQ2hCLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzlCLE9BQU8sQ0FBQyx3Q0FBd0MsQ0FBQyxDQUFDO1NBQ25EO1FBQ0QsT0FBTyxFQUFFLENBQUM7SUFDWixDQUFDO0NBQ0Y7QUExRUQsNEJBMEVDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDaGFpbiB9IGZyb20gJy4uL2NoYWluJztcbmltcG9ydCB7IFN0YXRlR3JhcGggfSBmcm9tICcuLi9zdGF0ZS1ncmFwaCc7XG5pbXBvcnQgeyBDYXRjaFByb3BzLCBJQ2hhaW5hYmxlLCBJTmV4dGFibGUsIFJldHJ5UHJvcHMgfSBmcm9tICcuLi90eXBlcyc7XG5pbXBvcnQgeyBTdGF0ZVR5cGUgfSBmcm9tICcuL3ByaXZhdGUvc3RhdGUtdHlwZSc7XG5pbXBvcnQgeyByZW5kZXJKc29uUGF0aCwgU3RhdGUgfSBmcm9tICcuL3N0YXRlJztcblxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBkZWZpbmluZyBhIFBhcmFsbGVsIHN0YXRlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUGFyYWxsZWxQcm9wcyB7XG4gIC8qKlxuICAgKiBBbiBvcHRpb25hbCBkZXNjcmlwdGlvbiBmb3IgdGhpcyBzdGF0ZVxuICAgKlxuICAgKiBAZGVmYXVsdCBObyBjb21tZW50XG4gICAqL1xuICByZWFkb25seSBjb21tZW50Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBKU09OUGF0aCBleHByZXNzaW9uIHRvIHNlbGVjdCBwYXJ0IG9mIHRoZSBzdGF0ZSB0byBiZSB0aGUgaW5wdXQgdG8gdGhpcyBzdGF0ZS5cbiAgICpcbiAgICogTWF5IGFsc28gYmUgdGhlIHNwZWNpYWwgdmFsdWUgSnNvblBhdGguRElTQ0FSRCwgd2hpY2ggd2lsbCBjYXVzZSB0aGUgZWZmZWN0aXZlXG4gICAqIGlucHV0IHRvIGJlIHRoZSBlbXB0eSBvYmplY3Qge30uXG4gICAqXG4gICAqIEBkZWZhdWx0ICRcbiAgICovXG4gIHJlYWRvbmx5IGlucHV0UGF0aD86IHN0cmluZztcblxuICAvKipcbiAgICogSlNPTlBhdGggZXhwcmVzc2lvbiB0byBzZWxlY3QgcGFydCBvZiB0aGUgc3RhdGUgdG8gYmUgdGhlIG91dHB1dCB0byB0aGlzIHN0YXRlLlxuICAgKlxuICAgKiBNYXkgYWxzbyBiZSB0aGUgc3BlY2lhbCB2YWx1ZSBKc29uUGF0aC5ESVNDQVJELCB3aGljaCB3aWxsIGNhdXNlIHRoZSBlZmZlY3RpdmVcbiAgICogb3V0cHV0IHRvIGJlIHRoZSBlbXB0eSBvYmplY3Qge30uXG4gICAqXG4gICAqIEBkZWZhdWx0ICRcbiAgICovXG4gIHJlYWRvbmx5IG91dHB1dFBhdGg/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEpTT05QYXRoIGV4cHJlc3Npb24gdG8gaW5kaWNhdGUgd2hlcmUgdG8gaW5qZWN0IHRoZSBzdGF0ZSdzIG91dHB1dFxuICAgKlxuICAgKiBNYXkgYWxzbyBiZSB0aGUgc3BlY2lhbCB2YWx1ZSBKc29uUGF0aC5ESVNDQVJELCB3aGljaCB3aWxsIGNhdXNlIHRoZSBzdGF0ZSdzXG4gICAqIGlucHV0IHRvIGJlY29tZSBpdHMgb3V0cHV0LlxuICAgKlxuICAgKiBAZGVmYXVsdCAkXG4gICAqL1xuICByZWFkb25seSByZXN1bHRQYXRoPzogc3RyaW5nO1xufVxuXG4vKipcbiAqIERlZmluZSBhIFBhcmFsbGVsIHN0YXRlIGluIHRoZSBzdGF0ZSBtYWNoaW5lXG4gKlxuICogQSBQYXJhbGxlbCBzdGF0ZSBjYW4gYmUgdXNlZCB0byBydW4gb25lIG9yIG1vcmUgc3RhdGUgbWFjaGluZXMgYXQgdGhlIHNhbWVcbiAqIHRpbWUuXG4gKlxuICogVGhlIFJlc3VsdCBvZiBhIFBhcmFsbGVsIHN0YXRlIGlzIGFuIGFycmF5IG9mIHRoZSByZXN1bHRzIG9mIGl0cyBzdWJzdGF0ZW1hY2hpbmVzLlxuICovXG5leHBvcnQgY2xhc3MgUGFyYWxsZWwgZXh0ZW5kcyBTdGF0ZSBpbXBsZW1lbnRzIElOZXh0YWJsZSB7XG4gIHB1YmxpYyByZWFkb25seSBlbmRTdGF0ZXM6IElOZXh0YWJsZVtdO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBQYXJhbGxlbFByb3BzID0ge30pIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcblxuICAgIHRoaXMuZW5kU3RhdGVzID0gW3RoaXNdO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCByZXRyeSBjb25maWd1cmF0aW9uIGZvciB0aGlzIHN0YXRlXG4gICAqXG4gICAqIFRoaXMgY29udHJvbHMgaWYgYW5kIGhvdyB0aGUgZXhlY3V0aW9uIHdpbGwgYmUgcmV0cmllZCBpZiBhIHBhcnRpY3VsYXJcbiAgICogZXJyb3Igb2NjdXJzLlxuICAgKi9cbiAgcHVibGljIGFkZFJldHJ5KHByb3BzOiBSZXRyeVByb3BzID0ge30pOiBQYXJhbGxlbCB7XG4gICAgc3VwZXIuX2FkZFJldHJ5KHByb3BzKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSByZWNvdmVyeSBoYW5kbGVyIGZvciB0aGlzIHN0YXRlXG4gICAqXG4gICAqIFdoZW4gYSBwYXJ0aWN1bGFyIGVycm9yIG9jY3VycywgZXhlY3V0aW9uIHdpbGwgY29udGludWUgYXQgdGhlIGVycm9yXG4gICAqIGhhbmRsZXIgaW5zdGVhZCBvZiBmYWlsaW5nIHRoZSBzdGF0ZSBtYWNoaW5lIGV4ZWN1dGlvbi5cbiAgICovXG4gIHB1YmxpYyBhZGRDYXRjaChoYW5kbGVyOiBJQ2hhaW5hYmxlLCBwcm9wczogQ2F0Y2hQcm9wcyA9IHt9KTogUGFyYWxsZWwge1xuICAgIHN1cGVyLl9hZGRDYXRjaChoYW5kbGVyLnN0YXJ0U3RhdGUsIHByb3BzKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDb250aW51ZSBub3JtYWwgZXhlY3V0aW9uIHdpdGggdGhlIGdpdmVuIHN0YXRlXG4gICAqL1xuICBwdWJsaWMgbmV4dChuZXh0OiBJQ2hhaW5hYmxlKTogQ2hhaW4ge1xuICAgIHN1cGVyLm1ha2VOZXh0KG5leHQuc3RhcnRTdGF0ZSk7XG4gICAgcmV0dXJuIENoYWluLnNlcXVlbmNlKHRoaXMsIG5leHQpO1xuICB9XG5cbiAgLyoqXG4gICAqIERlZmluZSBvbmUgb3IgbW9yZSBicmFuY2hlcyB0byBydW4gaW4gcGFyYWxsZWxcbiAgICovXG4gIHB1YmxpYyBicmFuY2goLi4uYnJhbmNoZXM6IElDaGFpbmFibGVbXSk6IFBhcmFsbGVsIHtcbiAgICBmb3IgKGNvbnN0IGJyYW5jaCBvZiBicmFuY2hlcykge1xuICAgICAgY29uc3QgbmFtZSA9IGBQYXJhbGxlbCAnJHt0aGlzLnN0YXRlSWR9JyBicmFuY2ggJHt0aGlzLmJyYW5jaGVzLmxlbmd0aCArIDF9YDtcbiAgICAgIHN1cGVyLmFkZEJyYW5jaChuZXcgU3RhdGVHcmFwaChicmFuY2guc3RhcnRTdGF0ZSwgbmFtZSkpO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm4gdGhlIEFtYXpvbiBTdGF0ZXMgTGFuZ3VhZ2Ugb2JqZWN0IGZvciB0aGlzIHN0YXRlXG4gICAqL1xuICBwdWJsaWMgdG9TdGF0ZUpzb24oKTogb2JqZWN0IHtcbiAgICByZXR1cm4ge1xuICAgICAgVHlwZTogU3RhdGVUeXBlLlBBUkFMTEVMLFxuICAgICAgQ29tbWVudDogdGhpcy5jb21tZW50LFxuICAgICAgUmVzdWx0UGF0aDogcmVuZGVySnNvblBhdGgodGhpcy5yZXN1bHRQYXRoKSxcbiAgICAgIC4uLnRoaXMucmVuZGVyTmV4dEVuZCgpLFxuICAgICAgLi4udGhpcy5yZW5kZXJJbnB1dE91dHB1dCgpLFxuICAgICAgLi4udGhpcy5yZW5kZXJSZXRyeUNhdGNoKCksXG4gICAgICAuLi50aGlzLnJlbmRlckJyYW5jaGVzKCksXG4gICAgfTtcbiAgfVxuXG4gIC8qKlxuICAgKiBWYWxpZGF0ZSB0aGlzIHN0YXRlXG4gICAqL1xuICBwcm90ZWN0ZWQgdmFsaWRhdGUoKTogc3RyaW5nW10ge1xuICAgIGlmICh0aGlzLmJyYW5jaGVzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgcmV0dXJuIFsnUGFyYWxsZWwgbXVzdCBoYXZlIGF0IGxlYXN0IG9uZSBicmFuY2gnXTtcbiAgICB9XG4gICAgcmV0dXJuIFtdO1xuICB9XG59XG4iXX0=