"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Map = exports.isPositiveInteger = void 0;
const chain_1 = require("../chain");
const fields_1 = require("../fields");
const state_graph_1 = require("../state-graph");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Returns true if the value passed is a positive integer
 * @param value the value ti validate
 */
exports.isPositiveInteger = (value) => {
    const isFloat = Math.floor(value) !== value;
    const isNotPositiveInteger = value < 0 || value > Number.MAX_SAFE_INTEGER;
    return !isFloat && !isNotPositiveInteger;
};
/**
 * Define a Map state in the state machine
 *
 * A `Map` state can be used to run a set of steps for each element of an input array.
 * A Map state will execute the same steps for multiple entries of an array in the state input.
 *
 * While the Parallel state executes multiple branches of steps using the same input, a Map state
 * will execute the same steps for multiple entries of an array in the state input.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/amazon-states-language-map-state.html
 */
class Map extends state_1.State {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.endStates = [this];
        this.maxConcurrency = props.maxConcurrency;
        this.itemsPath = props.itemsPath;
    }
    /**
     * Add retry configuration for this state
     *
     * This controls if and how the execution will be retried if a particular
     * error occurs.
     */
    addRetry(props = {}) {
        super._addRetry(props);
        return this;
    }
    /**
     * Add a recovery handler for this state
     *
     * When a particular error occurs, execution will continue at the error
     * handler instead of failing the state machine execution.
     */
    addCatch(handler, props = {}) {
        super._addCatch(handler.startState, props);
        return this;
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        super.makeNext(next.startState);
        return chain_1.Chain.sequence(this, next);
    }
    /**
     * Define iterator state machine in Map
     */
    iterator(iterator) {
        const name = `Map ${this.stateId} Iterator`;
        super.addIterator(new state_graph_1.StateGraph(iterator.startState, name));
        return this;
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.MAP,
            Comment: this.comment,
            ResultPath: state_1.renderJsonPath(this.resultPath),
            ...this.renderNextEnd(),
            ...this.renderInputOutput(),
            ...this.renderParameters(),
            ...this.renderRetryCatch(),
            ...this.renderIterator(),
            ...this.renderItemsPath(),
            MaxConcurrency: this.maxConcurrency,
        };
    }
    /**
     * Validate this state
     */
    validate() {
        const errors = [];
        if (this.iteration === undefined) {
            errors.push('Map state must have a non-empty iterator');
        }
        if (this.maxConcurrency !== undefined && !exports.isPositiveInteger(this.maxConcurrency)) {
            errors.push('maxConcurrency has to be a positive integer');
        }
        return errors;
    }
    renderItemsPath() {
        return {
            ItemsPath: state_1.renderJsonPath(this.itemsPath),
        };
    }
    /**
     * Render Parameters in ASL JSON format
     */
    renderParameters() {
        return fields_1.FieldUtils.renderObject({
            Parameters: this.parameters,
        });
    }
}
exports.Map = Map;
//# sourceMappingURL=data:application/json;base64,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