"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InputType = exports.TaskInput = void 0;
const fields_1 = require("./fields");
/**
 * Type union for task classes that accept multiple types of payload
 */
class TaskInput {
    /**
     *
     * @param type type of task input
     * @param value payload for the corresponding input type.
     * It can be a JSON-encoded object, context, data, etc.
     */
    constructor(type, value) {
        this.type = type;
        this.value = value;
    }
    /**
     * Use a literal string as task input
     *
     * This might be a JSON-encoded object, or just a text.
     */
    static fromText(text) {
        return new TaskInput(InputType.TEXT, text);
    }
    /**
     * Use an object as task input
     *
     * This object may contain Data and Context fields
     * as object values, if desired.
     */
    static fromObject(obj) {
        return new TaskInput(InputType.OBJECT, obj);
    }
    /**
     * Use a part of the execution data or task context as task input
     *
     * Use this when you want to use a subobject or string from
     * the current state machine execution or the current task context
     * as complete payload to a task.
     */
    static fromJsonPathAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.JsonPath.stringAt(path));
    }
    /**
     * Use a part of the execution data as task input
     *
     * Use this when you want to use a subobject or string from
     * the current state machine execution as complete payload
     * to a task.
     */
    static fromDataAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.JsonPath.stringAt(path));
    }
    /**
     * Use a part of the task context as task input
     *
     * Use this when you want to use a subobject or string from
     * the current task context as complete payload
     * to a task.
     */
    static fromContextAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.JsonPath.stringAt(path));
    }
}
exports.TaskInput = TaskInput;
/**
 * The type of task input
 */
var InputType;
(function (InputType) {
    /**
     * Use a literal string
     * This might be a JSON-encoded object, or just text.
     * valid JSON text: standalone, quote-delimited strings; objects; arrays; numbers; Boolean values; and null.
     *
     * example: `literal string`
     * example: {"json": "encoded"}
     */
    InputType[InputType["TEXT"] = 0] = "TEXT";
    /**
     * Use an object which may contain Data and Context fields
     * as object values, if desired.
     *
     * example:
     * {
     *  literal: 'literal',
     *  SomeInput: sfn.JsonPath.stringAt('$.someField')
     * }
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-data.html
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/input-output-contextobject.html
     */
    InputType[InputType["OBJECT"] = 1] = "OBJECT";
})(InputType = exports.InputType || (exports.InputType = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5wdXQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJpbnB1dC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxxQ0FBb0M7QUFFcEM7O0dBRUc7QUFDSCxNQUFhLFNBQVM7SUFxRHBCOzs7OztPQUtHO0lBQ0gsWUFBb0MsSUFBZSxFQUFrQixLQUFVO1FBQTNDLFNBQUksR0FBSixJQUFJLENBQVc7UUFBa0IsVUFBSyxHQUFMLEtBQUssQ0FBSztJQUFHLENBQUM7SUExRG5GOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQVk7UUFDakMsT0FBTyxJQUFJLFNBQVMsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDO0lBQzdDLENBQUM7SUFFRDs7Ozs7T0FLRztJQUNJLE1BQU0sQ0FBQyxVQUFVLENBQUMsR0FBMkI7UUFDbEQsT0FBTyxJQUFJLFNBQVMsQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQzlDLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFDSSxNQUFNLENBQUMsY0FBYyxDQUFDLElBQVk7UUFDdkMsT0FBTyxJQUFJLFNBQVMsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLGlCQUFRLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7SUFDaEUsQ0FBQztJQUVEOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyxVQUFVLENBQUMsSUFBWTtRQUNuQyxPQUFPLElBQUksU0FBUyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsaUJBQVEsQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLGFBQWEsQ0FBQyxJQUFZO1FBQ3RDLE9BQU8sSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxpQkFBUSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO0lBQ2hFLENBQUM7Q0FTRjtBQTVERCw4QkE0REM7QUFFRDs7R0FFRztBQUNILElBQVksU0F3Qlg7QUF4QkQsV0FBWSxTQUFTO0lBQ25COzs7Ozs7O09BT0c7SUFDSCx5Q0FBSSxDQUFBO0lBQ0o7Ozs7Ozs7Ozs7OztPQVlHO0lBQ0gsNkNBQU0sQ0FBQTtBQUNSLENBQUMsRUF4QlcsU0FBUyxHQUFULGlCQUFTLEtBQVQsaUJBQVMsUUF3QnBCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgSnNvblBhdGggfSBmcm9tICcuL2ZpZWxkcyc7XG5cbi8qKlxuICogVHlwZSB1bmlvbiBmb3IgdGFzayBjbGFzc2VzIHRoYXQgYWNjZXB0IG11bHRpcGxlIHR5cGVzIG9mIHBheWxvYWRcbiAqL1xuZXhwb3J0IGNsYXNzIFRhc2tJbnB1dCB7XG4gIC8qKlxuICAgKiBVc2UgYSBsaXRlcmFsIHN0cmluZyBhcyB0YXNrIGlucHV0XG4gICAqXG4gICAqIFRoaXMgbWlnaHQgYmUgYSBKU09OLWVuY29kZWQgb2JqZWN0LCBvciBqdXN0IGEgdGV4dC5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbVRleHQodGV4dDogc3RyaW5nKSB7XG4gICAgcmV0dXJuIG5ldyBUYXNrSW5wdXQoSW5wdXRUeXBlLlRFWFQsIHRleHQpO1xuICB9XG5cbiAgLyoqXG4gICAqIFVzZSBhbiBvYmplY3QgYXMgdGFzayBpbnB1dFxuICAgKlxuICAgKiBUaGlzIG9iamVjdCBtYXkgY29udGFpbiBEYXRhIGFuZCBDb250ZXh0IGZpZWxkc1xuICAgKiBhcyBvYmplY3QgdmFsdWVzLCBpZiBkZXNpcmVkLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tT2JqZWN0KG9iajogeyBba2V5OiBzdHJpbmddOiBhbnkgfSkge1xuICAgIHJldHVybiBuZXcgVGFza0lucHV0KElucHV0VHlwZS5PQkpFQ1QsIG9iaik7XG4gIH1cblxuICAvKipcbiAgICogVXNlIGEgcGFydCBvZiB0aGUgZXhlY3V0aW9uIGRhdGEgb3IgdGFzayBjb250ZXh0IGFzIHRhc2sgaW5wdXRcbiAgICpcbiAgICogVXNlIHRoaXMgd2hlbiB5b3Ugd2FudCB0byB1c2UgYSBzdWJvYmplY3Qgb3Igc3RyaW5nIGZyb21cbiAgICogdGhlIGN1cnJlbnQgc3RhdGUgbWFjaGluZSBleGVjdXRpb24gb3IgdGhlIGN1cnJlbnQgdGFzayBjb250ZXh0XG4gICAqIGFzIGNvbXBsZXRlIHBheWxvYWQgdG8gYSB0YXNrLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tSnNvblBhdGhBdChwYXRoOiBzdHJpbmcpIHtcbiAgICByZXR1cm4gbmV3IFRhc2tJbnB1dChJbnB1dFR5cGUuVEVYVCwgSnNvblBhdGguc3RyaW5nQXQocGF0aCkpO1xuICB9XG5cbiAgLyoqXG4gICAqIFVzZSBhIHBhcnQgb2YgdGhlIGV4ZWN1dGlvbiBkYXRhIGFzIHRhc2sgaW5wdXRcbiAgICpcbiAgICogVXNlIHRoaXMgd2hlbiB5b3Ugd2FudCB0byB1c2UgYSBzdWJvYmplY3Qgb3Igc3RyaW5nIGZyb21cbiAgICogdGhlIGN1cnJlbnQgc3RhdGUgbWFjaGluZSBleGVjdXRpb24gYXMgY29tcGxldGUgcGF5bG9hZFxuICAgKiB0byBhIHRhc2suXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21EYXRhQXQocGF0aDogc3RyaW5nKSB7XG4gICAgcmV0dXJuIG5ldyBUYXNrSW5wdXQoSW5wdXRUeXBlLlRFWFQsIEpzb25QYXRoLnN0cmluZ0F0KHBhdGgpKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBVc2UgYSBwYXJ0IG9mIHRoZSB0YXNrIGNvbnRleHQgYXMgdGFzayBpbnB1dFxuICAgKlxuICAgKiBVc2UgdGhpcyB3aGVuIHlvdSB3YW50IHRvIHVzZSBhIHN1Ym9iamVjdCBvciBzdHJpbmcgZnJvbVxuICAgKiB0aGUgY3VycmVudCB0YXNrIGNvbnRleHQgYXMgY29tcGxldGUgcGF5bG9hZFxuICAgKiB0byBhIHRhc2suXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Db250ZXh0QXQocGF0aDogc3RyaW5nKSB7XG4gICAgcmV0dXJuIG5ldyBUYXNrSW5wdXQoSW5wdXRUeXBlLlRFWFQsIEpzb25QYXRoLnN0cmluZ0F0KHBhdGgpKTtcbiAgfVxuXG4gIC8qKlxuICAgKlxuICAgKiBAcGFyYW0gdHlwZSB0eXBlIG9mIHRhc2sgaW5wdXRcbiAgICogQHBhcmFtIHZhbHVlIHBheWxvYWQgZm9yIHRoZSBjb3JyZXNwb25kaW5nIGlucHV0IHR5cGUuXG4gICAqIEl0IGNhbiBiZSBhIEpTT04tZW5jb2RlZCBvYmplY3QsIGNvbnRleHQsIGRhdGEsIGV0Yy5cbiAgICovXG4gIHByaXZhdGUgY29uc3RydWN0b3IocHVibGljIHJlYWRvbmx5IHR5cGU6IElucHV0VHlwZSwgcHVibGljIHJlYWRvbmx5IHZhbHVlOiBhbnkpIHt9XG59XG5cbi8qKlxuICogVGhlIHR5cGUgb2YgdGFzayBpbnB1dFxuICovXG5leHBvcnQgZW51bSBJbnB1dFR5cGUge1xuICAvKipcbiAgICogVXNlIGEgbGl0ZXJhbCBzdHJpbmdcbiAgICogVGhpcyBtaWdodCBiZSBhIEpTT04tZW5jb2RlZCBvYmplY3QsIG9yIGp1c3QgdGV4dC5cbiAgICogdmFsaWQgSlNPTiB0ZXh0OiBzdGFuZGFsb25lLCBxdW90ZS1kZWxpbWl0ZWQgc3RyaW5nczsgb2JqZWN0czsgYXJyYXlzOyBudW1iZXJzOyBCb29sZWFuIHZhbHVlczsgYW5kIG51bGwuXG4gICAqXG4gICAqIGV4YW1wbGU6IGBsaXRlcmFsIHN0cmluZ2BcbiAgICogZXhhbXBsZToge1wianNvblwiOiBcImVuY29kZWRcIn1cbiAgICovXG4gIFRFWFQsXG4gIC8qKlxuICAgKiBVc2UgYW4gb2JqZWN0IHdoaWNoIG1heSBjb250YWluIERhdGEgYW5kIENvbnRleHQgZmllbGRzXG4gICAqIGFzIG9iamVjdCB2YWx1ZXMsIGlmIGRlc2lyZWQuXG4gICAqXG4gICAqIGV4YW1wbGU6XG4gICAqIHtcbiAgICogIGxpdGVyYWw6ICdsaXRlcmFsJyxcbiAgICogIFNvbWVJbnB1dDogc2ZuLkpzb25QYXRoLnN0cmluZ0F0KCckLnNvbWVGaWVsZCcpXG4gICAqIH1cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3RlcC1mdW5jdGlvbnMvbGF0ZXN0L2RnL2NvbmNlcHRzLXN0YXRlLW1hY2hpbmUtZGF0YS5odG1sXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N0ZXAtZnVuY3Rpb25zL2xhdGVzdC9kZy9pbnB1dC1vdXRwdXQtY29udGV4dG9iamVjdC5odG1sXG4gICAqL1xuICBPQkpFQ1QsXG59XG4iXX0=