"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Condition = void 0;
/**
 * A Condition for use in a Choice state branch
 */
class Condition {
    /**
     * Matches if variable is present
     */
    static isPresent(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsPresent, true);
    }
    /**
     * Matches if variable is not present
     */
    static isNotPresent(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsPresent, false);
    }
    /**
     * Matches if variable is a string
     */
    static isString(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsString, true);
    }
    /**
     * Matches if variable is not a string
     */
    static isNotString(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsString, false);
    }
    /**
     * Matches if variable is numeric
     */
    static isNumeric(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsNumeric, true);
    }
    /**
     * Matches if variable is not numeric
     */
    static isNotNumeric(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsNumeric, false);
    }
    /**
     * Matches if variable is boolean
     */
    static isBoolean(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsBoolean, true);
    }
    /**
     * Matches if variable is not boolean
     */
    static isNotBoolean(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsBoolean, false);
    }
    /**
     * Matches if variable is a timestamp
     */
    static isTimestamp(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsTimestamp, true);
    }
    /**
     * Matches if variable is not a timestamp
     */
    static isNotTimestamp(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsTimestamp, false);
    }
    /**
     * Matches if variable is not null
     */
    static isNotNull(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsNull, false);
    }
    /**
     * Matches if variable is Null
     */
    static isNull(variable) {
        return new VariableComparison(variable, ComparisonOperator.IsNull, true);
    }
    /**
     * Matches if a boolean field has the given value
     */
    static booleanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.BooleanEquals, value);
    }
    /**
     * Matches if a boolean field equals to a value at a given mapping path
     */
    static booleanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.BooleanEqualsPath, value);
    }
    /**
     * Matches if a string field equals to a value at a given mapping path
     */
    static stringEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringEqualsPath, value);
    }
    /**
     * Matches if a string field has the given value
     */
    static stringEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringEquals, value);
    }
    /**
     * Matches if a string field sorts before a given value
     */
    static stringLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThan, value);
    }
    /**
     * Matches if a string field sorts before a given value at a particular mapping
     */
    static stringLessThanJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThanPath, value);
    }
    /**
     * Matches if a string field sorts equal to or before a given value
     */
    static stringLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThanEquals, value);
    }
    /**
     * Matches if a string field sorts equal to or before a given mapping
     */
    static stringLessThanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThanEqualsPath, value);
    }
    /**
     * Matches if a string field sorts after a given value
     */
    static stringGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThan, value);
    }
    /**
     * Matches if a string field sorts after a value at a given mapping path
     */
    static stringGreaterThanJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThanPath, value);
    }
    /**
     * Matches if a string field sorts after or equal to value at a given mapping path
     */
    static stringGreaterThanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThanEqualsPath, value);
    }
    /**
     * Matches if a string field sorts after or equal to a given value
     */
    static stringGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThanEquals, value);
    }
    /**
     * Matches if a numeric field has the given value
     */
    static numberEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericEquals, value);
    }
    /**
     * Matches if a numeric field has the value in a given mapping path
     */
    static numberEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericEqualsPath, value);
    }
    /**
     * Matches if a numeric field is less than the given value
     */
    static numberLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThan, value);
    }
    /**
     * Matches if a numeric field is less than the value at the given mapping path
     */
    static numberLessThanJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThanPath, value);
    }
    /**
     * Matches if a numeric field is less than or equal to the given value
     */
    static numberLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThanEquals, value);
    }
    /**
     * Matches if a numeric field is less than or equal to the numeric value at given mapping path
     */
    static numberLessThanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThanEqualsPath, value);
    }
    /**
     * Matches if a numeric field is greater than the given value
     */
    static numberGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThan, value);
    }
    /**
     * Matches if a numeric field is greater than the value at a given mapping path
     */
    static numberGreaterThanJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThanPath, value);
    }
    /**
     * Matches if a numeric field is greater than or equal to the given value
     */
    static numberGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThanEquals, value);
    }
    /**
     * Matches if a numeric field is greater than or equal to the value at a given mapping path
     */
    static numberGreaterThanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThanEqualsPath, value);
    }
    /**
     * Matches if a timestamp field is the same time as the given timestamp
     */
    static timestampEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampEquals, value);
    }
    /**
     * Matches if a timestamp field is the same time as the timestamp at a given mapping path
     */
    static timestampEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampEqualsPath, value);
    }
    /**
     * Matches if a timestamp field is before the given timestamp
     */
    static timestampLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThan, value);
    }
    /**
     * Matches if a timestamp field is before the timestamp at a given mapping path
     */
    static timestampLessThanJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThanPath, value);
    }
    /**
     * Matches if a timestamp field is before or equal to the given timestamp
     */
    static timestampLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThanEquals, value);
    }
    /**
     * Matches if a timestamp field is before or equal to the timestamp at a given mapping path
     */
    static timestampLessThanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThanEqualsPath, value);
    }
    /**
     * Matches if a timestamp field is after the given timestamp
     */
    static timestampGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThan, value);
    }
    /**
     * Matches if a timestamp field is after the timestamp at a given mapping path
     */
    static timestampGreaterThanJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThanPath, value);
    }
    /**
     * Matches if a timestamp field is after or equal to the given timestamp
     */
    static timestampGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThanEquals, value);
    }
    /**
     * Matches if a timestamp field is after or equal to the timestamp at a given mapping path
     */
    static timestampGreaterThanEqualsJsonPath(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThanEqualsPath, value);
    }
    /**
     * Matches if a field matches a string pattern that can contain a wild card (*) e.g: log-*.txt or *LATEST*.
     * No other characters other than "*" have any special meaning - * can be escaped: \\*
     */
    static stringMatches(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringMatches, value);
    }
    /**
     * Combine two or more conditions with a logical AND
     */
    static and(...conditions) {
        return new CompoundCondition(CompoundOperator.And, ...conditions);
    }
    /**
     * Combine two or more conditions with a logical OR
     */
    static or(...conditions) {
        return new CompoundCondition(CompoundOperator.Or, ...conditions);
    }
    /**
     * Negate a condition
     */
    static not(condition) {
        return new NotCondition(condition);
    }
}
exports.Condition = Condition;
/**
 * Comparison Operator types
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    ComparisonOperator[ComparisonOperator["StringEquals"] = 0] = "StringEquals";
    ComparisonOperator[ComparisonOperator["StringEqualsPath"] = 1] = "StringEqualsPath";
    ComparisonOperator[ComparisonOperator["StringLessThan"] = 2] = "StringLessThan";
    ComparisonOperator[ComparisonOperator["StringLessThanPath"] = 3] = "StringLessThanPath";
    ComparisonOperator[ComparisonOperator["StringGreaterThan"] = 4] = "StringGreaterThan";
    ComparisonOperator[ComparisonOperator["StringGreaterThanPath"] = 5] = "StringGreaterThanPath";
    ComparisonOperator[ComparisonOperator["StringLessThanEquals"] = 6] = "StringLessThanEquals";
    ComparisonOperator[ComparisonOperator["StringLessThanEqualsPath"] = 7] = "StringLessThanEqualsPath";
    ComparisonOperator[ComparisonOperator["StringGreaterThanEquals"] = 8] = "StringGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["StringGreaterThanEqualsPath"] = 9] = "StringGreaterThanEqualsPath";
    ComparisonOperator[ComparisonOperator["NumericEquals"] = 10] = "NumericEquals";
    ComparisonOperator[ComparisonOperator["NumericEqualsPath"] = 11] = "NumericEqualsPath";
    ComparisonOperator[ComparisonOperator["NumericLessThan"] = 12] = "NumericLessThan";
    ComparisonOperator[ComparisonOperator["NumericLessThanPath"] = 13] = "NumericLessThanPath";
    ComparisonOperator[ComparisonOperator["NumericGreaterThan"] = 14] = "NumericGreaterThan";
    ComparisonOperator[ComparisonOperator["NumericGreaterThanPath"] = 15] = "NumericGreaterThanPath";
    ComparisonOperator[ComparisonOperator["NumericLessThanEquals"] = 16] = "NumericLessThanEquals";
    ComparisonOperator[ComparisonOperator["NumericLessThanEqualsPath"] = 17] = "NumericLessThanEqualsPath";
    ComparisonOperator[ComparisonOperator["NumericGreaterThanEquals"] = 18] = "NumericGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["NumericGreaterThanEqualsPath"] = 19] = "NumericGreaterThanEqualsPath";
    ComparisonOperator[ComparisonOperator["BooleanEquals"] = 20] = "BooleanEquals";
    ComparisonOperator[ComparisonOperator["BooleanEqualsPath"] = 21] = "BooleanEqualsPath";
    ComparisonOperator[ComparisonOperator["TimestampEquals"] = 22] = "TimestampEquals";
    ComparisonOperator[ComparisonOperator["TimestampEqualsPath"] = 23] = "TimestampEqualsPath";
    ComparisonOperator[ComparisonOperator["TimestampLessThan"] = 24] = "TimestampLessThan";
    ComparisonOperator[ComparisonOperator["TimestampLessThanPath"] = 25] = "TimestampLessThanPath";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThan"] = 26] = "TimestampGreaterThan";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThanPath"] = 27] = "TimestampGreaterThanPath";
    ComparisonOperator[ComparisonOperator["TimestampLessThanEquals"] = 28] = "TimestampLessThanEquals";
    ComparisonOperator[ComparisonOperator["TimestampLessThanEqualsPath"] = 29] = "TimestampLessThanEqualsPath";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThanEquals"] = 30] = "TimestampGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThanEqualsPath"] = 31] = "TimestampGreaterThanEqualsPath";
    ComparisonOperator[ComparisonOperator["IsNull"] = 32] = "IsNull";
    ComparisonOperator[ComparisonOperator["IsBoolean"] = 33] = "IsBoolean";
    ComparisonOperator[ComparisonOperator["IsNumeric"] = 34] = "IsNumeric";
    ComparisonOperator[ComparisonOperator["IsString"] = 35] = "IsString";
    ComparisonOperator[ComparisonOperator["IsTimestamp"] = 36] = "IsTimestamp";
    ComparisonOperator[ComparisonOperator["IsPresent"] = 37] = "IsPresent";
    ComparisonOperator[ComparisonOperator["StringMatches"] = 38] = "StringMatches";
})(ComparisonOperator || (ComparisonOperator = {}));
/**
 * Compound Operator types
 */
var CompoundOperator;
(function (CompoundOperator) {
    CompoundOperator[CompoundOperator["And"] = 0] = "And";
    CompoundOperator[CompoundOperator["Or"] = 1] = "Or";
})(CompoundOperator || (CompoundOperator = {}));
/**
 * Scalar comparison
 */
class VariableComparison extends Condition {
    constructor(variable, comparisonOperator, value) {
        super();
        this.variable = variable;
        this.comparisonOperator = comparisonOperator;
        this.value = value;
        if (!/^\$|(\$[.[])/.test(variable)) {
            throw new Error(`Variable reference must be '$', start with '$.', or start with '$[', got '${variable}'`);
        }
    }
    renderCondition() {
        return {
            Variable: this.variable,
            [ComparisonOperator[this.comparisonOperator]]: this.value,
        };
    }
}
/**
 * Logical compound condition
 */
class CompoundCondition extends Condition {
    constructor(operator, ...conditions) {
        super();
        this.operator = operator;
        this.conditions = conditions;
        if (conditions.length === 0) {
            throw new Error('Must supply at least one inner condition for a logical combination');
        }
    }
    renderCondition() {
        return {
            [CompoundOperator[this.operator]]: this.conditions.map(c => c.renderCondition()),
        };
    }
}
/**
 * Logical unary condition
 */
class NotCondition extends Condition {
    constructor(comparisonOperation) {
        super();
        this.comparisonOperation = comparisonOperation;
    }
    renderCondition() {
        return {
            Not: this.comparisonOperation.renderCondition(),
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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