"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsSubscription = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
/**
 * Use an SQS queue as a subscription target
 */
class SqsSubscription {
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_sns_subscriptions_SqsSubscriptionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SqsSubscription);
            }
            throw error;
        }
    }
    /**
     * Returns a configuration for an SQS queue to subscribe to an SNS topic
     */
    bind(topic) {
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        if (!core_2.Construct.isConstruct(this.queue)) {
            throw new Error('The supplied Queue object must be an instance of Construct');
        }
        const snsServicePrincipal = new iam.ServicePrincipal('sns.amazonaws.com');
        // add a statement to the queue resource policy which allows this topic
        // to send messages to the queue.
        this.queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.queue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [snsServicePrincipal],
            conditions: {
                ArnEquals: { 'aws:SourceArn': topic.topicArn },
            },
        }));
        // if the queue is encrypted, add a statement to the key resource policy
        // which allows this topic to decrypt KMS keys
        if (this.queue.encryptionMasterKey) {
            this.queue.encryptionMasterKey.addToResourcePolicy(new iam.PolicyStatement({
                resources: ['*'],
                actions: ['kms:Decrypt', 'kms:GenerateDataKey'],
                principals: [snsServicePrincipal],
            }));
        }
        // if the topic and queue are created in different stacks
        // then we need to make sure the topic is created first
        if (topic instanceof sns.Topic && topic.stack !== this.queue.stack) {
            this.queue.stack.addDependency(topic.stack);
        }
        return {
            subscriberScope: this.queue,
            subscriberId: core_1.Names.nodeUniqueId(topic.node),
            endpoint: this.queue.queueArn,
            protocol: sns.SubscriptionProtocol.SQS,
            rawMessageDelivery: this.props.rawMessageDelivery,
            filterPolicy: this.props.filterPolicy,
            region: this.regionFromArn(topic),
            deadLetterQueue: this.props.deadLetterQueue,
        };
    }
    regionFromArn(topic) {
        // no need to specify `region` for topics defined within the same stack
        if (topic instanceof sns.Topic) {
            if (topic.stack !== this.queue.stack) {
                // only if we know the region, will not work for
                // env agnostic stacks
                if (!core_1.Token.isUnresolved(topic.env.region) &&
                    (topic.env.region !== this.queue.env.region)) {
                    return topic.env.region;
                }
            }
            return undefined;
        }
        return core_1.Stack.of(topic).splitArn(topic.topicArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).region;
    }
}
exports.SqsSubscription = SqsSubscription;
_a = JSII_RTTI_SYMBOL_1;
SqsSubscription[_a] = { fqn: "@aws-cdk/aws-sns-subscriptions.SqsSubscription", version: "1.173.0" };
//# sourceMappingURL=data:application/json;base64,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