import { Connections, IConnectable, ISecurityGroup, IVpc, Port, SubnetSelection } from '@aws-cdk/aws-ec2';
import { Duration, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
/**
 * Construction properties for a LoadBalancer.
 *
 * @stability stable
 */
export interface LoadBalancerProps {
    /**
     * VPC network of the fleet instances.
     *
     * @stability stable
     */
    readonly vpc: IVpc;
    /**
     * Whether this is an internet-facing Load Balancer.
     *
     * This controls whether the LB has a public IP address assigned. It does
     * not open up the Load Balancer's security groups to public internet access.
     *
     * @default false
     * @stability stable
     */
    readonly internetFacing?: boolean;
    /**
     * What listeners to set up for the load balancer.
     *
     * Can also be added by .addListener()
     *
     * @default -
     * @stability stable
     */
    readonly listeners?: LoadBalancerListener[];
    /**
     * What targets to load balance to.
     *
     * Can also be added by .addTarget()
     *
     * @default - None.
     * @stability stable
     */
    readonly targets?: ILoadBalancerTarget[];
    /**
     * Health check settings for the load balancing targets.
     *
     * Not required but recommended.
     *
     * @default - None.
     * @stability stable
     */
    readonly healthCheck?: HealthCheck;
    /**
     * Whether cross zone load balancing is enabled.
     *
     * This controls whether the load balancer evenly distributes requests
     * across each availability zone
     *
     * @default true
     * @stability stable
     */
    readonly crossZone?: boolean;
    /**
     * Which subnets to deploy the load balancer.
     *
     * Can be used to define a specific set of subnets to deploy the load balancer to.
     * Useful multiple public or private subnets are covering the same availability zone.
     *
     * @default - Public subnets if internetFacing, Private subnets otherwise
     * @stability stable
     */
    readonly subnetSelection?: SubnetSelection;
}
/**
 * Describe the health check to a load balancer.
 *
 * @stability stable
 */
export interface HealthCheck {
    /**
     * What port number to health check on.
     *
     * @stability stable
     */
    readonly port: number;
    /**
     * What protocol to use for health checking.
     *
     * The protocol is automatically determined from the port if it's not supplied.
     *
     * @default Automatic
     * @stability stable
     */
    readonly protocol?: LoadBalancingProtocol;
    /**
     * What path to use for HTTP or HTTPS health check (must return 200).
     *
     * For SSL and TCP health checks, accepting connections is enough to be considered
     * healthy.
     *
     * @default "/"
     * @stability stable
     */
    readonly path?: string;
    /**
     * After how many successful checks is an instance considered healthy.
     *
     * @default 2
     * @stability stable
     */
    readonly healthyThreshold?: number;
    /**
     * After how many unsuccessful checks is an instance considered unhealthy.
     *
     * @default 5
     * @stability stable
     */
    readonly unhealthyThreshold?: number;
    /**
     * Number of seconds between health checks.
     *
     * @default Duration.seconds(30)
     * @stability stable
     */
    readonly interval?: Duration;
    /**
     * Health check timeout.
     *
     * @default Duration.seconds(5)
     * @stability stable
     */
    readonly timeout?: Duration;
}
/**
 * Interface that is going to be implemented by constructs that you can load balance to.
 *
 * @stability stable
 */
export interface ILoadBalancerTarget extends IConnectable {
    /**
     * Attach load-balanced target to a classic ELB.
     *
     * @param loadBalancer [disable-awslint:ref-via-interface] The load balancer to attach the target to.
     * @stability stable
     */
    attachToClassicLB(loadBalancer: LoadBalancer): void;
}
/**
 * Add a backend to the load balancer.
 *
 * @stability stable
 */
export interface LoadBalancerListener {
    /**
     * External listening port.
     *
     * @stability stable
     */
    readonly externalPort: number;
    /**
     * What public protocol to use for load balancing.
     *
     * Either 'tcp', 'ssl', 'http' or 'https'.
     *
     * May be omitted if the external port is either 80 or 443.
     *
     * @stability stable
     */
    readonly externalProtocol?: LoadBalancingProtocol;
    /**
     * Instance listening port.
     *
     * Same as the externalPort if not specified.
     *
     * @default externalPort
     * @stability stable
     */
    readonly internalPort?: number;
    /**
     * What public protocol to use for load balancing.
     *
     * Either 'tcp', 'ssl', 'http' or 'https'.
     *
     * May be omitted if the internal port is either 80 or 443.
     *
     * The instance protocol is 'tcp' if the front-end protocol
     * is 'tcp' or 'ssl', the instance protocol is 'http' if the
     * front-end protocol is 'https'.
     *
     * @stability stable
     */
    readonly internalProtocol?: LoadBalancingProtocol;
    /**
     * SSL policy names.
     *
     * @stability stable
     */
    readonly policyNames?: string[];
    /**
     * ID of SSL certificate.
     *
     * @stability stable
     */
    readonly sslCertificateId?: string;
    /**
     * Allow connections to the load balancer from the given set of connection peers.
     *
     * By default, connections will be allowed from anywhere. Set this to an empty list
     * to deny connections, or supply a custom list of peers to allow connections from
     * (IP ranges or security groups).
     *
     * @default Anywhere
     * @stability stable
     */
    readonly allowConnectionsFrom?: IConnectable[];
}
/**
 * @stability stable
 */
export declare enum LoadBalancingProtocol {
    /**
     * @stability stable
     */
    TCP = "tcp",
    /**
     * @stability stable
     */
    SSL = "ssl",
    /**
     * @stability stable
     */
    HTTP = "http",
    /**
     * @stability stable
     */
    HTTPS = "https"
}
/**
 * A load balancer with a single listener.
 *
 * Routes to a fleet of of instances in a VPC.
 *
 * @stability stable
 */
export declare class LoadBalancer extends Resource implements IConnectable {
    /**
     * Control all connections from and to this load balancer.
     *
     * @stability stable
     */
    readonly connections: Connections;
    /**
     * An object controlling specifically the connections for each listener added to this load balancer.
     *
     * @stability stable
     */
    readonly listenerPorts: ListenerPort[];
    private readonly elb;
    private readonly securityGroup;
    private readonly listeners;
    private readonly instancePorts;
    private readonly targets;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: LoadBalancerProps);
    /**
     * Add a backend to the load balancer.
     *
     * @returns A ListenerPort object that controls connections to the listener port
     * @stability stable
     */
    addListener(listener: LoadBalancerListener): ListenerPort;
    /**
     * @stability stable
     */
    addTarget(target: ILoadBalancerTarget): void;
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerName(): string;
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerCanonicalHostedZoneNameId(): string;
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerCanonicalHostedZoneName(): string;
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerDnsName(): string;
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerSourceSecurityGroupGroupName(): string;
    /**
     * @stability stable
     * @attribute true
     */
    get loadBalancerSourceSecurityGroupOwnerAlias(): string;
    /**
     * Allow connections to all existing targets on new instance port
     */
    private newInstancePort;
    /**
     * Allow connections to target on all existing instance ports
     */
    private newTarget;
    /**
     * Allow connections for a single (port, target) pair
     */
    private allowTargetConnection;
}
/**
 * Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *     const listener = elb.addListener(...);
 *
 *     listener.connections.allowDefaultPortFromAnyIPv4();
 *     // or
 *     instance.connections.allowToDefaultPort(listener);
 *
 * @stability stable
 */
export declare class ListenerPort implements IConnectable {
    /**
     * @stability stable
     */
    readonly connections: Connections;
    /**
     * @stability stable
     */
    constructor(securityGroup: ISecurityGroup, defaultPort: Port);
}
