#!/usr/bin/env python3
from __future__ import annotations

import logging
import os
import subprocess
from pathlib import Path

log = logging.getLogger(__name__)


class hardware_control_base:
    def __init__(self, config, name="hardware_control_base", verbose=False):
        self._wd = os.getcwd()
        self._name = name
        self.run_dir = None
        member_variables = [
            attr
            for attr in dir(self)
            if not callable(getattr(self, attr)) and not attr.startswith("_")
        ]
        for member in config.keys():
            if member in member_variables:
                setattr(self, member, config[member])

        if self.run_dir == "emulator":
            self.run_dir = os.getcwd()
        else:
            self.run_dir = Path(self.run_dir)

    def send_command(self, cmd, purpose="send command", extra_error_messages=None):
        extra_error_messages = extra_error_messages or []

        os.chdir(self._wd)
        os.chdir(self.run_dir)
        result = subprocess.run(
            cmd,
            shell=True,
            stdout=None if log.getEffectiveLevel() < logging.INFO else subprocess.PIPE,
            stderr=None if log.getEffectiveLevel() < logging.INFO else subprocess.PIPE,
        )
        os.chdir(self._wd)

        if result.returncode < 0:
            for extra_error_message in extra_error_messages:
                log.info(f"[{self._name}] {extra_error_message}")
            raise RuntimeError(f"[{self._name}] fail to {purpose}!!")
        log.info(f"[{self._name}] {purpose}")

        return result.returncode

    def send_command_and_read(
        self,
        cmd,
        type=float,
        purpose="send command and read",
        unit="",
        extra_error_messages=None,
        max_nTry=0,
    ):

        extra_error_messages = extra_error_messages or []
        os.chdir(self._wd)
        os.chdir(self.run_dir)
        result = subprocess.run(cmd, shell=True, stdout=subprocess.PIPE)
        os.chdir(self._wd)

        log.info(result.stdout.decode())

        # A while loop to retry communications in case of timeout error.
        # The maximum number of tries is determined by the function argument max_nTry.
        nTry = 0
        while result.returncode != 0 and nTry < max_nTry:
            nTry += 1
            for extra_error_message in extra_error_messages:
                log.info(f"[{self._name}] {extra_error_message}")
            log.info(f"Try again. Send command and read attempt {nTry} time(s).")
            os.chdir(self._wd)
            os.chdir(self.run_dir)
            result = subprocess.run(cmd, shell=True, stdout=subprocess.PIPE)
            os.chdir(self._wd)
            log.info(result.stdout.decode())

        if result.returncode != 0:
            raise RuntimeError(f"[{self._name}] fail to {purpose}!!")
        value = type(result.stdout.decode())
        for _extra_error_message in extra_error_messages:
            log.info(f"[{self._name}] {purpose}: {value}{unit}")

        return value, result.returncode
