from __future__ import absolute_import
from typing import Dict

from aoa.api.iterator_base_api import IteratorBaseApi


class DatasetApi(IteratorBaseApi):
    path = "/api/datasets/"
    type = "DATASET"

    def _get_header_params(self):
        header_vars = ['AOA-Project-ID', 'Content-Type', 'Accept']
        header_vals = [
            self.aoa_client.project_id,
            'application/json',
            self.aoa_client.select_header_accept([
                'application/json',
                'application/hal+json',
                'text/uri-list',
                'application/x-spring-data-compact+json'])]

        return self.generate_params(header_vars, header_vals)

    def find_by_name(self, dataset_name: str, projection: str = None):
        """
        returns a dataset of a project by name

        Parameters:
           dataset_name (str): dataset name(string) to find
           projection (str): projection type

        Returns:
            (dict): datasets
        """
        query_vars = ['name', 'projection']
        query_vals = [dataset_name, projection]
        query_params = self.generate_params(query_vars, query_vals)

        return self.aoa_client.get_request(
            path=self.path + "search/findByName",
            header_params=self._get_header_params(),
            query_params=query_params)

    def find_by_dataset_template_id(self, dataset_template_id: str, projection: str = None, page: int = None, size: int = None, sort: str = None):
        """
        returns a dataset of a project by name

        Parameters:
           dataset_template_id (str): dataset_template_id
           projection (str): projection type
           page (int): page number
           size (int): number of records in a page
           sort (str): column name and sorting order
           e.g. name?asc: sort name in ascending order, name?desc: sort name in descending order

        Returns:
            (dict): datasets
        """
        query_vars = ['datasetTemplateId', 'projection', 'page', 'size', 'sort']
        query_vals = [dataset_template_id, projection, page, size, sort]
        query_params = self.generate_params(query_vars, query_vals)

        return self.aoa_client.get_request(
            path=self.path + "search/findByDatasetTemplateId",
            header_params=self._get_header_params(),
            query_params=query_params)

    def save(self, dataset: Dict[str, str]):
        """
        register a dataset

        Parameters:
           dataset (dict): dataset to register

        Returns:
            (dict): dataset
        """
        self.required_params(['name', 'metadata'], dataset)

        return self.aoa_client.post_request(
            path=self.path,
            header_params=self._get_header_params(),
            query_params={},
            body=dataset)

