"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RichAction = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * Helper routines to work with Actions
 *
 * Can't put these on Action themselves since we only have an interface
 * and every library would need to reimplement everything (there is no
 * `ActionBase`).
 *
 * So here go the members that should have gone onto the Action class
 * but can't.
 *
 * It was probably my own idea but I don't want it anymore:
 * https://github.com/aws/aws-cdk/issues/10393
 */
class RichAction {
    constructor(action, pipeline) {
        this.action = action;
        this.pipeline = pipeline;
        this.actionProperties = action.actionProperties;
    }
    bind(scope, stage, options) {
        return this.action.bind(scope, stage, options);
    }
    onStateChange(name, target, options) {
        return this.action.onStateChange(name, target, options);
    }
    get isCrossRegion() {
        return !actionDimensionSameAsPipelineDimension(this.effectiveRegion, this.pipeline.env.region);
    }
    get isCrossAccount() {
        return !actionDimensionSameAsPipelineDimension(this.effectiveAccount, this.pipeline.env.account);
    }
    /**
     * Returns the Stack of the resource backing this action
     * if they belong to the same environment.
     * Returns `undefined` if either this action is not backed by a resource,
     * or if the resource does not belong to the same env as its Stack
     * (which can happen for imported resources).
     */
    get resourceStack() {
        const actionResource = this.actionProperties.resource;
        if (!actionResource) {
            return undefined;
        }
        const actionResourceStack = core_1.Stack.of(actionResource);
        const actionResourceStackEnv = {
            region: actionResourceStack.region,
            account: actionResourceStack.account,
        };
        return sameEnv(actionResource.env, actionResourceStackEnv) ? actionResourceStack : undefined;
    }
    /**
     * The region this action wants to execute in.
     * `undefined` means it wants to execute in the same region as the pipeline.
     */
    get effectiveRegion() {
        var _a, _b;
        return (_b = (_a = this.action.actionProperties.resource) === null || _a === void 0 ? void 0 : _a.env.region) !== null && _b !== void 0 ? _b : this.action.actionProperties.region;
    }
    /**
     * The account this action wants to execute in.
     * `undefined` means it wants to execute in the same account as the pipeline.
     */
    get effectiveAccount() {
        var _a, _b, _c, _d, _e;
        return (_e = (_b = (_a = this.action.actionProperties.role) === null || _a === void 0 ? void 0 : _a.env.account) !== null && _b !== void 0 ? _b : (_d = (_c = this.action.actionProperties) === null || _c === void 0 ? void 0 : _c.resource) === null || _d === void 0 ? void 0 : _d.env.account) !== null && _e !== void 0 ? _e : this.action.actionProperties.account;
    }
}
exports.RichAction = RichAction;
function actionDimensionSameAsPipelineDimension(actionDim, pipelineDim) {
    // if the action's dimension is `undefined`,
    // it means it is in the same region/account as the pipeline
    if (!actionDim) {
        return true;
    }
    // if the action's region/account is AWS::Region/AWS::AccountId,
    // we assume it's also in the same region/account as the pipeline
    if (core_1.Token.isUnresolved(actionDim)) {
        return true;
    }
    // here, we know the action's dimension is explicitly set;
    // in this case, it must be equal to the pipeline's dimension
    // for the action to be considered in the same region/account
    return core_1.Token.compareStrings(actionDim, pipelineDim) === core_1.TokenComparison.SAME;
}
/**
 * Whether the two envs represent the same environment
 */
function sameEnv(env1, env2) {
    return sameEnvDimension(env1.region, env2.region)
        && sameEnvDimension(env1.account, env2.account);
}
/**
 * Whether two string probably contain the same environment dimension (region or account)
 *
 * Used to compare either accounts or regions, and also returns true if both
 * are unresolved (in which case both are expted to be "current region" or "current account").
 */
function sameEnvDimension(dim1, dim2) {
    return [core_1.TokenComparison.SAME, core_1.TokenComparison.BOTH_UNRESOLVED].includes(core_1.Token.compareStrings(dim1, dim2));
}
//# sourceMappingURL=data:application/json;base64,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