"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Test Case 1 - Pattern deployment w/ default properties
// --------------------------------------------------------------
test('Pattern deployment w/ default properties', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ticker_symbol',
                                sqlType: 'VARCHAR(4)',
                                mapping: '$.ticker_symbol'
                            }, {
                                name: 'sector',
                                sqlType: 'VARCHAR(16)',
                                mapping: '$.sector'
                            }, {
                                name: 'change',
                                sqlType: 'REAL',
                                mapping: '$.change'
                            }, {
                                name: 'price',
                                sqlType: 'REAL',
                                mapping: '$.price'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    };
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-firehose-s3-and-analytics-stack', props);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test Case 2 - Test the getter methods
// --------------------------------------------------------------
test('Test properties', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    };
    const app = new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-kinesis-firehose-kinesis-analytics', props);
    // Assertions
    expect(app.kinesisAnalytics !== null);
    expect(app.kinesisFirehose !== null);
    expect(app.kinesisFirehoseRole !== null);
    expect(app.kinesisFirehoseLogGroup !== null);
    expect(app.s3Bucket !== null);
    expect(app.s3LoggingBucket !== null);
});
// --------------------------------------------------------------
// Test Case 3 - Override kinesisFirehose props
// --------------------------------------------------------------
test('test kinesisFirehose override ', () => {
    const stack = new core_1.Stack();
    new lib_1.KinesisFirehoseToAnalyticsAndS3(stack, 'test-firehose-s3', {
        kinesisFirehoseProps: {
            extendedS3DestinationConfiguration: {
                bufferingHints: {
                    intervalInSeconds: 600,
                    sizeInMBs: 55
                },
            }
        },
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        }
    });
    expect(stack).toHaveResourceLike("AWS::KinesisFirehose::DeliveryStream", {
        ExtendedS3DestinationConfiguration: {
            BufferingHints: {
                IntervalInSeconds: 600,
                SizeInMBs: 55
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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