"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const sqs = require("aws-cdk-lib/aws-sqs");
const kms = require("aws-cdk-lib/aws-kms");
const s3 = require("aws-cdk-lib/aws-s3");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
// --------------------------------------------------------------
// Test the getter methods
// --------------------------------------------------------------
test('Test getter methods', () => {
    // Initial Setup
    const stack = new aws_cdk_lib_1.Stack();
    const filter = { prefix: 'the/place', suffix: '*.mp3' };
    const props = {
        deployDeadLetterQueue: true,
        maxReceiveCount: 0,
        queueProps: {},
        s3EventTypes: [s3.EventType.OBJECT_REMOVED],
        s3EventFilters: [filter]
    };
    const app = new lib_1.S3ToSqs(stack, 'test-s3-sqs', props);
    // Assertion 1
    expect(app.sqsQueue !== null);
    // Assertion 2
    expect(app.deadLetterQueue !== null);
    // Assertion 3
    expect(app.s3Bucket !== null);
});
// --------------------------------------------------------------
// Test deployment w/ existing queue
// --------------------------------------------------------------
test('Test deployment w/ existing queue', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const kmsKey = new kms.Key(stack, 'ExistingQueueEncryptionKey', {});
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj',
        encryptionMasterKey: kmsKey
    });
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {
        existingQueueObj: queue
    });
    // Assertion 1
    expect(stack).toHaveResource("Custom::S3BucketNotifications", {
        NotificationConfiguration: {
            QueueConfigurations: [
                {
                    Events: ['s3:ObjectCreated:*'],
                    QueueArn: {
                        "Fn::GetAtt": [
                            "existingqueueobjF8AF0ED1",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
// --------------------------------------------------------------
// Test deployment w/ existing s3 Bucket
// --------------------------------------------------------------
test('Test deployment w/ existing Bucket', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const [myBucket] = defaults.buildS3Bucket(stack, {});
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {
        existingBucketObj: myBucket
    });
    // Assertion 1
    expect(stack).toHaveResource("Custom::S3BucketNotifications", {
        NotificationConfiguration: {
            QueueConfigurations: [
                {
                    Events: ['s3:ObjectCreated:*'],
                    QueueArn: {
                        "Fn::GetAtt": [
                            "tests3sqsqueue810CCE19",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
// --------------------------------------------------------------
// Pattern deployment w/ bucket block public access override
// --------------------------------------------------------------
test('Pattern deployment w/ bucket versioning turned off', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const props = {
        bucketProps: {
            blockPublicAccess: {
                blockPublicAcls: false,
                blockPublicPolicy: true,
                ignorePublicAcls: false,
                restrictPublicBuckets: true
            }
        }
    };
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', props);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: false,
            BlockPublicPolicy: true,
            IgnorePublicAcls: false,
            RestrictPublicBuckets: true
        }
    });
});
// --------------------------------------------------------------
// Test deployment w/ specific s3 event types
// --------------------------------------------------------------
test('Test deployment w/ s3 event types and filters', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    const filter = {
        prefix: 'the/place',
        suffix: '*.mp3'
    };
    const props = {
        deployDeadLetterQueue: true,
        maxReceiveCount: 0,
        queueProps: {},
        s3EventTypes: [s3.EventType.OBJECT_REMOVED],
        s3EventFilters: [filter]
    };
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', props);
    // Assertion 1
    expect(stack).toHaveResource("Custom::S3BucketNotifications", {
        NotificationConfiguration: {
            QueueConfigurations: [
                {
                    Events: ['s3:ObjectRemoved:*'],
                    Filter: {
                        Key: {
                            FilterRules: [
                                {
                                    Name: 'suffix',
                                    Value: '*.mp3'
                                },
                                {
                                    Name: 'prefix',
                                    Value: 'the/place'
                                }
                            ]
                        }
                    },
                    QueueArn: {
                        "Fn::GetAtt": [
                            "tests3sqsqueue810CCE19",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
// --------------------------------------------------------------
// Test deployment w/ SSE encryption enabled using customer managed KMS CMK
// --------------------------------------------------------------
test('Test deployment w/ SSE encryption enabled using customer managed KMS CMK', () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    // Helper declaration
    new lib_1.S3ToSqs(stack, 'test-s3-sqs', {
        enableEncryptionWithCustomerManagedKey: true
    });
    // Assertion 1
    expect(stack).toHaveResource("Custom::S3BucketNotifications");
    // Assertion 2
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "tests3sqsEncryptionKeyFD4D5946",
                "Arn"
            ]
        }
    });
    // Assertion 3
    expect(stack).toHaveResource('AWS::KMS::Key', {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Test bad call with existingBucket and bucketProps
// --------------------------------------------------------------
test("Test bad call with existingBucket and bucketProps", () => {
    // Stack
    const stack = new aws_cdk_lib_1.Stack();
    const testBucket = new s3.Bucket(stack, 'test-bucket', {});
    const app = () => {
        // Helper declaration
        new lib_1.S3ToSqs(stack, "bad-s3-args", {
            existingBucketObj: testBucket,
            bucketProps: {
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY
            },
        });
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide bucketProps or existingBucketObj, but not both.\n');
});
// --------------------------------------------------------------
// s3 bucket with bucket, loggingBucket, and auto delete objects
// --------------------------------------------------------------
test('s3 bucket with bucket, loggingBucket, and auto delete objects', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSqs(stack, 's3-sqs', {
        bucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        },
        loggingBucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true
        }
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        AccessControl: "LogDeliveryWrite"
    });
    expect(stack).toHaveResource("Custom::S3AutoDeleteObjects", {
        ServiceToken: {
            "Fn::GetAtt": [
                "CustomS3AutoDeleteObjectsCustomResourceProviderHandler9D90184F",
                "Arn"
            ]
        },
        BucketName: {
            Ref: "s3sqsS3LoggingBucketD877FC52"
        }
    });
});
// --------------------------------------------------------------
// s3 bucket with one content bucket and no logging bucket
// --------------------------------------------------------------
test('s3 bucket with one content bucket and no logging bucket', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSqs(stack, 's3-sqs', {
        bucketProps: {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        },
        logS3AccessLogs: false
    });
    expect(stack).toCountResources("AWS::S3::Bucket", 1);
});
//# sourceMappingURL=data:application/json;base64,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